package com.twistpair.wave.thinclient.protocol.headers;

import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpVersion;
import com.twistpair.wave.thinclient.util.IWtcMemoryStream;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcpHeader extends IWtcpHeader
{
    public static final int CURRENT_VERSION      = WtcpVersion.HeaderMain;
    public static final int SIZE                 = 4;
    public static final int SEQUENCE_NUMBER_BITS = 10;
    public static final int MAX_LENGTH           = 65535;                 // ushort.MaxValue;
    public static final int MAX_PAYLOAD_LENGTH   = MAX_LENGTH - SIZE;

    public static class ExtendedNumber
    {
        public int  small = 0;
        public int  wraps = 0;
        public long large = 0;

        public void reset()
        {
            large = small = wraps = 0;
        }
    }

    public int getOffset()
    {
        return 0;
    }

    public int getSize()
    {
        return SIZE;
    }

    public int getSequenceNumberBits()
    {
        return SEQUENCE_NUMBER_BITS;
    }

    public long extendSequenceNumber(ExtendedNumber number)
    {
        int lastSequenceNumber = normalizeSequenceNumber(number.large);
        if (number.small < lastSequenceNumber)
        {
            if (lastSequenceNumber - number.small > getSequenceNumberMaximum() / 4)
            {
                number.wraps++;
            }
        }
        number.large = (long) ((number.wraps << getSequenceNumberBits()) | number.small);
        return number.large;
    }

    /// <summary>
    /// Version         = 0b11100000 0b00000000 (mask 0xE000)
    /// MessageType     = 0b00011100 0b00000000 (mask 0x1C00)
    /// SequenceNumber  = 0b00000011 0b11111111 (mask 0x03FF)
    /// </summary>
    private int ushort0;    // TODO:(pv) Make this mutable WtcUInt16
    /// <summary>
    /// MessageInfo     = 0b11111111 0b11111111 (mask 0xFFFF)
    /// </summary>
    private int messageInfo; // TODO:(pv) Make this mutable WtcUInt16

    /// <summary>
    /// Version         = 0b11100000 0b00000000 (mask 0xE000)
    /// </summary>
    public int getVersion()
    {
        return (byte) (((ushort0 & 0xE000) >> 13) + 1);
    }

    public void setVersion(int value)
    {
        int mask = 0xE000;
        ushort0 = ((ushort0 & ~mask) | (((value - 1) << 13) & mask));
    }

    /// <summary>
    /// MessageType     = 0b00011100 0b00000000 (mask 0x1C00)
    /// </summary>
    public byte getMessageType()
    {
        return (byte) ((ushort0 & 0x1C00) >> 10);
    }

    public void setMessageType(int value)
    {
        int mask = 0x1C00;
        ushort0 = ((ushort0 & ~mask) | ((value << 10) & mask));
    }

    /// <summary>
    /// SequenceNumber  = 0b00000011 0b11111111 (mask 0x03FF)
    /// </summary>
    public int getSequenceNumber()
    {
        return ((ushort0 & 0x03FF) >> 0);
    }

    public void setSequenceNumber(int value)
    {
        int mask = 0x03FF;
        ushort0 = ((ushort0 & ~mask) | ((value << 0) & mask));
    }

    public int getPayloadLength()
    {
        switch (getMessageType())
        {
            case WtcpMessageType.KeyExchange:
            case WtcpMessageType.Control:
            case WtcpMessageType.Media:
                return messageInfo;
        }
        return 0;
    }

    public void setPayloadLength(int value)
    {
        messageInfo = value;
    }

    public int getMessageInfo()
    {
        return messageInfo;
    }

    public WtcpHeader() //
    {
        this(WtcpMessageType.Control);
    }

    public WtcpHeader(int messageType) //
    {
        this(messageType, 0, 0);
    }

    public WtcpHeader(int messageType, int sequenceNumber, int messageInfo) //
    {
        this(CURRENT_VERSION, messageType, sequenceNumber, messageInfo);
    }

    public WtcpHeader(int version, int messageType, int sequenceNumber, int messageInfo) //
    {
        super();
        setVersion(version);
        setMessageType(messageType);
        setSequenceNumber(sequenceNumber);
        this.messageInfo = messageInfo;
    }

    public WtcpHeader(IWtcMemoryStream buffer) //
    {
        super(buffer);
    }

    public void copyFrom(WtcpHeader other)
    {
        this.ushort0 = other.ushort0;
        this.messageInfo = other.messageInfo;
    }

    public void dumpHostToNetworkOrder(IWtcMemoryStream buffer)
    {
        super.dumpHostToNetworkOrder(buffer);
        buffer.writeUInt16(ushort0);
        buffer.writeUInt16(messageInfo);
    }

    public boolean loadNetworkToHostOrder(IWtcMemoryStream buffer)
    {
        if (!super.loadNetworkToHostOrder(buffer))
        {
            return false;
        }
        ushort0 = buffer.readUInt16();
        messageInfo = buffer.readUInt16();
        return true;
    }

    public void reset()
    {
        ushort0 = messageInfo = 0;
    }

    public String toString(char format)
    {
        format = Character.toLowerCase(format);
        StringBuffer sb = new StringBuffer();
        sb.append('{')
        //.append("ushort0=0b").append(WtcString.toBitString(ushort0, 16)).append(" (") //
        .append("v=").append(WtcString.formatNumber(getVersion(), 1)).append(',') //
        .append("t=").append(WtcpMessageType.toString(getMessageType(), format)); //
        switch (format)
        {
            case 'd':
                sb.append(",s=").append(WtcString.formatNumber(getSequenceNumber(), 4)) //
                //.append(')') //
                .append(",i=").append(WtcString.formatNumber(getMessageInfo(), 5));
                break;
            default:
                sb.append(",s=0x").append(WtcString.toHexString(getSequenceNumber(), 2)) //
                //.append(')') //
                .append(",i=0x").append(WtcString.toHexString(getMessageInfo(), 2));
                break;
        }
        sb.append('}');
        return sb.toString();
    }
}
