package com.twistpair.wave.thinclient.protocol;

import java.util.Vector;

import com.twistpair.wave.thinclient.util.WtcString;

/**
 * This file needs to be kept in sync w/ the C++ WtcProtocol.h
 */
public class WtcpConstants
{
    /**
     * NOTE that the version is logically "v1", but is actually [base] 0 in the protocol. 
     */
    public interface WtcpVersion
    {
        public static final byte HeaderMain    = 1;
        public static final byte HeaderControl = 1;
    }

    public static class WtcpMessageType
    {
        public static final byte Reserved0   = 0;

        public static final byte Hello       = 1;
        public static final byte UdpHello    = 2;
        public static final byte KeyExchange = 3;
        public static final byte Control     = 4;
        public static final byte Media       = 5;

        public static final byte Reserved6   = 6;
        public static final byte Reserved7   = 7;

        public static String getName(byte messageType)
        {
            switch (messageType)
            {
                case Reserved0:
                    return "Reserved0";
                case Hello:
                    return "Hello";
                case UdpHello:
                    return "UdpHello";
                case KeyExchange:
                    return "KeyExchange";
                case Control:
                    return "Control";
                case Media:
                    return "Media";
                case Reserved6:
                    return "Reserved6";
                case Reserved7:
                    return "Reserved7";
            }
            return "?";
        }

        public static String toString(byte messageType)
        {
            return toString(messageType, 'd');
        }

        public static String toString(byte messageType, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(messageType) + "\"(" + WtcString.formatNumber(messageType, 1) + ')';
                default:
                    return '\"' + getName(messageType) + "\"(0x" + WtcString.toHexString(messageType, 1) + ')';
            }
        }
    }

    public static class WtcpOpCode
    {
        //
        // Misc
        //
        public static final short WTCPControlSignaling      = 1;
        public static final short Ping                      = 2;
        public static final short ReKex                     = 3;

        //
        // Session
        //
        public static final short SessionOpen               = 5;
        public static final short SessionClose              = 6;
        public static final short SetCredentials            = 7;
        public static final short SessionResume             = 8;

        //
        // Channel
        //
        public static final short ChannelSetActive          = 10;
        public static final short ChannelList               = 11;
        public static final short ChannelActivity           = 12;
        public static final short ChannelPushToTalk         = 13;
        public static final short ChannelMute               = 14;
        public static final short ChannelPropertiesGet      = 15;
        public static final short ChannelPropertiesSet      = 16;

        //
        // Endpoint
        //
        public static final short EndpointPropertiesSet     = 20;
        public static final short EndpointLookup            = 21;
        public static final short EndpointPropertiesGet     = 22;
        public static final short EndpointPropertyFilterSet = 23;

        //
        // Call
        //
        public static final short PhoneLinesSetActive       = 30;
        public static final short PhoneLineStatus           = 31;
        public static final short CallMake                  = 32;
        public static final short CallOffer                 = 33;
        public static final short CallAnswer                = 34;
        public static final short CallHangup                = 35;
        public static final short CallDtmf                  = 36;
        public static final short CallProgress              = 37;
        public static final short CallPushToTalkOn          = 38;
        public static final short CallPushToTalkOff         = 39;

        //
        // Address Book
        //
        public static final short AddressBook               = 50;

        public static String getName(int opCode)
        {
            switch (opCode)
            {
                case WTCPControlSignaling:
                    return "WTCPControlSignaling";
                case Ping:
                    return "Ping";
                case ReKex:
                    return "ReKex";
                case SessionOpen:
                    return "SessionOpen";
                case SessionClose:
                    return "SessionClose";
                case SetCredentials:
                    return "SetCredentials";
                case SessionResume:
                    return "SessionResume";
                case ChannelSetActive:
                    return "ChannelSetActive";
                case ChannelList:
                    return "ChannelList";
                case ChannelActivity:
                    return "ChannelActivity";
                case ChannelPushToTalk:
                    return "ChannelPushToTalk";
                case ChannelMute:
                    return "ChannelMute";
                case ChannelPropertiesGet:
                    return "ChannelPropertiesGet";
                case ChannelPropertiesSet:
                    return "ChannelPropertiesSet";
                case EndpointPropertiesSet:
                    return "EndpointPropertiesSet";
                case EndpointLookup:
                    return "EndpointLookup";
                case EndpointPropertiesGet:
                    return "EndpointPropertiesGet";
                case EndpointPropertyFilterSet:
                    return "EndpointPropertyFilterSet";
                case PhoneLinesSetActive:
                    return "PhoneLinesSetActive";
                case PhoneLineStatus:
                    return "PhoneLineStatus";
                case CallMake:
                    return "CallMake";
                case CallOffer:
                    return "CallOffer";
                case CallAnswer:
                    return "CallAnswer";
                case CallHangup:
                    return "CallHangup";
                case CallDtmf:
                    return "CallDtmf";
                case CallProgress:
                    return "CallProgress";
                case CallPushToTalkOn:
                    return "CallPushToTalkOn";
                case CallPushToTalkOff:
                    return "CallPushToTalkOff";
                case AddressBook:
                    return "AddressBook";
            }
            return "?";
        }

        public static String toString(int opCode)
        {
            return toString(opCode, 'd');
        }

        public static String toString(int opCode, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(opCode) + "\"(" + WtcString.formatNumber(opCode, 4) + ')';
                default:
                    return '\"' + getName(opCode) + "\"(0x" + WtcString.toHexString(opCode, 2) + ')';
            }
        }
    }

    public static class WtcpOpType
    {
        //[Description("Logically \"0 | {WtcpOpCode}\"")]
        public static final byte Reserved0   = 0;
        //[Description("Logically \"1024 | {WtcpOpCode}\"")]
        public static final byte Request     = 1;
        //[Description("Logically \"2048 | {WtcpOpCode}\"")]
        public static final byte Response    = 2;
        //[Description("Logically \"3072 | {WtcpOpCode}\"")]
        public static final byte Unsolicited = 3;
        //[Description("Logically \"4096 | {WtcpOpCode}\"")]
        public static final byte Error       = 4;

        public static String getName(int opType)
        {
            switch (opType)
            {
                case Reserved0:
                    return "Reserved0";
                case Request:
                    return "Request";
                case Response:
                    return "Response";
                case Unsolicited:
                    return "Unsolicited";
                case Error:
                    return "Error";
            }
            return "?";
        }

        public static String toString(int opType)
        {
            return toString(opType, 'd');
        }

        public static String toString(int opType, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(opType) + "\"(" + WtcString.formatNumber(opType, 1) + ')';
                default:
                    return '\"' + getName(opType) + "\"(0x" + WtcString.toHexString(opType, 1) + ')';
            }
        }
    }

    public static class WtcpChannelChange
    {
        public static final byte Added   = 1;
        public static final byte Removed = 2;
        public static final byte Changed = 3;

        public static String getName(int changeType)
        {
            switch (changeType)
            {
                case Added:
                    return "Added";
                case Removed:
                    return "Removed";
                case Changed:
                    return "Changed";
            }
            return "?";
        }

        public static String toString(int changeType)
        {
            return toString(changeType, 'd');
        }

        public static String toString(int changeType, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(changeType) + "\"(" + WtcString.formatNumber(changeType, 1) + ')';
                default:
                    return '\"' + getName(changeType) + "\"(0x" + WtcString.toHexString(changeType, 1) + ')';
            }
        }
    }

    public interface WtcpAudioCodec
    {
        //[Browsable(false)]
        public static final byte Invalid                  = 0;
        //[Description("PCM Mono 16bit 8kHz")]
        public static final byte PCM_Mono16bBigEndian8kHz = 1;
        // http://en.wikipedia.org/wiki/Full_Rate
        //[Browsable(false), Description("GSM Full Rate")]
        public static final byte GSM_full_rate            = 10;
        //[Browsable(false)]
        public static final byte MSGSM                    = 11;
        //[Description("G.711 uLaw 64k")]
        public static final byte G711_Ulaw64k             = 20;
        //[Description("G.711 aLaw 64k")]
        public static final byte G711_Alaw64k             = 21;
        //[Browsable(false)]
        public static final byte AMR_MR0475               = 30;
        //[Browsable(false)]
        public static final byte AMR_MR0515               = 31;
        //[Browsable(false)]
        public static final byte AMR_MR0590               = 32;
        //[Browsable(false)]
        public static final byte AMR_MR0670               = 33;
        //[Browsable(false)]
        public static final byte AMR_MR0740               = 34;
        //[Browsable(false)]
        public static final byte AMR_MR0795               = 35;
        //[Browsable(false)]
        public static final byte AMR_MR1020               = 36;
        // http://en.wikipedia.org/wiki/Adaptive_Multi-Rate_audio_codec
        //[Browsable(false), Description("AMR MR 1220")]
        public static final byte AMR_MR1220               = 37;
        //[Browsable(false)]
        public static final byte AMR_MRDTX                = 38;
        public static final byte Speex_NB_2150            = 40;
        public static final byte Speex_NB_5950            = 41;
        public static final byte Speex_NB_8000            = 42;
        public static final byte Speex_NB_11000           = 43;
        //[Description("Speex narrowband @ 15 kbps")]
        public static final byte Speex_NB_15000           = 44;
        public static final byte Speex_NB_18200           = 45;
        public static final byte Speex_NB_24600           = 46;
    }

    public static class WtcpErrorCodes
    {
        public static final byte OK                              = 0;
        public static final byte Error                           = 1; // generic error. Always prefer specific errors though.
        public static final byte InvalidCodec                    = 2;
        public static final byte InvalidAudioScale               = 3;
        public static final byte InvalidSessionTimeout           = 4;
        public static final byte InvalidLicense                  = 5;
        public static final byte LicenseMaxReach                 = 6;
        public static final byte InvalidUsernameOrPassword       = 7;
        public static final byte InvalidProfileId                = 8;
        public static final byte InvalidChannelId                = 9;
        public static final byte ChannelAccessDenied             = 10;
        public static final byte NoSuchPage                      = 11;
        public static final byte InvalidEndpointId               = 12;
        public static final byte ChannelNotActive                = 13;
        public static final byte ResponseTooLong                 = 14;
        public static final byte DtmfNotEnabled                  = 15;
        public static final byte InvalidDtmf                     = 16;
        public static final byte InvalidE164Number               = 17;
        public static final byte PttBonk                         = 18;
        public static final byte TryAgain                        = 19; // this is really "internal failure"
        public static final byte InvalidOpCode                   = 20;
        public static final byte InvalidCallId                   = 21;
        public static final byte NoCallResource                  = 22;
        public static final byte TooMany                         = 23;
        public static final byte InACall                         = 24;
        public static final byte InvalidPhoneLine                = 25;
        public static final byte PhoneLineUnavailable            = 26;
        public static final byte CredentialsInProgress           = 27;
        public static final byte InvalidCallType                 = 28;
        public static final byte EndpointHasNoPhoneLine          = 29;
        public static final byte CannotSetOtherEndpointProperty  = 30;
        public static final byte InvalidSessionId                = 31;
        public static final byte WtcpTransmissionError           = 32;
        public static final byte KexTimeout                      = 33;
        public static final byte SessionOpenTimeout              = 34;
        public static final byte CredentialsTimeout              = 35;
        public static final byte ReadOnly                        = 36;
        public static final byte InvalidEndpointPropFilterType   = 37;
        public static final byte InvalidEndpointPropFilterString = 38;
        public static final byte Empty                           = 39;
        public static final byte InitSequenceFailed              = 40;
        public static final byte ServerError1                    = 41; // This is a PANIC.
        public static final byte ServerError2                    = 42; // This is a PANIC b/c of failed PM update.
        public static final byte MgsError                        = 43; // error communicating with MgS.
        public static final byte HttpError                       = 44;
        public static final byte InsufficientServerResources1    = 45; // no more mic device.
        public static final byte InsufficientServerResources2    = 46; // no more spkr device.
        public static final byte InvalidFormat                   = 47;
        public static final byte ProxyAtCapacity                 = 48;
        public static final byte BackendDrop                     = 49;
        public static final byte SessionNotOpened                = 50;
        public static final byte FeatureNotImplemented           = 51;
        public static final byte UserHasNoProfile                = 52;
        public static final byte UserHasNoZone                   = 53;
        public static final byte MgsXmlError                     = 54; // WRONG!
        public static final byte SelectProfileNow                = 55;
        public static final byte ProxyNotReady                   = 56;
        public static final byte ClientNotReady                  = 57;
        public static final byte ReKeyNeeded                     = 58;
        public static final byte CallPttAccessDenied             = 59;
        public static final byte CallPttNotActive                = 60;
        public static final byte AddressBookNotAvailable         = 61;
        public static final byte CallPttTooLong                  = 62;
        public static final byte CallPttBusy                     = 63;
        public static final byte UnknownErrorCode                = 64;

        // Internal error codes
        //
        // Codes past WtcpErrorCode_UnknownErrorCode are used internally to the proxy.
        // 
        public static final byte NullPtr                         = 65;
        public static final byte XmlParsingError                 = 66;

        public static String getName(int errorCode)
        {
            switch (errorCode)
            {
                case OK:
                    return "OK"; // 0
                case Error:
                    return "Error"; // 1
                case InvalidCodec:
                    return "InvalidCodec"; // 2
                case InvalidAudioScale:
                    return "InvalidAudioScale"; // 3
                case InvalidSessionTimeout:
                    return "InvalidSessionTimeout"; // 4
                case InvalidLicense:
                    return "InvalidLicense"; // 5
                case LicenseMaxReach:
                    return "LicenseMaxReach"; // 6
                case InvalidUsernameOrPassword:
                    return "InvalidUsernameOrPassword"; // 7
                case InvalidProfileId:
                    return "InvalidProfileId"; // 8
                case InvalidChannelId:
                    return "InvalidChannelId"; // 9
                case ChannelAccessDenied:
                    return "ChannelAccessDenied"; // 10
                case NoSuchPage:
                    return "NoSuchPage"; // 11
                case InvalidEndpointId:
                    return "InvalidEndpointId"; // 12
                case ChannelNotActive:
                    return "ChannelNotActive"; // 13
                case ResponseTooLong:
                    return "ResponseTooLong"; // 14
                case DtmfNotEnabled:
                    return "DtmfNotEnabled"; // 15
                case InvalidDtmf:
                    return "InvalidDtmf"; // 16
                case InvalidE164Number:
                    return "InvalidE164Number"; // 17
                case PttBonk:
                    return "PttBonk"; // 18
                case TryAgain:
                    return "TryAgain"; // 19
                case InvalidOpCode:
                    return "InvalidOpCode"; // 20
                case InvalidCallId:
                    return "InvalidCallId"; // 21
                case NoCallResource:
                    return "NoCallResource"; // 22
                case TooMany:
                    return "TooMany"; // 23
                case InACall:
                    return "InACall"; // 24
                case InvalidPhoneLine:
                    return "InvalidPhoneLine"; // 25
                case PhoneLineUnavailable:
                    return "PhoneLineUnavailable"; // 26
                case CredentialsInProgress:
                    return "CredentialsInProgress"; // 27
                case InvalidCallType:
                    return "InvalidCallType"; // 28
                case EndpointHasNoPhoneLine:
                    return "EndpointHasNoPhoneLine"; // 29
                case CannotSetOtherEndpointProperty:
                    return "CannotSetOtherEndpointProperty"; // 30
                case InvalidSessionId:
                    return "InvalidSessionId"; // 31
                case WtcpTransmissionError:
                    return "WtcpTransmissionError"; // 32
                case KexTimeout:
                    return "KexTimeout"; // 33
                case SessionOpenTimeout:
                    return "SessionOpenTimeout"; // 34
                case CredentialsTimeout:
                    return "CredentialsTimeout"; // 35
                case ReadOnly:
                    return "ReadOnly"; // 36
                case InvalidEndpointPropFilterType:
                    return "InvalidEndpointPropFilterType"; // 37
                case InvalidEndpointPropFilterString:
                    return "InvalidEndpointPropFilterString"; // 38
                case Empty:
                    return "Empty"; // 39
                case InitSequenceFailed:
                    return "InitSequenceFailed"; // 40
                case ServerError1:
                    return "ServerError1"; // 41
                case ServerError2:
                    return "ServerError2"; // 42
                case MgsError:
                    return "MgsError"; // 43
                case HttpError:
                    return "HttpError"; // 44
                case InsufficientServerResources1:
                    return "InsufficientServerResources1"; // 45
                case InsufficientServerResources2:
                    return "InsufficientServerResources2"; // 46
                case InvalidFormat:
                    return "InvalidFormat"; // 47
                case ProxyAtCapacity:
                    return "ProxyAtCapacity"; // 48
                case BackendDrop:
                    return "BackendDrop"; // 49
                case SessionNotOpened:
                    return "SessionNotOpened"; // 50
                case FeatureNotImplemented:
                    return "FeatureNotImplemented"; // 51
                case UserHasNoProfile:
                    return "UserHasNoProfile"; // 52
                case UserHasNoZone:
                    return "UserHasNoZone"; // 53
                case MgsXmlError:
                    return "MgsXmlError"; // 54
                case SelectProfileNow:
                    return "SelectProfileNow"; // 55
                case ProxyNotReady:
                    return "ProxyNotReady"; // 56
                case ClientNotReady:
                    return "ClientNotReady"; // 57
                case ReKeyNeeded:
                    return "ReKeyNeeded"; // 58
                case CallPttAccessDenied:
                    return "CallPttAccessDenied"; // 59
                case CallPttNotActive:
                    return "CallPttNotActive"; // 60
                case AddressBookNotAvailable:
                    return "AddressBookNotAvailable"; // 61
                case CallPttTooLong:
                    return "CallPttTooLong"; // 62
                case CallPttBusy:
                    return "CallPttBusy"; // 63
                case UnknownErrorCode:
                    return "UnknownErrorCode"; // 64
                case NullPtr:
                    return "NullPtr"; // 65
                case XmlParsingError:
                    return "XmlParsingError"; // 66
            }
            return "?";
        }

        public static String toString(int errorCode)
        {
            return toString(errorCode, 'd');
        }

        public static String toString(int errorCode, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(errorCode) + "\"(" + WtcString.formatNumber(errorCode, 1) + ')';
                default:
                    return '\"' + getName(errorCode) + "\"(0x" + WtcString.toHexString(errorCode, 2) + ')';
            }
        }
    }

    public static class WtcpCallProgressState
    {
        public static final byte Disconnected = 0;
        public static final byte Proceeding   = 1;
        public static final byte Ringing      = 2;
        public static final byte Connected    = 3;

        public static String getName(int progress)
        {
            switch (progress)
            {
                case Disconnected:
                    return "Disconnected"; // 0
                case Proceeding:
                    return "Proceeding"; // 1
                case Ringing:
                    return "Ringing"; // 2
                case Connected:
                    return "Connected"; // 3
            }
            return "?";
        }

        public static String toString(int progress)
        {
            return toString(progress, 'd');
        }

        public static String toString(int progress, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(progress) + "\"(" + WtcString.formatNumber(progress, 1) + ')';
                default:
                    return '\"' + getName(progress) + "\"(0x" + WtcString.toHexString(progress, 2) + ')';
            }
        }
    }

    /*
     * TODO:(pv) define Disposition values
     * "r": Available (aka: "Ready"),
     * "p": Phone,
     * "a": Away,
     * "b": Busy,
     * "i": Idle,
     * "t": Intercom,
     * "u": Unknown
     * 
     * TODO:(pv) define LOCation values/schema...
     */

    public static class WtcpCallType
    {
        public static final byte Endpoint = 1;
        public static final byte E164     = 2;

        // TODO:(pv/grava) Implement this ability on the Proxy/Engine 
        //public static final byte SipUri = 3;

        public static String getName(int callType)
        {
            switch (callType)
            {
                case Endpoint:
                    return "Endpoint"; // 1
                case E164:
                    return "E164"; // 2
            }
            return "?";
        }

        public static String toString(int callType)
        {
            return toString(callType, 'd');
        }

        public static String toString(int callType, char format)
        {
            switch (format)
            {
                case 'd':
                    return '\"' + getName(callType) + "\"(" + WtcString.formatNumber(callType, 1) + ')';
                default:
                    return '\"' + getName(callType) + "\"(0x" + WtcString.toHexString(callType, 2) + ')';
            }
        }
    }

    // TODO:(pv) Verify WtcpChannelFlags.All and Reserved15 work as a short and doesn't need to be an int.
    public interface WtcpChannelFlags
    {
        public static final short None           = 0x0000;
        public static final short Active         = 0x0001;
        public static final short RemoteMute     = 0x0002;
        public static final short ClientSpeaking = 0x0004;
        public static final short RemoteSpeaking = 0x0008;
        public static final short Reserved4      = 0x0010;
        public static final short Reserved5      = 0x0020;
        public static final short Reserved6      = 0x0040;
        public static final short Reserved7      = 0x0080;
        public static final short Reserved8      = 0x0100;
        public static final short Reserved9      = 0x0200;
        public static final short Reserved10     = 0x0400;
        public static final short Reserved11     = 0x0800;
        public static final short Reserved12     = 0x1000;
        public static final short Reserved13     = 0x2000;
        public static final short Reserved14     = 0x4000;
        public static final short Reserved15     = (short) 0x8000;
        public static final short All            = (short) 0xFFFF;
    }

    // TODO:(pv) Verify WtcpEndpointFlags.All works as an int and doesn't need to be a long.
    public static class WtcpEndpointFlags
    {
        public static final int None        = 0x00000000;
        public static final int Moderator   = 0x00000001; // endpoint has moderator privileges
        public static final int Speaking    = 0x00000002; // endpoint is speaking
        public static final int Typing      = 0x00000004; // endpoint is typing a text message
        public static final int AudioTxMute = 0x00000008; // endpoint cannot send audio
        public static final int TextMute    = 0x00000010; // endpoint cannot send text
        public static final int Unavailable = 0x00000020; // endpoint is available / unavailable
        public static final int Visible     = 0x00000040; // endpoint is visible
        public static final int AudioRxMute = 0x00000080; // endpoint cannot receive audio
        public static final int TxSilence   = 0x00000100; // endpoint is sending nothing or sending silent packets
        public static final int RxSilence   = 0x00000200; // endpoint is receiving nothing or receiving silent packets
        public static final int EmergencyTx = 0x00000400; // endpoint is in the midst of an emergency transmission
        public static final int Self        = 0x00000800; // endpoint is local client (you!)
        public static final int Point2Point = 0x00001000; // endpoint can do p2p (that is: it has registered a phone line)
        public static final int Gone        = 0x00002000; // endpoint has left the building
        public static final int Reserved14  = 0x00004000;
        public static final int Reserved15  = 0x00008000;
        public static final int User16      = 0x00010000; // user definable flag
        public static final int User17      = 0x00020000; // user definable flag
        public static final int User18      = 0x00040000; // user definable flag
        public static final int User19      = 0x00080000; // user definable flag
        public static final int User20      = 0x00100000; // user definable flag
        public static final int User21      = 0x00200000; // user definable flag
        public static final int User22      = 0x00400000; // user definable flag
        public static final int User23      = 0x00800000; // user definable flag
        public static final int User24      = 0x01000000; // user definable flag
        public static final int User25      = 0x02000000; // user definable flag
        public static final int User26      = 0x04000000; // user definable flag
        public static final int User27      = 0x08000000; // user definable flag
        public static final int User28      = 0x10000000; // user definable flag
        public static final int User29      = 0x20000000; // user definable flag
        public static final int User30      = 0x40000000; // user definable flag
        public static final int User31      = 0x80000000; // user definable flag
        public static final int All         = 0xFFFFFFFF;

        public static String toString(long flags)
        {
            Vector values = new Vector();
            if (flags == WtcpEndpointFlags.None)
            {
                return "None";
            }
            if (flags == WtcpEndpointFlags.All)
            {
                return "All";
            }
            if ((flags & WtcpEndpointFlags.Moderator) == WtcpEndpointFlags.Moderator)
            {
                values.addElement("Moderator");
            }
            if ((flags & WtcpEndpointFlags.Speaking) == WtcpEndpointFlags.Speaking)
            {
                values.addElement("Speaking");
            }
            if ((flags & WtcpEndpointFlags.Typing) == WtcpEndpointFlags.Typing)
            {
                values.addElement("Typing");
            }
            if ((flags & WtcpEndpointFlags.AudioTxMute) == WtcpEndpointFlags.AudioTxMute)
            {
                values.addElement("AudioTxMute");
            }
            if ((flags & WtcpEndpointFlags.TextMute) == WtcpEndpointFlags.TextMute)
            {
                values.addElement("TextMute");
            }
            if ((flags & WtcpEndpointFlags.Unavailable) == WtcpEndpointFlags.Unavailable)
            {
                values.addElement("Unavailable");
            }
            if ((flags & WtcpEndpointFlags.Visible) == WtcpEndpointFlags.Visible)
            {
                values.addElement("Visible");
            }
            if ((flags & WtcpEndpointFlags.AudioRxMute) == WtcpEndpointFlags.AudioRxMute)
            {
                values.addElement("AudioRxMute");
            }
            if ((flags & WtcpEndpointFlags.TxSilence) == WtcpEndpointFlags.TxSilence)
            {
                values.addElement("TxSilence");
            }
            if ((flags & WtcpEndpointFlags.RxSilence) == WtcpEndpointFlags.RxSilence)
            {
                values.addElement("RxSilence");
            }
            if ((flags & WtcpEndpointFlags.EmergencyTx) == WtcpEndpointFlags.EmergencyTx)
            {
                values.addElement("EmergencyTx");
            }
            if ((flags & WtcpEndpointFlags.Self) == WtcpEndpointFlags.Self)
            {
                values.addElement("Self");
            }
            if ((flags & WtcpEndpointFlags.Point2Point) == WtcpEndpointFlags.Point2Point)
            {
                values.addElement("Point2Point");
            }
            if ((flags & WtcpEndpointFlags.Gone) == WtcpEndpointFlags.Gone)
            {
                values.addElement("Gone");
            }
            if ((flags & WtcpEndpointFlags.Reserved14) == WtcpEndpointFlags.Reserved14)
            {
                values.addElement("Reserved14");
            }
            if ((flags & WtcpEndpointFlags.Reserved15) == WtcpEndpointFlags.Reserved15)
            {
                values.addElement("Reserved15");
            }
            if ((flags & WtcpEndpointFlags.User16) == WtcpEndpointFlags.User16)
            {
                values.addElement("User16");
            }
            if ((flags & WtcpEndpointFlags.User17) == WtcpEndpointFlags.User17)
            {
                values.addElement("User17");
            }
            if ((flags & WtcpEndpointFlags.User18) == WtcpEndpointFlags.User18)
            {
                values.addElement("User18");
            }
            if ((flags & WtcpEndpointFlags.User19) == WtcpEndpointFlags.User19)
            {
                values.addElement("User19");
            }
            if ((flags & WtcpEndpointFlags.User20) == WtcpEndpointFlags.User20)
            {
                values.addElement("User20");
            }
            if ((flags & WtcpEndpointFlags.User21) == WtcpEndpointFlags.User21)
            {
                values.addElement("User21");
            }
            if ((flags & WtcpEndpointFlags.User22) == WtcpEndpointFlags.User22)
            {
                values.addElement("User22");
            }
            if ((flags & WtcpEndpointFlags.User23) == WtcpEndpointFlags.User23)
            {
                values.addElement("User23");
            }
            if ((flags & WtcpEndpointFlags.User24) == WtcpEndpointFlags.User24)
            {
                values.addElement("User24");
            }
            if ((flags & WtcpEndpointFlags.User25) == WtcpEndpointFlags.User25)
            {
                values.addElement("User25");
            }
            if ((flags & WtcpEndpointFlags.User26) == WtcpEndpointFlags.User26)
            {
                values.addElement("User26");
            }
            if ((flags & WtcpEndpointFlags.User27) == WtcpEndpointFlags.User27)
            {
                values.addElement("User27");
            }
            if ((flags & WtcpEndpointFlags.User28) == WtcpEndpointFlags.User28)
            {
                values.addElement("User28");
            }
            if ((flags & WtcpEndpointFlags.User29) == WtcpEndpointFlags.User29)
            {
                values.addElement("User29");
            }
            if ((flags & WtcpEndpointFlags.User30) == WtcpEndpointFlags.User30)
            {
                values.addElement("User30");
            }
            if ((flags & WtcpEndpointFlags.User31) == WtcpEndpointFlags.User31)
            {
                values.addElement("User31");
            }

            String value;
            StringBuffer sb = new StringBuffer();
            sb.append('(');
            for (int i = 0; i < values.size(); i++)
            {
                value = (String) values.elementAt(i);
                if (i != 0)
                {
                    sb.append('|');
                }
                sb.append(value);
            }
            sb.append(')');
            return sb.toString();
        }
    }

    public interface WtcpEndpointFilterType
    {
        /**
         * Receive Unsolicited EndpointPropertiesGet of property changes based on the requested keys:
         * <ul>
         * <li>"*": all property keys</li>
         * <li>"a;B": semi-colon separated property keys "a" and "B"</li>
         * <li>"": none</li>
         * </ul>
         */
        public static final int SemiColonSeparated = 1;
    }
}
