package com.twistpair.wave.thinclient.media;

public abstract class WtcMediaDeviceMicrophone
{
    public interface IWtcMediaMicrophoneStateListener
    {
        // TODO:(pv) public void onMicrophoneOpening();

        public void onMicrophoneOpened(Runnable runAfterOpened);

        /**
         * Here would be a good place for the listener to play a "off beep".
         * If the close was the result of an error, the listener should play a "off bonk".
         * @param error the close was the result of an error, otherwise the close was the result of a success 
         */
        public void onMicrophoneClosed(boolean error, Runnable runAfterClosed);
    }

    public interface IWtcMediaMicrophoneBufferListener
    {
        /**
         * The listener should quickly copy this buffer and put it on a queue for appropriate processing in another thread. 
         * @param buffer
         * @param offset
         * @param length
         */
        public void onMicrophoneBuffer(byte[] buffer, int offset, int length);
    }

    /**
     * NOT thread safe!
     */
    private static long sBytesCapturedLifetime = 0;

    public static long getBytesCapturedLifetime()
    {
        return sBytesCapturedLifetime;
    }

    /**
     * NOT thread safe!
     */
    private static long sBytesEncodedLifetime = 0;

    public static long getBytesEncodedLifetime()
    {
        return sBytesEncodedLifetime;
    }

    private Runnable mRunAfterOpened;
    private Runnable mRunAfterClosed;

    public abstract boolean isOpen();

    public abstract void setLevel(int level);

    public abstract int getLevel();

    protected WtcMediaCodec                   mMediaEncoder;

    protected final Object                    mSyncState      = new Object();
    private IWtcMediaMicrophoneStateListener  mListenerState  = null;
    protected final Object                    mSyncBuffer     = new Object();
    private IWtcMediaMicrophoneBufferListener mListenerBuffer = null;

    protected WtcMediaDeviceMicrophone()
    {
        setMediaEncoder(null);
    }

    /**
     * NOT thread safe!
     * @param mediaEncoder
     */
    public void setMediaEncoder(WtcMediaCodec mediaEncoder)
    {
        if (mediaEncoder == null)
        {
            // no media encoder defined defaults to "PCM2PCM" raw short[] to byte[] copy:
            mediaEncoder = new WtcMediaCodecRawCopy();
        }
        this.mMediaEncoder = mediaEncoder;
    }

    /**
     * NOT thread safe!
     * @return the media encoder
     */
    public WtcMediaCodec getMediaEncoder()
    {
        return mMediaEncoder;
    }

    public void setStateListener(IWtcMediaMicrophoneStateListener listener)
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncState)
        {
            mListenerState = listener;
        }
    }

    public void setBufferListener(IWtcMediaMicrophoneBufferListener listener)
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncBuffer)
        {
            mListenerBuffer = listener;
        }
    }

    /**
     * @param runAfterOpened callback to run after the microphone is opened
     * @return true if the microphone was already opened [and runAfterOpened was called]; false otherwise 
     */
    public boolean open(Runnable runAfterOpened)
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncState)
        {
            this.mRunAfterOpened = runAfterOpened;

            close(false, null);

            if (isOpen())
            {
                onMicrophoneOpened();
                return true;
            }

            return false;
        }
    }

    protected void onMicrophoneOpened()
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncState)
        {
            if (mListenerState != null)
            {
                mListenerState.onMicrophoneOpened(mRunAfterOpened);
            }
            else if (mRunAfterOpened != null)
            {
                mRunAfterOpened.run();
            }
            mRunAfterOpened = null;
        }
    }

    /**
     * @param error
     * @param runAfterClosed
     * @return true if the microphone was already closed [and runAfterClosed was called]; false otherwise 
     */
    public boolean close(boolean error, Runnable runAfterClosed)
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncState)
        {
            this.mRunAfterClosed = runAfterClosed;

            if (!isOpen())
            {
                onMicrophoneClosed(error);
                return true;
            }

            return false;
        }
    }

    protected void onMicrophoneClosed(boolean error)
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncState)
        {
            if (mListenerState != null)
            {
                mListenerState.onMicrophoneClosed(error, mRunAfterClosed);
            }
            else if (mRunAfterClosed != null)
            {
                mRunAfterClosed.run();
            }
            mRunAfterClosed = null;
        }
    }

    public void onMicrophoneBuffer(byte[] buffer, int offset, int length)
    {
        // TODO:(pv) It would be nice to eliminate this lock entirely...
        synchronized (mSyncBuffer)
        {
            sBytesCapturedLifetime += length;

            if (mListenerBuffer != null)
            {
                mListenerBuffer.onMicrophoneBuffer(buffer, offset, length);
            }
        }
    }

    /**
     * NOT thread safe!
     * @param bufferUnencoded
     * @param offset
     * @param length
     * @param bufferEncoded
     * @return number of bytes encoded
     */
    public int encode(short[] bufferUnencoded, int offset, int length, byte[] bufferEncoded)
    {
        int bytesEncoded = mMediaEncoder.encode(bufferUnencoded, offset, length, bufferEncoded);
        sBytesEncodedLifetime += bytesEncoded;
        return bytesEncoded;
    }
}
