package com.twistpair.wave.thinclient.media;

import java.io.IOException;

import android.media.AudioRecord;

import com.twistpair.wave.thinclient.WtcDebug;
import com.twistpair.wave.thinclient.logging.WtcLog;

public abstract class AudioRecorder //
                implements Runnable
{
    private static final String  TAG              = WtcLog.TAG(AudioRecorder.class);

    @SuppressWarnings("unused")
    private static final boolean DBG              = (WtcDebug.DBG_LEVEL >= 1);
    private static final boolean VDBG             = (WtcDebug.DBG_LEVEL >= 2);

    private final int            mAudioSource;
    private final int            mAudioSampleRate;
    private final int            mAudioChannelConfig;
    private final int            mAudioEncodingFormat;
    private final int            mAudioFrameBytes;

    private final Object         mSyncAudioRecord = new Object();
    private AudioRecord          mAudioRecord;

    private boolean              mIsRecording;

    /**
     * Per http://developer.android.com/reference/android/media/AudioManager.html#startBluetoothSco()
     * "The following restrictions apply on input streams:
     * <ul>
     * <li>the format must be mono</li>
     * <li>the sampling must be 8kHz</li>
     * </ul>"
     * @param audioSource One of MediaRecorder.AudioSource.* 
     * @param audioSampleRate Must be 8kHz if recording from Bluetooth headset over SCO
     * @param audioChannelConfig
     * @param audioEncodingFormat
     * @param audioFrameBytes
     */
    public AudioRecorder(int audioSource, //
                    int audioSampleRate, int audioChannelConfig, int audioEncodingFormat, //
                    int audioFrameBytes)
    {
        mAudioSource = audioSource;
        mAudioSampleRate = audioSampleRate;
        mAudioChannelConfig = audioChannelConfig;
        mAudioEncodingFormat = audioEncodingFormat;
        mAudioFrameBytes = audioFrameBytes;
    }

    /**
     * Finds the smallest multiple of AudioRecord.getMinBufferSize(...) that successfully initializes an AudioRecord.
     * 
     * AudioRecord.getMinBufferSize by itself does not actually report the minimum buffer size required to successfully create an AudioRecord.
     * On several devices, creating an AudioRecord with the value returned from getMinBufferSize results in the following error:
     *  Most: "Invalid buffer size: minFrameCount 557, frameCount 256"
     *  Captivate (minBufferSize=640, created with minBufferSize*4): "Invalid buffer size: minFrameCount 1486, frameCount 1280"
     * @param audioSource One of MediaRecorder.AudioSource.*
     * @param audioSampleRate
     * @param audioChannelConfig
     * @param audioEncodingFormat
     * @param maxMultiplier
     * @return the smallest multiple of AudioRecord.getMinBufferSize(...) that successfully initializes an AudioRecord
     * @throws IllegalArgumentException
     */
    public static int findMinBufferSizeInBytes(int audioSource, //
                    int audioSampleRate, int audioChannelConfig, int audioEncodingFormat, //
                    int maxMultiplier) //
                    throws IllegalArgumentException
    {
        WtcLog.debug(TAG, "audioSource=" + audioSource);
        WtcLog.debug(TAG, "audioSampleRate=" + audioSampleRate);
        WtcLog.debug(TAG, "audioChannelConfig=" + audioChannelConfig);
        WtcLog.debug(TAG, "audioEncodingFormat=" + audioEncodingFormat);
        int minBufferSize = AudioRecord.getMinBufferSize(audioSampleRate, audioChannelConfig, audioEncodingFormat);
        if (minBufferSize == AudioRecord.ERROR_BAD_VALUE || minBufferSize == AudioRecord.ERROR)
        {
            throw new IllegalArgumentException();
        }
        WtcLog.info(TAG, "minBufferSize=" + minBufferSize);

        for (int i = 1; i < maxMultiplier; i++)
        {
            int bufferSize = minBufferSize * i;
            WtcLog.info(TAG, "Trying bufferSize=" + bufferSize + " (" + minBufferSize + " * " + i + ")");

            AudioRecord audioRecord = null;
            try
            {
                audioRecord = new AudioRecord(audioSource, //
                                audioSampleRate, audioChannelConfig, audioEncodingFormat, //
                                bufferSize);
                if (audioRecord.getState() == AudioRecord.STATE_INITIALIZED)
                {
                    WtcLog.info(TAG, "Found bufferSize=" + bufferSize + " (" + minBufferSize + " * " + i + ")");
                    return bufferSize;
                }
            }
            catch (IllegalArgumentException e)
            {
                // ignore
                WtcLog.warn(TAG, "findMinBufferSize - IllegalArgumentException", e);
            }
            finally
            {
                if (audioRecord != null)
                {
                    audioRecord.release();
                    audioRecord = null;
                }
            }
        }

        throw new IllegalArgumentException("Failed to initialize AudioRecord with bufferSize <= " + maxMultiplier
                        + " * minBufferSize=" + minBufferSize);
    }

    public void stop()
    {
        WtcLog.info(TAG, "+stop()");
        synchronized (mSyncAudioRecord)
        {
            mIsRecording = false;

            //
            // Don't actually stop/close the speaker yet; let it complete whatever buffer it is currently recording.
            //
        }
        WtcLog.info(TAG, "-stop()");
    }

    protected abstract void onAudioRecorderStarted(int bufferLength);

    /**
     * @param buffer
     * @param length 
     * @param offset 
     * @return the number of shorts used in buffer, or 0 or AudioRecorder.ERROR* to exit
     * @throws InterruptedException
     */
    protected abstract int onAudioRecorderGotBuffer(short[] buffer, int offset, int length) throws InterruptedException;

    protected void onAudioRecorderStopped(Exception error)
    {
        synchronized (mSyncAudioRecord)
        {
            mIsRecording = false;

            if (mAudioRecord != null)
            {
                WtcLog.info(TAG, "stop: mAudioRecord.getState() == " + mAudioRecord.getState());
                if (mAudioRecord.getState() == AudioRecord.STATE_INITIALIZED)
                {
                    WtcLog.info(TAG, "stop: +mAudioRecord.stop()");
                    mAudioRecord.stop();
                    WtcLog.info(TAG, "stop: -mAudioRecord.stop()");
                }
                WtcLog.info(TAG, "stop: +mAudioRecord.release()");
                mAudioRecord.release();
                WtcLog.info(TAG, "stop: -mAudioRecord.release()");
                mAudioRecord = null;
            }
        }
    }

    public void run()
    {
        Exception error = null;

        try
        {
            WtcLog.info(TAG, "+run()");

            mIsRecording = true;

            int minBufferSizeInBytes = findMinBufferSizeInBytes(mAudioSource, //
                            mAudioSampleRate, mAudioChannelConfig, mAudioEncodingFormat, //
                            10);

            // The device has a minimum buffer size, so we must open the device with a buffer at least that large.
            WtcLog.info(TAG, "minBufferSizeInBytes(ORIGINAL)=" + minBufferSizeInBytes);

            // mMicrophone.encode can only operate on multiples of mAudioFrameBytes, so we must open the device with a buffer as some multiple of that.
            WtcLog.info(TAG, "mAudioFrameBytes=" + mAudioFrameBytes);

            // Calculate the first multiple of mAudioFrameBytes larger than minBufferSizeInBytes:
            minBufferSizeInBytes = (int) Math.ceil(minBufferSizeInBytes / (double) mAudioFrameBytes) * mAudioFrameBytes;
            WtcLog.info(TAG, "minBufferSizeInBytes(FRAMED)=" + minBufferSizeInBytes);

            WtcLog.info(TAG, "+audioRecord = new AudioRecord(..., " + minBufferSizeInBytes + ")");
            mAudioRecord = new AudioRecord(mAudioSource, //
                            mAudioSampleRate, mAudioChannelConfig, mAudioEncodingFormat, minBufferSizeInBytes);
            WtcLog.info(TAG, "-audioRecord = new AudioRecord(..., " + minBufferSizeInBytes + ")");

            //ByteBuffer bufferDirect = ByteBuffer.allocateDirect(minBufferSizeInBytes);
            short[] buffer = new short[minBufferSizeInBytes];
            int length;

            onAudioRecorderStarted(buffer.length);

            android.os.Process.setThreadPriority(android.os.Process.THREAD_PRIORITY_AUDIO);
            mAudioRecord.startRecording();

            while (mIsRecording)
            {
                if (VDBG)
                {
                    WtcLog.info(TAG, "run: +mAudioRecord.read(...)");
                }
                // TODO:(pv) Consider using possibly more performant ByteBuffer overload
                length = mAudioRecord.read(buffer, 0, buffer.length);
                if (VDBG)
                {
                    WtcLog.info(TAG, "run: -mAudioRecord.read(...); length=" + length);
                }

                // Even if !mIsRecording, don't exit the while loop just yet;
                // Process whatever buffer the mic was recording when it was being stopped...

                if (length > 0)
                {
                    if (VDBG)
                    {
                        WtcLog.info(TAG, "run: +length = onAudioRecorderGotBuffer(buffer, 0, length)");
                    }
                    length = onAudioRecorderGotBuffer(buffer, 0, length);
                    if (VDBG)
                    {
                        WtcLog.info(TAG, "run: -length = onAudioRecorderGotBuffer(buffer, 0, length)");
                    }
                }

                if (length <= 0)
                {
                    final String detailMessage;
                    switch (length)
                    {
                        case 0:
                            WtcLog.warn(TAG, "length == 0; // ending recording");
                            detailMessage = null;
                            break;
                        case AudioRecord.ERROR:
                            detailMessage = "AudioRecord.ERROR(" + length + ")";
                            break;
                        case AudioRecord.ERROR_BAD_VALUE:
                            detailMessage = "AudioRecord.ERROR_BAD_VALUE(" + length + ")";
                            break;
                        case AudioRecord.ERROR_INVALID_OPERATION:
                            detailMessage = "AudioRecord.ERROR_INVALID_OPERATION(" + length + ")";
                            break;
                        default:
                            detailMessage = "AudioRecord.ERROR UNKNOWN(" + length + ")";
                            break;
                    }
                    if (detailMessage != null)
                    {
                        WtcLog.error(TAG, detailMessage);
                        throw new IOException(detailMessage);
                    }
                    break; // exit while loop
                }
            }
        }
        catch (Exception e)
        {
            WtcLog.error(TAG, "run: EXCEPTION", e);
            error = e;
        }
        finally
        {
            onAudioRecorderStopped(error);
            WtcLog.info(TAG, "-run()");
        }
    }
}
