package com.twistpair.wave.thinclient.media;

import java.io.IOException;

import android.media.AudioTrack;

import com.twistpair.wave.thinclient.WtcDebug;
import com.twistpair.wave.thinclient.logging.WtcLog;

public abstract class AudioPlayer //
                implements Runnable
{
    private static final String  TAG  = WtcLog.TAG(AudioPlayer.class);

    @SuppressWarnings("unused")
    private static final boolean DBG  = (WtcDebug.DBG_LEVEL >= 1);
    private static final boolean VDBG = (WtcDebug.DBG_LEVEL >= 2);

    private final int            mAudioStreamType;
    private final int            mAudioSampleRate;
    private final int            mAudioChannelConfig;
    private final int            mAudioEncodingFormat;

    /**
     * Per http://developer.android.com/reference/android/media/AudioManager.html#startBluetoothSco()
     * "The following restrictions apply on output streams:
     * <ul>
     * <li>the stream type must be STREAM_VOICE_CALL</li>
     * <li>the format must be mono</li>
     * <li>the sampling must be 8kHz</li>
     * </ul>" 
     * @param audioStreamType One of AudioManager.STREAM_*; Must be AudioManager.STREAM_VOICE_CALL if playing to Bluetooth headset over SCO  
     * @param audioSampleRate Must be 8kHz or 16kHz if playing to Bluetooth headset over SCO
     * @param audioChannelConfig Must be AudioFormat.CHANNEL_OUT_MONO if playing to Bluetooth headset over SCO
     * @param audioEncodingFormat
     */
    public AudioPlayer(int audioStreamType, //
                    int audioSampleRate, int audioChannelConfig, int audioEncodingFormat)
    {
        mAudioStreamType = audioStreamType;
        mAudioSampleRate = audioSampleRate;
        mAudioChannelConfig = audioChannelConfig;
        mAudioEncodingFormat = audioEncodingFormat;
    }

    /**
     * Finds the smallest multiple of AudioTrack.getMinBufferSize(...) that successfully initializes an AudioTrack.
     * 
     * AudioTrack.getMinBufferSize by itself does not actually report the minimum buffer size required to successfully create an AudioTrack.
     * On several devices, creating an AudioTrack with the value returned from getMinBufferSize results in the following error:
     *  Most: "Invalid buffer size: minFrameCount 557, frameCount 256"
     *  Captivate (minBufferSize=640, created with minBufferSize*4): "Invalid buffer size: minFrameCount 1486, frameCount 1280"
     * @param audioStreamType
     * @param audioSampleRate
     * @param audioChannelConfig
     * @param audioEncodingFormat
     * @param maxMultiplier
     * @return the smallest multiple of AudioTrack.getMinBufferSize(...) that successfully initializes an AudioTrack
     * @throws IllegalArgumentException
     */
    public static int findMinBufferSizeInBytes(int audioStreamType, //
                    int audioSampleRate, int audioChannelConfig, int audioEncodingFormat, //
                    int maxMultiplier) //
                    throws IllegalArgumentException
    {
        WtcLog.debug(TAG, "audioStreamType=" + audioStreamType);
        WtcLog.debug(TAG, "audioSampleRate=" + audioSampleRate);
        WtcLog.debug(TAG, "audioChannelConfig=" + audioChannelConfig);
        WtcLog.debug(TAG, "audioEncodingFormat=" + audioEncodingFormat);
        int minBufferSize = AudioTrack.getMinBufferSize(audioSampleRate, audioChannelConfig, audioEncodingFormat);
        if (minBufferSize == AudioTrack.ERROR_BAD_VALUE || minBufferSize == AudioTrack.ERROR)
        {
            throw new IllegalArgumentException("getMinBufferSize(...)");
        }
        WtcLog.info(TAG, "minBufferSize=" + minBufferSize);

        for (int i = 1; i < maxMultiplier; i++)
        {
            int bufferSize = minBufferSize * i;
            WtcLog.info(TAG, "Trying bufferSize=" + bufferSize + " (" + minBufferSize + " * " + i + ")");

            AudioTrack audioTrack = null;
            try
            {
                audioTrack = new AudioTrack(audioStreamType, //
                                audioSampleRate, audioChannelConfig, audioEncodingFormat, //
                                bufferSize, AudioTrack.MODE_STREAM);
                if (audioTrack.getState() == AudioTrack.STATE_INITIALIZED)
                {
                    WtcLog.info(TAG, "Found bufferSize=" + bufferSize + " (" + minBufferSize + " * " + i + ")");
                    return bufferSize;
                }
            }
            catch (IllegalArgumentException e)
            {
                // ignore
                WtcLog.warn(TAG, "findMinBufferSize - IllegalArgumentException", e);
            }
            finally
            {
                if (audioTrack != null)
                {
                    audioTrack.release();
                    audioTrack = null;
                }
            }
        }

        throw new IllegalArgumentException("Failed to initialize AudioTrack with bufferSize <= " + maxMultiplier
                        + " * minBufferSize=" + minBufferSize);
    }

    protected abstract void onAudioPlayerStarted(int bufferLength);

    /**
     * @param buffer
     * @return the number of shorts used in buffer, or 0 or AudioTrack.ERROR* to exit
     * @throws InterruptedException
     */
    protected abstract int onAudioPlayerGetBuffer(short[] buffer) throws InterruptedException;

    protected abstract void onAudioPlayerStopped(Exception error);

    public void run()
    {
        AudioTrack audioTrack = null;
        Exception error = null;

        try
        {
            WtcLog.info(TAG, "+run()");

            int minBufferSizeInBytes = findMinBufferSizeInBytes(mAudioStreamType, //
                            mAudioSampleRate, mAudioChannelConfig, mAudioEncodingFormat, //
                            10);

            WtcLog.info(TAG, "+audioTrack = new AudioTrack(..., " + minBufferSizeInBytes + ", ...)");
            audioTrack = new AudioTrack(mAudioStreamType, //
                            mAudioSampleRate, mAudioChannelConfig, mAudioEncodingFormat, //
                            minBufferSizeInBytes, //
                            AudioTrack.MODE_STREAM);
            WtcLog.info(TAG, "-audioTrack = new AudioTrack(..., " + minBufferSizeInBytes + ", ...)");

            short[] buffer = new short[minBufferSizeInBytes];
            int length;

            onAudioPlayerStarted(buffer.length);

            android.os.Process.setThreadPriority(android.os.Process.THREAD_PRIORITY_AUDIO);
            audioTrack.play();

            while (true)
            {
                if (VDBG)
                {
                    WtcLog.info(TAG, "run: +length = onAudioPlayerGetBuffer(buffer)");
                }
                length = onAudioPlayerGetBuffer(buffer);
                if (VDBG)
                {
                    WtcLog.info(TAG, "run: -length = onAudioPlayerGetBuffer(buffer); length=" + length);
                }

                if (length > 0)
                {
                    if (VDBG)
                    {
                        WtcLog.info(TAG, "run: +length = audioTrack.write(...)");
                    }
                    length = audioTrack.write(buffer, 0, length);
                    if (VDBG)
                    {
                        WtcLog.info(TAG, "run: -length = audioTrack.write(...); length=" + length);
                    }
                }

                if (length <= 0)
                {
                    final String detailMessage;
                    switch (length)
                    {
                        case 0:
                            WtcLog.warn(TAG, "run: length == 0; // ending playback");
                            detailMessage = null;
                            break;
                        case AudioTrack.ERROR:
                            detailMessage = "AudioTrack.ERROR(" + length + ")";
                            break;
                        case AudioTrack.ERROR_BAD_VALUE:
                            detailMessage = "AudioTrack.ERROR_BAD_VALUE(" + length + ")";
                            break;
                        case AudioTrack.ERROR_INVALID_OPERATION:
                            detailMessage = "AudioTrack.ERROR_INVALID_OPERATION(" + length + ")";
                            break;
                        default:
                            detailMessage = "AudioTrack.ERROR UNKNOWN(" + length + ")";
                            break;
                    }
                    if (detailMessage != null)
                    {
                        WtcLog.error(TAG, detailMessage);
                        throw new IOException(detailMessage);
                    }
                    break; // exit while loop
                }
            }
        }
        catch (InterruptedException e)
        {
            WtcLog.warn(TAG, "run: InterruptedException; ignoring");
        }
        catch (Exception e)
        {
            WtcLog.error(TAG, "run: EXCEPTION", e);
            error = e;
        }
        finally
        {
            if (audioTrack != null)
            {
                try
                {
                    audioTrack.flush();
                    if (audioTrack.getState() == AudioTrack.STATE_INITIALIZED)
                    {
                        audioTrack.stop();
                    }
                    audioTrack.release();
                    audioTrack = null;
                }
                catch (Exception e)
                {
                    WtcLog.error(TAG, "run: EXCEPTION track.flush()/stop()/release()", e);
                }
            }

            onAudioPlayerStopped(error);

            WtcLog.info(TAG, "-run()");
        }
    }
}
