package com.twistpair.wave.thinclient.kexcrypto;

import com.twistpair.wave.thinclient.kexcrypto.WtcCryptoUtilPlatform.WtcEncryptorAes256Ecb;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoBase.IWtcPayloadTransformer;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoBase.IWtcTransformer;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoBase.WtcKexCryptoException;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoBase.WtcTransformerAes256EcbBase;
import com.twistpair.wave.thinclient.logging.WtcLog;

/**
 * <p>
 * See: <a href="http://en.wikipedia.org/wiki/Block_cipher_modes_of_operation#Counter_.28CTR.29">http://en.wikipedia.org/wiki/Block_cipher_modes_of_operation#Counter_.28CTR.29</a>
 * </p>
 */
public class WtcPayloadTransformerAes256Counter implements IWtcPayloadTransformer
{
    private static final String TAG          = WtcLog.TAG(WtcPayloadTransformerAes256Counter.class);

    /**
     * Equal to WtcTransformerAes256EcbBase.BLOCK_LENGTH
     */
    private static final int    BLOCK_LENGTH = WtcTransformerAes256EcbBase.BLOCK_LENGTH;

    protected IWtcTransformer[] transformers;
    protected byte[]            iv;

    public int getBlockLength()
    {
        return BLOCK_LENGTH;
    }

    /**
     * @param transformers Expects non-null/empty WtcEncryptorAes256Ecb[]
     */
    public void initialize(IWtcTransformer[] transformers, byte[] iv) throws IllegalArgumentException
    {
        if (transformers == null || transformers.length == 0)
        {
            throw new IllegalArgumentException("transformers must not be null or empty");
        }
        for (int i = 0; i < transformers.length; i++)
        {
            if (!(transformers[i] instanceof WtcEncryptorAes256Ecb))
            {
                throw new IllegalArgumentException("transformers[" + i + "] must be instance of WtcEncryptorAes256Ecb");
            }
        }
        this.transformers = transformers;

        if (iv == null || iv.length != getBlockLength())
        {
            throw new IllegalArgumentException("iv.length != getBlockLength(): " + getBlockLength());
        }
        this.iv = iv;
    }

    public void transformPayload(long extendedSequenceNumber, byte[] messageBuffer, int payloadOffset, int payloadLength,
                    byte[] workingBlockBuffer) throws WtcKexCryptoException
    {
        try
        {
            //WtcLog.info(TAG, "+transformPayload(" + extendedSequenceNumber + ", " + messageBuffer + ", " + payloadOffset + ", "
            //                + payloadLength + ")");

            if (workingBlockBuffer == null || workingBlockBuffer.length != iv.length)
            {
                throw new IllegalArgumentException("workingBlockBuffer.length != iv.length: " + iv.length);
            }

            IWtcTransformer transformer = transformers[(int) (extendedSequenceNumber % transformers.length)];
            int transformOffset, transformLength;
            int blockIndex = 0;

            //WtcLog.info(TAG, "messageBuffer(before)=" + WtcString.toHexString(messageBuffer, 0, messageLength));
            while (payloadLength > 0)
            {
                /*
                 * Directly setting the first 8 array values is faster than calling System.arraycopy(iv, 0, workingBlockBuffer, 0, 8).
                 * I verified this by alternating between both options ("A"=="direct setting", "B"=="System.arraycopy") and timing them.
                 * After >450K transforms, option "B" used noticeably more milliseconds than option "A".
                 * msA=10103, numA=455417, averageA=0.02218406427515881
                 * msB=12372, numB=455417, averageB=0.02716631131468522
                 */
                workingBlockBuffer[0] = iv[0];
                workingBlockBuffer[1] = iv[1];
                workingBlockBuffer[2] = iv[2];
                workingBlockBuffer[3] = iv[3];
                workingBlockBuffer[4] = iv[4];
                workingBlockBuffer[5] = iv[5];
                workingBlockBuffer[6] = iv[6];
                workingBlockBuffer[7] = iv[7];
                workingBlockBuffer[8] = (byte) (iv[8] ^ ((blockIndex & 0xff000000) >> 24));
                workingBlockBuffer[9] = (byte) (iv[9] ^ ((blockIndex & 0x00ff0000) >> 16));
                workingBlockBuffer[10] = (byte) (iv[10] ^ ((blockIndex & 0x0000ff00) >> 8));
                workingBlockBuffer[11] = (byte) (iv[11] ^ ((blockIndex & 0x000000ff) >> 0));
                workingBlockBuffer[12] = (byte) (iv[12] ^ ((extendedSequenceNumber & 0xff000000) >> 24));
                workingBlockBuffer[13] = (byte) (iv[13] ^ ((extendedSequenceNumber & 0x00ff0000) >> 16));
                workingBlockBuffer[14] = (byte) (iv[14] ^ ((extendedSequenceNumber & 0x0000ff00) >> 8));
                workingBlockBuffer[15] = (byte) (iv[15] ^ ((extendedSequenceNumber & 0x000000ff) >> 0));

                //WtcLog.info(TAG, "counter=" + WtcString.toHexString(counter));
                transformer.transform(workingBlockBuffer, 0, workingBlockBuffer, 0);
                //WtcLog.info(TAG, "buffer=" + WtcString.toHexString(buffer));

                // XOR the current message buffer block with the encrypted counter buffer.
                // The beauty of ECB Counter is that we don't need to walk to the end of the block.
                transformLength = Math.min(payloadLength, BLOCK_LENGTH);
                for (transformOffset = 0; transformOffset < transformLength; transformOffset++)
                {
                    messageBuffer[payloadOffset + transformOffset] ^= workingBlockBuffer[transformOffset];
                }
                //WtcLog.info(TAG, "messageBuffer(blockIndex=" + blockIndex + ")=" + WtcString.toHexString(messageBuffer, 0, messageLength));

                payloadOffset += transformLength;
                payloadLength -= transformLength;
                blockIndex++;
            }
            //WtcLog.info(TAG, "messageBuffer(after) =" + WtcString.toHexString(messageBuffer, 0, messageLength));
        }
        catch (Exception e)
        {
            WtcLog.error(TAG, "EXCEPTION: transformPayload", e);
            throw new WtcKexCryptoException("transformPayload", e);
        }
        //finally
        //{
        //    WtcLog.info(TAG, "-transformPayload(" + extendedSequenceNumber + ", " + messageBuffer + ", " + payloadOffset + ", "
        //                    + payloadLength + ")");
        //}
    }
}
