package com.twistpair.wave.thinclient;

import com.twistpair.wave.thinclient.net.WtcInetSocketAddressPlatform;
import com.twistpair.wave.thinclient.net.WtcUri;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpCode;
import com.twistpair.wave.thinclient.protocol.WtcpMessage;
import com.twistpair.wave.thinclient.protocol.headers.WtcpHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpErrorCode;
import com.twistpair.wave.thinclient.util.WtcString;
import com.twistpair.wave.thinclient.util.WtcThrowablePlatform;

public class WtcStackException //
                extends Exception
{
    private static final long serialVersionUID = -5817927286364811188L;

    /**
     * Intentionally Throwable so that BlackBerry doesn't swallow the stack trace.
     */
    public final Throwable    innerThrowable;

    /**
     * Only allow instances of subclasses.
     */
    protected WtcStackException()
    {
        this(null);
    }

    protected WtcStackException(Throwable innerThrowable)
    {
        this.innerThrowable = innerThrowable;
    }

    public static class WtcStackSecurityException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = -7598473310715972155L;

        public final int          kexSize;

        protected WtcStackSecurityException(int kexSize, Throwable innerThrowable)
        {
            super(innerThrowable);
            this.kexSize = kexSize;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [kexSize=" + kexSize + ", innerThrowable=" + innerThrowable
                            + ", stackTrace=" + WtcThrowablePlatform.toStackTraceString(innerThrowable) + "]";
        }
    }

    /**
     * WtcStack failed to initialize KEX Request prior to locating and connected to Proxy. 
     */
    public static class WtcStackSecurityInitializationException //
                    extends WtcStackSecurityException
    {
        private static final long serialVersionUID = 5103920114279898806L;

        public WtcStackSecurityInitializationException(int kexSize, Throwable innerThrowable)
        {
            super(kexSize, innerThrowable);
        }
    }

    /**
     * WtcStack failed to complete the KEX handshake based on the KEX Response from the Proxy.
     */
    public static class WtcStackSecurityAgreementException //
                    extends WtcStackSecurityException
    {
        private static final long serialVersionUID = -3110075628675143764L;

        public WtcStackSecurityAgreementException(int kexSize, Throwable innerThrowable)
        {
            super(kexSize, innerThrowable);
        }
    }

    /**
     * WtcStack failed to locate a Proxy.
     */
    public static class WtcStackProxyLocateException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = -8726861258264729352L;

        protected WtcStackProxyLocateException()
        {
            super();
        }

        protected WtcStackProxyLocateException(Throwable innerThrowable)
        {
            super(innerThrowable);
        }
    }

    public static class WtcStackProxyLocateLocatorException //
                    extends WtcStackProxyLocateException
    {
        private static final long serialVersionUID = -6208816106633309795L;

        public final WtcUri       locator;

        public WtcStackProxyLocateLocatorException(WtcUri locator, Throwable innerThrowable)
        {
            super(innerThrowable);
            this.locator = locator;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [locator=" + locator + ", innerThrowable=" + innerThrowable
                            + ", stackTrace=" + WtcThrowablePlatform.toStackTraceString(innerThrowable) + "]";
        }
    }

    /**
     * WtcStack failed to connect to a Proxy.
     */
    public static class WtcStackProxyConnectException //
                    extends WtcStackException
    {
        private static final long                 serialVersionUID = 7666409524855840032L;

        public final WtcInetSocketAddressPlatform proxy;

        public WtcStackProxyConnectException(WtcInetSocketAddressPlatform proxy, Throwable innerThrowable)
        {
            super(innerThrowable);
            this.proxy = proxy;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [proxy=" + proxy + ", innerThrowable=" + innerThrowable
                            + ", stackTrace=" + WtcThrowablePlatform.toStackTraceString(innerThrowable) + "]";
        }
    }

    public static class WtcStackThreadException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = 4892213481531605691L;

        public final String       source;
        public final WtcpMessage  message;

        protected WtcStackThreadException(String source, WtcpMessage message, Throwable innerThrowable)
        {
            super(innerThrowable);
            this.source = source;
            this.message = message;
        }

        public String toString()
        {
            StringBuffer sb = new StringBuffer() //
            .append(WtcString.getShortClassName(this)) //
            .append(" [source=").append(source) //
            .append(", message="); //
            switch (message.getMessageType())
            {
                case WtcpMessageType.Control:
                    sb.append(message.toString('a'));
                    break;
                default:
                    sb.append(message.toString('x'));
                    break;
            }
            sb.append(", innerThrowable=").append(innerThrowable)//
            .append(", stackTrace=").append(WtcThrowablePlatform.toStackTraceString(innerThrowable))//
            .append(']');
            return sb.toString();
        }
    }

    /**
     * WtcStack encountered an error while sending a WtcpMessage.
     */
    public static class WtcStackThreadSendException //
                    extends WtcStackThreadException
    {
        private static final long serialVersionUID = 7260979662066040144L;

        public WtcStackThreadSendException(String source, WtcpMessage message, Throwable innerThrowable)
        {
            super(source, message, innerThrowable);
        }
    }

    /**
     * WtcStack encountered an error while processing a received WtcpMessage.
     */
    public static class WtcStackThreadProcessReceivedMessagesException //
                    extends WtcStackThreadException
    {
        private static final long serialVersionUID = -3452870499323479648L;

        public WtcStackThreadProcessReceivedMessagesException(String source, WtcpMessage message, Throwable innerThrowable)
        {
            super(source, message, innerThrowable);
        }
    }

    /**
     * WtcStack failed to receive a timely Response to a WtcpMessage Request.
     */
    public static class WtcStackMessageRequestResponseTimeoutException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = -2055063305631083722L;

        public final long         expectedMs;
        public final long         actualMs;
        public final byte         messageType;
        public final int          opCode;
        public final int          transactionId;

        public WtcStackMessageRequestResponseTimeoutException(long expectedMs, long actualMs, //
                        byte messageType, int opCode, int transactionId)
        {
            this.expectedMs = expectedMs;
            this.actualMs = actualMs;
            this.messageType = messageType;
            this.opCode = opCode;
            this.transactionId = transactionId;
        }

        public String toString()
        {
            return new StringBuffer() //
            .append(WtcString.getShortClassName(this)) //
            .append(" [") //
            .append("expectedMs=").append(expectedMs) //
            .append(", actualMs=").append(actualMs) //
            .append(", messageType=").append(WtcpMessageType.toString(messageType)) //
            .append(", opCode=").append(WtcpOpCode.toString(opCode)) //
            .append(", transactionId=").append(transactionId) //
            .append(']') //
            .toString();
        }
    }

    public static class WtcStackMessageReceiveException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = -6610253798267797196L;

        /**
         * Only allow instances of subclasses.
         */
        protected WtcStackMessageReceiveException()
        {
        }
    }

    /**
     * WtcStack read more than the expected/maximum payload bytes, or more than WtcpHeader.MAX_PAYLOAD_LENGTH bytes.
     */
    public static class WtcStackMessageReceiveOverflowException //
                    extends WtcStackMessageReceiveException
    {
        private static final long serialVersionUID = -4196740541075378088L;

        public final int          maximumLength;
        public final int          actualLength;

        public WtcStackMessageReceiveOverflowException(int maximumLength, int actualLength)
        {
            this.maximumLength = maximumLength;
            this.actualLength = actualLength;
        }

        public String toString()
        {
            return new StringBuffer() //
            .append(WtcString.getShortClassName(this)) //
            .append(" [") //
            .append("maximumLength=").append(maximumLength) //
            .append(", actualLength=").append(actualLength) //
            .append(']') //
            .toString();
        }
    }

    /**
     * WtcStack read less than the expected/minimum payload bytes.
     * TODO:(pv) This class isn't actually used? Remove?
     */
    public static class WtcStackMessageReceiveUnderflowException //
                    extends WtcStackMessageReceiveException
    {
        private static final long serialVersionUID = -6582578480348308616L;

        public final int          minimumLength;
        public final int          actualLength;

        public WtcStackMessageReceiveUnderflowException(int minimumLength, int actualLength)
        {
            this.minimumLength = minimumLength;
            this.actualLength = actualLength;
        }

        public String toString()
        {
            return new StringBuffer() //
            .append(WtcString.getShortClassName(this)) //
            .append(" [") //
            .append("minimumLength=").append(minimumLength) //
            .append(", actualLength=").append(actualLength) //
            .append(']') //
            .toString();
        }
    }

    /**
     * WtcStack encountered an unexpected WtcpMessageType.
     * This can happen if encryption/decryption fails.
     */
    public static class WtcStackMessageReceiveTypeUnknownException //
                    extends WtcStackMessageReceiveException
    {
        private static final long serialVersionUID = -7268841278103344103L;

        public final byte         messageType;

        public WtcStackMessageReceiveTypeUnknownException(byte messageType)
        {
            this.messageType = messageType;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [messageType=" + WtcpMessageType.toString(messageType) + "]";
        }
    }

    /**
     * WtcStack received an invalid WtcpHeader.
     */
    public static class WtcStackMessageReceiveHeaderInvalidException //
                    extends WtcStackMessageReceiveException
    {
        private static final long serialVersionUID = -7949453646823246973L;

        public final WtcpHeader   header;

        public WtcStackMessageReceiveHeaderInvalidException(WtcpHeader header)
        {
            this.header = header;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [header=" + header + "]";
        }
    }

    /**
     * WtcStack received an unexpected Response from the Proxy.
     */
    public static class WtcStackMessageReceiveResponseUnexpectedException //
                    extends WtcStackMessageReceiveException
    {
        private static final long serialVersionUID = 360662926229738916L;

        public final int          opCode;
        public final int          transactionId;

        public WtcStackMessageReceiveResponseUnexpectedException(int opCode, int transactionId)
        {
            this.opCode = opCode;
            this.transactionId = transactionId;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [opCode=" + WtcpOpCode.toString(opCode) //
                            + ", transactionId=" + transactionId + "]";
        }
    }

    /**
     * WAVE User has no Profiles.
     * <p>A WAVE User must always have at least one Profile in order to use WAVE.<br>
     * This is a fatal exception that requires the User to contact their WAVE Administrator.</p>
     */
    public static class WtcStackUserProfilesEmptyException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = -7950612428704358476L;

        public String toString()
        {
            return WtcString.getShortClassName(this);
        }
    }

    /**
     * WtcStack processed a SessionClose message (both Solicited and Unsolicited).
     */
    public static class WtcStackSessionCloseException //
                    extends WtcStackException
    {
        private static final long  serialVersionUID = 9018546009310564639L;

        public final boolean       unsolicited;
        public final WtcpErrorCode errorCode;

        public WtcStackSessionCloseException(boolean unsolicited, WtcpErrorCode errorCode)
        {
            this.unsolicited = unsolicited;
            if (errorCode == null)
            {
                errorCode = WtcpErrorCode.OK;
            }
            this.errorCode = errorCode;
        }

        public String toString()
        {
            return WtcString.getShortClassName(this) + " [unsolicited=" + unsolicited + ", errorCode=" + errorCode + "]";
        }
    }

    /**
     * WtcStack detected that the remote Proxy closed the socket.
     * There is no error code associated with this; it "just happens". 
     */
    public static class WtcStackRemoteDisconnectException //
                    extends WtcStackException
    {
        private static final long serialVersionUID = -116123158739893286L;

        public String toString()
        {
            return WtcString.getShortClassName(this);
        }
    }
}
