package com.twistpair.wave.thinclient;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.twistpair.wave.thinclient.WtcLocatorException.WtcLocatorResponseInvalidException;
import com.twistpair.wave.thinclient.net.WtcInetSocketAddressPlatform;
import com.twistpair.wave.thinclient.util.WtcString;

public class WtcProxyInfo
{
    public static String toString(WtcProxyInfo[] proxyInfos)
    {
        StringBuffer sb = new StringBuffer();
        sb.append('[');
        for (int i = 0; i < proxyInfos.length; i++)
        {
            if (i > 0)
            {
                sb.append(',');
            }
            sb.append(proxyInfos[i].toString());
        }
        sb.append(']');
        return sb.toString();
    }

    private String                         serverId;
    private int                            proxyIndex;
    private WtcInetSocketAddressPlatform[] inetSocketAddresses;
    private String                         softwareVersionCheckURL;

    /**
     * Creates a "direct" WtcProxyInfo with no info for serverId, proxyIndex, or softwareVersionCheckURL.
     * Note that this type of WtcProxyInfo cannot be used to check for software updates.
     * @param inetSocketAddress the direct address to the WTCP Proxy Server.
     */
    public WtcProxyInfo(WtcInetSocketAddressPlatform inetSocketAddress)
    {
        initialize(null, -1, new WtcInetSocketAddressPlatform[]
        {
            inetSocketAddress
        }, null);
    }

    /**
     * Creates a WtcProxyInfo based on an XML response from a WTCP Locator Server.
     * @param proxyNode
     * @throws WtcLocatorResponseException
     */
    public WtcProxyInfo(Node proxyNode) throws WtcLocatorResponseInvalidException
    {
        NamedNodeMap attributes = proxyNode.getAttributes();
        if (attributes == null)
        {
            throw new WtcLocatorResponseInvalidException("Proxy node has no attributes", proxyNode);
        }

        String serverId = attributes.getNamedItem("ServerID").getNodeValue();
        String proxyIndex = attributes.getNamedItem("ProxyIndex").getNodeValue();
        String address1 = attributes.getNamedItem("Address1").getNodeValue();
        String port1 = attributes.getNamedItem("Port1").getNodeValue();
        String address2 = attributes.getNamedItem("Address2").getNodeValue();
        String port2 = attributes.getNamedItem("Port2").getNodeValue();
        String softwareVersionCheckURL = attributes.getNamedItem("SoftwareVersionCheckURL").getNodeValue();

        if (WtcString.isNullOrEmpty(serverId) //
                        || WtcString.isNullOrEmpty(proxyIndex) //
                        || WtcString.isNullOrEmpty(address1) //
                        || WtcString.isNullOrEmpty(port1))
        {
            throw new WtcLocatorResponseInvalidException(
                            "Proxy node missing required attributes: ServerID, ProxyIndex, Address1, Port1", proxyNode);
        }

        WtcInetSocketAddressPlatform[] inetSocketAddresses;

        if (WtcString.isNullOrEmpty(address2) || WtcString.isNullOrEmpty(port2))
        {
            inetSocketAddresses = new WtcInetSocketAddressPlatform[]
            {
                new WtcInetSocketAddressPlatform(address1, Integer.parseInt(port1)),
            };
        }
        else
        {
            inetSocketAddresses =
                new WtcInetSocketAddressPlatform[]
                {
                    new WtcInetSocketAddressPlatform(address1, Integer.parseInt(port1)),
                    new WtcInetSocketAddressPlatform(address2, Integer.parseInt(port2)),
                };
        }

        initialize(serverId, Integer.parseInt(proxyIndex), inetSocketAddresses, softwareVersionCheckURL);
    }

    private void initialize(String serverId, int proxyIndex, WtcInetSocketAddressPlatform[] inetSocketAddresses,
                    String softwareVersionCheckURL)
    {
        if (inetSocketAddresses == null || inetSocketAddresses.length == 0)
        {
            throw new IllegalArgumentException("inetSocketAddresses cannot be null or empty");
        }

        this.serverId = serverId;
        this.proxyIndex = proxyIndex;
        this.inetSocketAddresses = inetSocketAddresses;
        this.softwareVersionCheckURL = softwareVersionCheckURL;
    }

    public String toString()
    {
        StringBuffer sb = new StringBuffer().append("{serverId=\"").append(serverId) //
        .append("\", proxyIndex=").append(proxyIndex) // 
        .append(", inetSocketAddresses=[");
        for (int i = 0; i < inetSocketAddresses.length; i++)
        {
            if (i > 0)
            {
                sb.append(',');
            }
            sb.append(inetSocketAddresses[i]);
        }
        sb.append("], softwareVersionCheckURL=\"").append(softwareVersionCheckURL).append("\"}").toString();
        return sb.toString();
    }

    public String getServerId()
    {
        return serverId;
    }

    public int getProxyIndex()
    {
        return proxyIndex;
    }

    public WtcInetSocketAddressPlatform[] getInetSocketAddresses()
    {
        return inetSocketAddresses;
    }

    public String getSoftwareVersionCheckURL()
    {
        return softwareVersionCheckURL;
    }
}
