package com.twistpair.wave.thinclient;

import java.util.Enumeration;

import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpMessageType;
import com.twistpair.wave.thinclient.util.WtcIntegerObjectMapPlatform;

/**
 * TODO:(pv) Make this a singleton that tracks stats on *ALL* sessions, and report stats per-session or all-sessions.
 * TODO:(pv) Track count and smallest/average/largest size of each message op code.
 */
public class WtcConnectionStatistics
{
    private static final String  TAG         = WtcLog.TAG(WtcConnectionStatistics.class);

    private static final boolean VERBOSE_LOG = false;

    public class LatencyHistory
    {
        private final String    TAG          = WtcLog.TAG(LatencyHistory.class);

        public static final int HISTORY_SIZE = 10;

        /**
         * The total number of latency values ever added.
         */
        public long getCount()
        {
            return count;
        }

        private long count;

        /**
         * Summation of all latency values ever added.
         * Think of this as the total time [wasted] waiting for a response from the Proxy.
         */
        public long getSumMs()
        {
            return sumMs;
        }

        private long sumMs;

        /**
         * The best latency value ever added.
         * I initially thought this wouldn't be interesting to track,
         * but if this value is high then it proves that the connection is bad.
         */
        public int getBestMs()
        {
            return bestMs;
        }

        private int bestMs;

        /**
         * The worst latency value ever added.
         */
        public int getWorstMs()
        {
            return worstMs;
        }

        private int worstMs;

        /**
         * Average of all latency values ever added.
         * Actually: (int) (sumMs / count)
         */
        public int getAverageMs()
        {
            return averageMs;
        }

        private int averageMs;

        /**
         * Average of the last HISTORY_SIZE latency values that have been added.
         */
        public int getAverageMsLastX()
        {
            return averageMsLastX;
        }

        private int  averageMsLastX;

        /**
         * Summation of the last HISTORY_SIZE latency values that have been added.
         */
        private int  sumMsLastX;

        /**
         * Used to decrement sumMsLastX before adding a new latency value.
         * Basically, if sumLastX represents the sum of the last latency values,
         * then decrement sumLastX by this last known latency value,
         * and then add the new latency value.
         * This is lighter weight than maintaining a HISTORY_SIZE element collection of actual values.
         */
        private long previousMs;

        /**
         * @return the calculated worst case milliseconds that it should take to receive a Response to a Request 
         */
        public int getRequestTimeoutMs()
        {
            return requestTimeoutMs;
        }

        private int requestTimeoutMs;

        public LatencyHistory()
        {
            reset();
        }

        public synchronized void reset()
        {
            // TODO:(pv) Selectively reset "all" or "session" stats
            count = 0;
            sumMs = 0;
            bestMs = Integer.MAX_VALUE;
            worstMs = Integer.MIN_VALUE;
            averageMs = 0;
            averageMsLastX = 0;
            sumMsLastX = 0;
            previousMs = 0;
            requestTimeoutMs = WtcStack.TIMEOUT_REQUEST_TX_RESPONSE_RX_MS_DEFAULT;
        }

        public synchronized void log()
        {
            WtcLog.debug(TAG, "$LATENCY: " + this);
        }

        /**
         * NOTE:(pv) My experience/estimate for average latencies are:
         * <ul>
         * <li>80% of the time latency &lt; 1s; messages should time out in ~5s</li>
         * <li>15% of the time 1s &lt; latency &lt; 10s; messages should time out in ~15s</li>
         * <li>5% of the time 10s &lt; latency &lt; 20s; messages should time out in ~25s</li>
         * </ul>
         * Any latency over 20s results in a highly unusable product and needs to be corrected.<br>
         * At first I thought of having a logarithmic, or more complex, timeout.<br>
         * Although fairly simple math, that seems too heavy/complex for our needs.<br>
         * For now, a reasonable linear message timeout should be sufficient.
         * @param latencyMs
         * @return REQUEST_TIMEOUT_MIN + (2 * Math.max(averageMs, averageMsLastX))
         */
        public synchronized int add(int latencyMs)
        {
            if (VERBOSE_LOG)
            {
                WtcLog.debug(TAG, "$LATENCY: add(" + latencyMs + ")");
            }

            if (latencyMs < bestMs)
            {
                bestMs = latencyMs;
            }

            if (latencyMs > worstMs)
            {
                worstMs = latencyMs;
            }

            count++;
            sumMs += latencyMs;
            averageMs = (int) (sumMs / count);

            if (count >= HISTORY_SIZE)
            {
                sumMsLastX = (int) (sumMsLastX - previousMs + latencyMs);
                averageMsLastX = sumMsLastX / HISTORY_SIZE;
            }
            else
            {
                sumMsLastX = (int) sumMs;
                averageMsLastX = averageMs;
            }

            requestTimeoutMs = WtcStack.TIMEOUT_REQUEST_TX_RESPONSE_RX_MS_MIN + (2 * Math.max(averageMs, averageMsLastX));

            if (VERBOSE_LOG)
            {
                this.log();
            }

            previousMs = latencyMs;

            return requestTimeoutMs;
        }

        //@Override
        public String toString()
        {
            return new StringBuffer() //
            .append('{') //
            .append("count=").append(count) //
            .append(", sumMs=").append(sumMs) //
            .append(", bestMs=").append(bestMs) //
            .append(", worstMs=").append(worstMs) //
            .append(", averageMs=").append(averageMs) //
            .append(", sumMsLastX=").append(sumMsLastX) //
            .append(", averageMsLastX=").append(averageMsLastX) //
            .append(", requestTimeoutMs=").append(requestTimeoutMs) //
            .append('}') //
            .toString();
        }
    }

    public class Jitter
    {
        private final String    TAG          = WtcLog.TAG(Jitter.class);

        public static final int HISTORY_SIZE = 10;

        /**
         * The total number of latency values ever added.
         */
        public long getCount()
        {
            return count;
        }

        private long count;

        /**
         * Summation of all latency values ever added.
         * Think of this as the total time [wasted] waiting for a response from the Proxy.
         */
        public long getSumMs()
        {
            return sumMs;
        }

        private long sumMs;

        /**
         * The best latency value ever added.
         * I initially thought this wouldn't be interesting to track,
         * but if this value is high then it proves that the connection is bad.
         */
        public int getBestMs()
        {
            return bestMs;
        }

        private int bestMs;

        /**
         * The worst latency value ever added.
         */
        public int getWorstMs()
        {
            return worstMs;
        }

        private int worstMs;

        /**
         * Average of all latency values ever added.
         * Actually: (int) (sumMs / count)
         */
        public int getAverageMs()
        {
            return averageMs;
        }

        private int averageMs;

        /**
         * Average of the last HISTORY_SIZE latency values that have been added.
         */
        public int getAverageMsLastX()
        {
            return averageMsLastX;
        }

        private int  averageMsLastX;

        /**
         * Summation of the last HISTORY_SIZE latency values that have been added.
         */
        private int  sumMsLastX;

        /**
         * Used to decrement sumMsLastX before adding a new latency value.
         * Basically, if sumLastX represents the sum of the last latency values,
         * then decrement sumLastX by this last known latency value,
         * and then add the new latency value.
         * This is lighter weight than maintaining a HISTORY_SIZE element collection of actual values.
         */
        private long previousMs;

        public Jitter()
        {
            reset();
        }

        public synchronized void reset()
        {
            // TODO:(pv) Selectively reset "all" or "session" stats
            count = 0;
            sumMs = 0;
            bestMs = Integer.MAX_VALUE;
            worstMs = Integer.MIN_VALUE;
            averageMs = 0;
            averageMsLastX = 0;
            sumMsLastX = 0;
            previousMs = 0;
            //requestTimeoutMs = WtcStack.REQUEST_TIMEOUT_MS_DEFAULT;
        }

        public synchronized void log()
        {
            WtcLog.debug(TAG, "$JITTER: count=" + count + ", sumMs=" + sumMs //
            //+ ", bestMs=" + bestMs + ", worstMs=" + worstMs + ", averageMs=" + averageMs //
            //+ ", sumMsLastX=" + sumMsLastX + ", averageMsLastX=" + averageMsLastX //
            //+ ", requestTimeoutMs=" + requestTimeoutMs //
            );
        }

        public synchronized void add(int latencyMs)
        {
            if (VERBOSE_LOG)
            {
                WtcLog.debug(TAG, "$JITTER: add(" + latencyMs + ")");
            }

            if (latencyMs < bestMs)
            {
                bestMs = latencyMs;
            }

            if (latencyMs > worstMs)
            {
                worstMs = latencyMs;
            }

            count++;
            sumMs += latencyMs;
            averageMs = (int) (sumMs / count);

            if (count >= HISTORY_SIZE)
            {
                sumMsLastX = (int) (sumMsLastX - previousMs + latencyMs);
                averageMsLastX = sumMsLastX / HISTORY_SIZE;
            }
            else
            {
                sumMsLastX = (int) sumMs;
                averageMsLastX = averageMs;
            }

            //requestTimeoutMs = (int) (WtcStack.REQUEST_TIMEOUT_MS_MIN + (2 * Math.max(averageMs, averageMsLastX)));

            if (VERBOSE_LOG)
            {
                this.log();
            }

            previousMs = latencyMs;

            //return requestTimeoutMs;
        }
    }

    private int                               locatorAttempts      = 0;
    private int                               locatorSuccess       = 0;
    private int                               proxyConnectAttempts = 0;
    private int                               proxyConnectSuccess  = 0;
    private int                               proxyDisconnects     = 0;

    //private long                duration             = 0;

    private final WtcMessageCounter           messageCounterTotal;

    /**
     * Map&lt;WtcpMessageType(Byte), WtcMessageCounter&gt;
     */
    private final WtcIntegerObjectMapPlatform messageCounterByType;

    public final LatencyHistory               latency              = new LatencyHistory();
    public final Jitter                       jitter               = new Jitter();

    public WtcConnectionStatistics()
    {
        messageCounterTotal = new WtcMessageCounter();
        messageCounterByType = new WtcIntegerObjectMapPlatform();

        // TODO:(pv) Find a J2ME way to use reflection and get all values of WtcpMessageType
        messageCounterByType.put(WtcpMessageType.Reserved0, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.Hello, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.UdpHello, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.KeyExchange, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.Control, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.Media, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.Reserved6, new WtcMessageCounter());
        messageCounterByType.put(WtcpMessageType.Reserved7, new WtcMessageCounter());

        reset();
    }

    public void reset()
    {
        // TODO:(pv) Selectively reset "all" or "session" stats
        locatorAttempts = 0;
        locatorSuccess = 0;
        proxyConnectAttempts = 0;
        proxyConnectSuccess = 0;
        proxyDisconnects = 0;

        //duration = 0;

        messageCounterTotal.reset();
        Enumeration keys = messageCounterByType.keys();
        while (keys.hasMoreElements())
        {
            byte messageType = ((Integer) keys.nextElement()).byteValue();
            WtcMessageCounter messageCounter = (WtcMessageCounter) messageCounterByType.get(messageType);
            messageCounter.reset();
        }

        // TODO:(pv) Why is this commented out in wtcsdk-dotnet?
        latency.reset();
        jitter.reset();
    }

    public String toString()
    {
        StringBuffer sb = new StringBuffer() //
        .append('{') //
        .append("locatorAttempts=").append(locatorAttempts) //
        .append(", locatorSuccess=").append(locatorSuccess) //
        .append(", proxyConnectAttempts=").append(proxyConnectAttempts) //
        .append(", proxyConnectSuccess=").append(proxyConnectSuccess) //
        .append(", proxyDisconnects=").append(proxyDisconnects) //
        //.append(", duration=").append(duration)
        .append(", messageCounterTotal=").append(messageCounterTotal);

        sb.append(", messageCounterByType={");
        Enumeration keys = messageCounterByType.keys();
        while (keys.hasMoreElements())
        {
            byte messageType = ((Integer) keys.nextElement()).byteValue();
            WtcMessageCounter messageCounter = (WtcMessageCounter) messageCounterByType.get(messageType);

            sb.append(WtcpMessageType.toString(messageType)).append(':').append(messageCounter);
            if (keys.hasMoreElements())
            {
                sb.append(", ");
            }
        }
        sb.append('}');

        return sb.append(", Latency=").append(latency) //
        .append('}') //
        .toString();
    }

    public void log()
    {
        WtcLog.debug(TAG, "$STATISTICS: " + toString());
        latency.log();
        //jitter.log();
    }

    public void incLocatorAttempts()
    {
        locatorAttempts++;
    }

    public int getLocatorAttempts()
    {
        return locatorAttempts;
    }

    public void incLocatorSuccess()
    {
        locatorSuccess++;
    }

    public int getLocatorSuccess()
    {
        return locatorSuccess;
    }

    public void incProxyConnectAttempts()
    {
        proxyConnectAttempts++;
    }

    public int getProxyConnectAttempts()
    {
        return proxyConnectAttempts;
    }

    public void incProxyConnectSuccess()
    {
        proxyConnectSuccess++;
    }

    public int getProxyConnectSuccess()
    {
        return proxyConnectSuccess;
    }

    public void incProxyDisconnects()
    {
        proxyDisconnects++;
    }

    public int getProxyDisconnects()
    {
        return proxyDisconnects;
    }

    void incTxed(byte messageType, long length)
    {
        messageCounterTotal.incTxed(length);
        getMessageCounterByType(messageType).incTxed(length);
        // TODO:(pv) listener.onConnectionStatisticsUpdated(this);
    }

    void incRxed(byte messageType, long length)
    {
        messageCounterTotal.incRxed(length);
        getMessageCounterByType(messageType).incRxed(length);
        // TODO:(pv) listener.onConnectionStatisticsUpdated(this);
    }

    public WtcMessageCounter getMessageCounterTotal()
    {
        return messageCounterTotal;
    }

    public WtcMessageCounter getMessageCounterByType(byte messageType)
    {
        return (WtcMessageCounter) messageCounterByType.get(messageType);
    }

    public class WtcMessageCounter
    {
        long txedMessages;

        public long getTxedMessages()
        {
            return txedMessages;
        }

        long txedBytes;

        public long getTxedBytes()
        {
            return txedBytes;
        }

        long rxedMessages;

        public long getRxedMessages()
        {
            return rxedMessages;
        }

        long rxedBytes;

        public long getRxedBytes()
        {
            return rxedBytes;
        }

        public void reset()
        {
            txedMessages = 0;
            txedBytes = 0;
            rxedMessages = 0;
            rxedBytes = 0;
        }

        void incTxed(long length)
        {
            txedMessages++;
            txedBytes += length;
        }

        void incRxed(long length)
        {
            rxedMessages++;
            rxedBytes += length;
        }

        public String toString()
        {
            return new StringBuffer() //
            .append('{') //
            .append("TxedMessages=").append(txedMessages) //
            .append(", TxedBytes=").append(txedBytes) //
            .append(", RxedMessages=").append(rxedMessages) //
            .append(", RxedBytes=").append(rxedBytes) //
            .append('}') //
            .toString();
        }
    }
}
