package com.twistpair.wave.thinclient;

import com.twistpair.wave.thinclient.WtcClientException.WtcClientUnsupportedCodecException;
import com.twistpair.wave.thinclient.WtcStackException.WtcStackSessionCloseException;
import com.twistpair.wave.thinclient.kexcrypto.WtcKexCryptoClient.WtcKexPrimeSize;
import com.twistpair.wave.thinclient.logging.WtcLog;
import com.twistpair.wave.thinclient.media.WtcMediaCodec;
import com.twistpair.wave.thinclient.media.WtcMediaCodecGSMPlatform;
import com.twistpair.wave.thinclient.media.WtcMediaDeviceMicrophone;
import com.twistpair.wave.thinclient.media.WtcMediaDeviceSpeaker;
import com.twistpair.wave.thinclient.net.WtcInetSocketAddressPlatform;
import com.twistpair.wave.thinclient.net.WtcUri;
import com.twistpair.wave.thinclient.net.WtcUriPlatform;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpAudioCodec;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpChannelChange;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpEndpointFilterType;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpEndpointFlags;
import com.twistpair.wave.thinclient.protocol.WtcpConstants.WtcpOpType;
import com.twistpair.wave.thinclient.protocol.headers.WtcpControlHeader;
import com.twistpair.wave.thinclient.protocol.headers.WtcpMediaHeader;
import com.twistpair.wave.thinclient.protocol.types.WtcpAddressBookInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallAnswer;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallDtmf;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallHangup;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallInfo;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallOffer;
import com.twistpair.wave.thinclient.protocol.types.WtcpCallProgress;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelActivity;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelIdErrorDictionary;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelIdList;
import com.twistpair.wave.thinclient.protocol.types.WtcpChannelInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpEndpointInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpEndpointProperties;
import com.twistpair.wave.thinclient.protocol.types.WtcpErrorCode;
import com.twistpair.wave.thinclient.protocol.types.WtcpKeyValueList;
import com.twistpair.wave.thinclient.protocol.types.WtcpProfileInfoList;
import com.twistpair.wave.thinclient.protocol.types.WtcpStringList;
import com.twistpair.wave.thinclient.util.WtcInt16;
import com.twistpair.wave.thinclient.util.WtcInt32;
import com.twistpair.wave.thinclient.util.WtcInt8;
import com.twistpair.wave.thinclient.util.WtcString;
import com.twistpair.wave.thinclient.util.WtcVersionString;

/**
 * <p>A stateful wrapper around the stateless WtcStack that handles WtcStackListener events.</p>
 * <p>The event handling maintains state and populates variables for use by an application.</p>
 * <p>Also wraps a WtcStackListener to allow passing of events up an application for further notification/processing.</p>
 */
public class WtcClient extends WtcStackListener
{
    private static final String            TAG                            = WtcLog.TAG(WtcClient.class);

    public static final String             WAVE_LICENSE_WTC_ANDROID       = "{E030C868-E1FC-43E2-8CF1-85B977BD590C}";
    public static final String             WAVE_LICENSE_WTC_ANDROID_IPTEL = "{E420E0EA-BD45-49fa-8EBD-832E33149880}";

    public static final String             LICENSE_DEFAULT                = WAVE_LICENSE_WTC_ANDROID;
    public static final int                KEX_DEFAULT                    = WtcKexPrimeSize.P1024;
    public static final byte               AUDIO_CODEC_DEFAULT            = WtcpAudioCodec.PCM_Mono16bBigEndian8kHz;
    //public static final byte                AUDIO_CODEC_DEFAULT            = WtcpAudioCodec.GSM_full_rate;
    //public static final byte                AUDIO_CODEC_DEFAULT            = WtcpAudioCodec.Speex_NB_8000;
    public static final byte               AUDIO_SCALE_DEFAULT            = 2;
    public static final byte               SESSION_TIMEOUT_DEFAULT        = WtcStack.TIMEOUT_SESSION_SECONDS_MAX;
    public static final int                CONNECT_TIMEOUT_DEFAULT        = 0;
    public static final String             PLATFORM_DESCRIPTION_DEFAULT   = TAG;

    private final WtcMediaDeviceMicrophone microphone;
    private final WtcMediaDeviceSpeaker    speaker;

    private final WtcClientChannelManager  channelManager;

    private final Object                   syncConnection                 = new Object();

    private final WtcVersionString         version;

    private WtcStack                       stack;

    // Fields specified at connection time
    private String                         platformDescription;
    private String                         username;
    private String                         password;
    private WtcInt8                        sessionTimeoutSeconds;
    private String                         license;
    private String                         profileId;
    private WtcInt8                        audioCodec;
    private WtcInt8                        audioScale;

    // Fields populated during connection
    private String                         sessionId;

    /**
     * <b>NOT</b> synchronized! (for performance reasons)<br>
     * All uses of this variable should save a reference and work off of that!
     */
    private WtcClientListener              listener;

    private WtcClientPhoneLineManager      phoneLineManager;
    private WtcClientPhoneCallManager      phoneCallManager;

    public WtcMediaDeviceMicrophone getMicrophone()
    {
        return microphone;
    }

    public WtcMediaDeviceSpeaker getSpeaker()
    {
        return speaker;
    }

    /**
     * @return never null 
     */
    public WtcClientChannelManager getChannelManager()
    {
        return channelManager;
    }

    /**
     * @return may be null
     */
    public WtcClientPhoneLineManager getPhoneLineManager()
    {
        return phoneLineManager;
    }

    /**
     * @return may be null
     */
    public WtcClientPhoneCallManager getPhoneCallManager()
    {
        return phoneCallManager;
    }

    public String getSessionId()
    {
        return sessionId;
    }

    public WtcClient(WtcMediaDeviceMicrophone microphone, WtcMediaDeviceSpeaker speaker, //
                    WtcVersionString version)
    {
        try
        {
            WtcLog.debug(TAG, "+WtcClient");

            if (microphone == null)
            {
                throw new IllegalArgumentException("microphone cannot be null");
            }

            if (speaker == null)
            {
                throw new IllegalArgumentException("speaker cannot be null");
            }

            this.microphone = microphone;
            this.speaker = speaker;

            this.channelManager = new WtcClientChannelManager(this);

            this.version = version;

            clear();
        }
        finally
        {
            WtcLog.debug(TAG, "-WtcClient");
        }
    }

    private void clear()
    {
        synchronized (syncConnection)
        {
            //microphone.close(false, null);
            //speaker.close(false);

            // TODO:(pv) Disconnect here if not null?
            stack = null;

            channelManager.clear();

            if (phoneLineManager != null)
            {
                phoneLineManager.clear();
                phoneLineManager = null;
            }

            if (phoneCallManager != null)
            {
                phoneCallManager.clear();
                phoneCallManager = null;
            }

            // Populated in connect(...)
            platformDescription = null;
            username = null;
            password = null;
            sessionTimeoutSeconds = null;
            license = null;
            profileId = null;
            audioCodec = null;
            audioScale = null;

            // Populated in onSessionOpened/onSessionResumed and clears in onSessionClosed
            sessionId = null;
        }
    }

    public void connect(WtcClientListener listener, //
                    WtcConnectionStatistics connectionStatistics, //
                    String platformDescription, WtcUri[] uriServers, //
                    String username, String password)
    {
        connect(listener, connectionStatistics, platformDescription, KEX_DEFAULT, uriServers, //
                        username, password);
    }

    public void connect(WtcClientListener listener, //
                    WtcConnectionStatistics connectionStatistics, //
                    String platformDescription, int kexSize, WtcUri[] uriServers, //
                    String username, String password)
    {
        connect(listener, connectionStatistics, platformDescription, kexSize, uriServers, username, password, //
                        SESSION_TIMEOUT_DEFAULT);
    }

    public void connect(WtcClientListener listener, //
                    WtcConnectionStatistics connectionStatistics, //
                    String platformDescription, int kexSize, WtcUri[] uriServers, //
                    String username, String password, //
                    byte sessionTimeoutSeconds)
    {
        try
        {
            connect(listener, connectionStatistics, platformDescription, kexSize, uriServers, username, password, //
                            sessionTimeoutSeconds, //
                            AUDIO_CODEC_DEFAULT, AUDIO_SCALE_DEFAULT);
        }
        catch (WtcClientUnsupportedCodecException e)
        {
            WtcLog.error(TAG, "Should never happen, as long as AUDIO_CODEC_DEFAULT is valid", e);
        }
    }

    public void connect(WtcClientListener listener, //
                    WtcConnectionStatistics connectionStatistics, //
                    String platformDescription, int kexSize, WtcUri[] uriServers, //
                    String username, String password, //
                    byte sessionTimeoutSeconds, // 
                    byte audioCodec, byte audioScale) //
                    throws WtcClientUnsupportedCodecException
    {
        try
        {
            connect(listener, connectionStatistics, platformDescription, kexSize, uriServers, username, password, //
                            sessionTimeoutSeconds, //
                            audioCodec, audioScale, //
                            LICENSE_DEFAULT, null);
        }
        catch (WtcClientUnsupportedCodecException e)
        {
            WtcLog.error(TAG, "Should never happen, as long as AUDIO_CODEC_DEFAULT is valid", e);
        }
    }

    public void connect(WtcClientListener listener, //
                    WtcConnectionStatistics connectionStatistics, //
                    String platformDescription, int kexSize, WtcUri[] uriServers, //
                    String username, String password, //
                    byte sessionTimeoutSeconds, // 
                    byte audioCodec, byte audioScale, //
                    String license, String profileId) //
                    throws WtcClientUnsupportedCodecException
    {
        try
        {
            connect(listener, connectionStatistics, platformDescription, kexSize, uriServers, username, password, //
                    sessionTimeoutSeconds, //
                    audioCodec, audioScale, //
                    license, profileId, //
                    CONNECT_TIMEOUT_DEFAULT, null);
        }
        catch (WtcClientUnsupportedCodecException e)
        {
            WtcLog.error(TAG, "Should never happen, as long as AUDIO_CODEC_DEFAULT is valid", e);
        }
    }

    public void connect(WtcClientListener listener, //
                    WtcConnectionStatistics connectionStatistics, //
                    String platformDescription, int kexSize, WtcUri[] uriServers, //
                    String username, String password, //
                    byte sessionTimeoutSeconds, //
                    byte audioCodec, byte audioScale, // 
                    String license, String profileId, //
                    int timeoutConnect, WtcInetSocketAddressPlatform localAddress) //
                    throws WtcClientUnsupportedCodecException
    {
        String sig = "connect(" + WtcString.quote(platformDescription) + ", " + kexSize + ", " + WtcUri.toString(uriServers) //
                        + ", " + WtcString.quote(username) + ", " + WtcStack.censor(password) //
                        + ", " + sessionTimeoutSeconds //
                        + ", " + audioCodec + ", " + audioScale // 
                        + ", " + WtcString.quote(license) + ", " + WtcString.quote(profileId) //
                        + ", " + timeoutConnect + ", " + localAddress + ")";

        try
        {
            WtcLog.debug(TAG, "+" + sig);

            if (listener == null)
            {
                throw new IllegalArgumentException("listener cannot be null");
            }

            if (WtcUriPlatform.isNullOrEmpty(uriServers))
            {
                throw new IllegalArgumentException("uriServer cannot be null, empty, or contain null/EMPTY values");
            }

            if (WtcString.isNullOrEmpty(username))
            {
                throw new IllegalArgumentException("username cannot be null or \"\"");
            }

            if (WtcString.isNullOrEmpty(license))
            {
                throw new IllegalArgumentException("license cannot be null or \"\"");
            }

            if (sessionTimeoutSeconds < WtcStack.TIMEOUT_SESSION_SECONDS_MIN //
                            || sessionTimeoutSeconds > WtcStack.TIMEOUT_SESSION_SECONDS_MAX)
            {
                throw new IllegalArgumentException("sessionTimeoutSeconds must be >= " + WtcStack.TIMEOUT_SESSION_SECONDS_MIN
                                + " and <= " + WtcStack.TIMEOUT_SESSION_SECONDS_MAX);
            }

            // TODO:(pv) Validate audioScale

            if (password == null)
            {
                password = "";
            }

            if (profileId == null)
            {
                profileId = "";
            }

            if (platformDescription == null)
            {
                platformDescription = PLATFORM_DESCRIPTION_DEFAULT;
            }

            synchronized (syncConnection)
            {
                // TODO:(pv) Do we want to do a hard or a soft disconnect?
                // TODO:(pv) If we are connected, maybe pass a runnable object to reconnect after disconnect to fire after the disconnect completes?
                disconnect();

                this.listener = listener;
                //setListener(listener);

                this.username = username;
                this.password = password;
                this.license = license;
                this.profileId = profileId;
                this.audioCodec = new WtcInt8(audioCodec);
                this.audioScale = new WtcInt8(audioScale);
                this.sessionTimeoutSeconds = new WtcInt8(sessionTimeoutSeconds);
                this.platformDescription = platformDescription;

                WtcMediaCodec codec = null;
                switch (audioCodec)
                {
                    case WtcpAudioCodec.PCM_Mono16bBigEndian8kHz:
                        // ignore; null codec will internally create WtcMediaCodecRawCopy.
                        // Assuming for now that mic/speaker handle only PCM.
                        // TODO:(pv) Implement detection of platform hardware PCM or AMR capabilities
                        break;

                    case WtcpAudioCodec.GSM_full_rate:
                        codec = new WtcMediaCodecGSMPlatform();
                        break;

                    default:
                        throw new WtcClientUnsupportedCodecException(audioCodec);
                }
                microphone.setMediaEncoder(codec);
                speaker.setMediaDecoder(codec);

                stack =
                    new WtcStack(microphone, speaker, connectionStatistics, version, uriServers, kexSize, timeoutConnect,
                                    localAddress);
                stack.setListener(this);
                stack.connect();
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-" + sig);
        }
    }

    public void disconnect()
    {
        //if (syncConnection != null)
        //{
        // NOTE:(pv) I am seeing a NullReferenceException here even though syncConnection is final and never null 
        synchronized (syncConnection)
        {
            boolean isSessionOpen = !WtcString.isNullOrEmpty(sessionId);
            disconnect(isSessionOpen);
        }
        //}
    }

    public void disconnect(boolean sendSessionClose)
    {
        try
        {
            WtcLog.debug(TAG, "+disconnect(sendSessionClose=" + sendSessionClose + ")");

            synchronized (syncConnection)
            {
                Integer transactionId = null;

                try
                {
                    if (sendSessionClose && stack != null)
                    {
                        // This will result in a Response, a Timeout, or some other Error in onDisconnected
                        transactionId = stack.sendSessionClose();
                    }
                }
                catch (Exception e)
                {
                    WtcLog.error(TAG, "disconnect(sendSessionClose=" + sendSessionClose + ')', e);
                    transactionId = null;
                }

                if (transactionId == null)
                {
                    // We are forcefully closing

                    // TODO:(pv) *ALWAYS* detach as a listener?!?!?!
                    // This seems like it could cause a problem during shutdown if caller expects onDisconnected event.
                    this.listener = null;
                    //setListener(null);

                    if (stack != null)
                    {
                        // TODO:(pv) Might this cause lockups when disconnecting?
                        stack.disconnect();

                        // TODO:(pv) Wait for shutdown?

                        stack.setListener(null);

                        stack = null;
                    }
                    clear();
                }
                else
                {
                    // We want to get the Response/Timeout/Error event, so don't *yet* detach as an event listener
                }
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-disconnect(sendSessionClose=" + sendSessionClose + ")");
        }
    }

    protected void onDisconnected(WtcStack stack, //
                    WtcInetSocketAddressPlatform proxyAddress, Exception exception, WtcConnectionStatistics statistics)
    {
        String sig = "onDisconnected(stack, " + WtcString.quote(proxyAddress) + ", " + exception + ", " + statistics + ")";

        boolean error = false;

        try
        {
            if (exception != null)
            {
                if (exception instanceof WtcStackSessionCloseException)
                {
                    WtcStackSessionCloseException sessionCloseException = (WtcStackSessionCloseException) exception;
                    WtcpErrorCode errorCode = sessionCloseException.errorCode;
                    error = (errorCode != null && errorCode.isError());
                }
                else
                {
                    error = true;
                }
            }

            if (error)
            {
                WtcLog.warn(TAG, "+" + sig, exception);
            }
            else
            {
                WtcLog.info(TAG, "+" + sig);
            }

            if (stack != this.stack)
            {
                WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
                return;
            }

            WtcClientListener listener = this.listener;
            if (listener != null)
            {
                listener.onDisconnected(this, proxyAddress, exception, statistics);
            }
        }
        catch (Exception e)
        {
            WtcLog.warn(TAG, "onDisconnected EXCEPTION", e);
        }
        finally
        {
            try
            {
                // "Failsafe" detach from listening to any more events from this stack 
                stack.setListener(null);

                // Internally calls this.stack.setListener(null) and clears session fields
                disconnect(false);
            }
            catch (Exception e)
            {
                WtcLog.warn(TAG, "onDisconnected EXCEPTION", e);
            }

            if (error)
            {
                WtcLog.warn(TAG, "-onDisconnected(...)");
            }
            else
            {
                WtcLog.info(TAG, "-onDisconnected(...)");
            }
        }
    }

    protected void onProxyLocating(WtcStack stack, //
                    WtcUri remoteAddress)
    {
        WtcLog.debug(TAG, "onProxyLocating(stack, " + remoteAddress + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onProxyLocating(this, remoteAddress);
        }
    }

    protected void onProxyLocated(WtcStack stack, //
                    WtcProxyInfo[] proxyInfos)
    {
        WtcLog.debug(TAG, "onProxyLocated(stack, " + WtcProxyInfo.toString(proxyInfos) + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onProxyLocated(this, proxyInfos);
        }
    }

    protected void onProxyConnecting(WtcStack stack, //
                    WtcInetSocketAddressPlatform addressProxy)
    {
        WtcLog.debug(TAG, "onProxyConnecting(stack, " + WtcString.quote(addressProxy) + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onProxyConnecting(this, addressProxy);
        }
    }

    protected void onProxyConnected(WtcStack stack, //
                    WtcProxyInfo proxyInfo, WtcInetSocketAddressPlatform proxyAddress)
    {
        WtcLog.debug(TAG, "onProxyConnected(stack, " + proxyInfo + ", " + WtcString.quote(proxyAddress) + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onProxyConnected(this, proxyInfo, proxyAddress);
        }
    }

    protected void onMessageReceivedMedia(WtcStack stack, //
                    WtcpMediaHeader mediaHeader, int payloadLength)
    {
        // NOTE:(pv) This method will be called *very* frequently; logging here will overload the system.
        // It is OK to log this while trying to get the code to function, 
        // but production code should have anything heavy (ie: logging) commented out. 
        //WtcLog.debug(TAG, "onMessageReceivedMedia(stack, " + mediaHeader + ", " + payloadLength + ")");
        //if (stack != this.stack)
        //{
        //WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
        //return;
        //}
        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onMessageReceivedMedia(this, mediaHeader, payloadLength);
        }
    }

    protected void onProxySecuring(WtcStack stack, //
                    int kexSize)
    {
        WtcLog.debug(TAG, "onProxySecuring(stack, " + kexSize + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onProxySecuring(this, kexSize);
        }
    }

    protected synchronized void onProxySecured(WtcStack stack, //
                    int kexSize)
    {
        WtcLog.debug(TAG, "onProxySecured(stack, " + kexSize + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onProxySecured(this, kexSize);
        }

        synchronized (syncConnection)
        {
            // This is where the *Client* state machine primes its initial hook in to the session...
            stack.sendSessionOpen(audioCodec, audioScale, sessionTimeoutSeconds, platformDescription);
        }
    }

    protected void onSessionOpen(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    String sessionId, WtcInt32 serverTime, WtcVersionString serverVersion)
    {
        String sig =
            "onSessionOpen(stack, " + controlHeader + ", " + WtcString.quote(sessionId) + ", " + serverTime + ", "
                            + serverVersion + ")";
        try
        {
            WtcLog.debug(TAG, "+" + sig);
            if (stack != this.stack)
            {
                WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
                return;
            }

            synchronized (syncConnection)
            {
                this.sessionId = sessionId;
            }

            WtcClientListener listener = this.listener;
            if (listener != null)
            {
                listener.onSessionOpen(this, controlHeader.getOpType(), controlHeader.transactionId, sessionId, serverTime,
                                serverVersion);
            }

            setProfile(stack, profileId);
        }
        finally
        {
            WtcLog.debug(TAG, "-" + sig);
        }
    }

    protected void onSessionOpen(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onSessionOpen(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onSessionOpenError(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode);
        }
    }

    /**
     * Should only be called during a session logon sequence.
     * @param profileId if null or "" then requests a list of profile ids, otherwise sets the profile id.
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer setProfile(String profileId)
    {
        return setProfile(this.stack, profileId);
    }

    /**
     * @param stack
     * @param profileId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    private Integer setProfile(WtcStack stack, String profileId)
    {
        synchronized (syncConnection)
        {
            if (stack != null)
            {
                this.profileId = profileId;

                return stack.sendSetCredentials(username, password, license, profileId);
            }
        }
        return null;
    }

    protected void onSetCredentials(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    String localEndpointId, //
                    byte profileIndex, WtcpProfileInfoList profiles, //
                    WtcpChannelInfoList channels, //
                    WtcpStringList phoneLines)
    {
        String sig =
            "onSetCredentials(stack, " + controlHeader + ", " + WtcString.quote(localEndpointId) + ", " + profileIndex + ", "
                            + profiles + ", " + channels + ", " + phoneLines + ")";
        try
        {
            WtcLog.debug(TAG, "+" + sig);
            if (stack != this.stack)
            {
                WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
                return;
            }

            if (profileIndex != -1)
            {
                synchronized (syncConnection)
                {
                    channelManager.setChannels(channels);
                    phoneLineManager = new WtcClientPhoneLineManager(this, listener, stack, phoneLines);
                    phoneCallManager = new WtcClientPhoneCallManager(this, listener);
                }
            }

            WtcClientListener listener = this.listener;
            if (listener != null)
            {
                WtcLog.info(TAG, "onSetCredentials(...): Sending channels to client listener: " + channels);
                listener.onSetCredentials(this, controlHeader.getOpType(), controlHeader.transactionId, localEndpointId,
                                profileIndex, profiles, channels, phoneLines);
            }
        }
        finally
        {
            WtcLog.debug(TAG, "-" + sig);
        }
    }

    protected void onSetCredentials(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode, WtcpProfileInfoList profiles)
    {
        WtcLog.debug(TAG, "onSetCredentials(stack, " + controlHeader + ", " + errorCode + ", " + profiles + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onSetCredentialsError(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode, profiles);
        }
    }

    protected void onSessionResume(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode, String sessionId)
    {
        WtcLog.debug(TAG, "onSessionResume(stack, " + controlHeader + ", " + errorCode + ", " + WtcString.quote(sessionId)
                        + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        synchronized (syncConnection)
        {
            this.sessionId = sessionId;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onSessionResume(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode, sessionId);
        }
    }

    protected void onSessionClose(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onSessionClose(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        synchronized (syncConnection)
        {
            this.sessionId = null;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onSessionClose(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode);
        }
    }

    protected void OnChannelChange(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    boolean reconnect, int change, int channelId)
    {
        WtcLog.debug(TAG, "+onChannelSetActive(stack, " + controlHeader + ", reconnect=" + reconnect + ", change="
                        + WtcpChannelChange.toString(change) + ", channelId=" + channelId + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelChange(this, listener, controlHeader, reconnect, change, channelId);
    }

    /**
     * Called by WtcClientChannelManager.WtcClientChannel.activate(...).
     * Returns a boolean because the server can close the session if the user takes too long to initially set their active channel(s).
     * @param channels
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer channelsSetActive(WtcpChannelIdList channels)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendChannelSetActive(channels);
    }

    protected void onChannelSetActive(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpChannelIdErrorDictionary channelIdErrors)
    {
        WtcLog.debug(TAG, "+onChannelSetActive(stack, " + controlHeader + ", " + channelIdErrors + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelOperation(this, listener, controlHeader, channelIdErrors);
    }

    /**
     * This method serves no real purpose; it is only implemented here for consistency/thoroughness.
     * @param channelId
     * @return transactionId i the Request was successfully placed in the queue, otherwise null
     */
    protected Integer channelGetActivity(int channelId)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendChannelActivityRequest(WtcInt32.valueOf(channelId, true));
    }

    protected void onChannelActivity(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpChannelActivity channelActivity)
    {
        WtcLog.debug(TAG, "onChannelActivity(stack, " + controlHeader + ", " + channelActivity + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelActivity(this, listener, controlHeader, channelActivity);
    }

    /**
     * This event serves no real purpose; it is only implemented here for consistency/thoroughness.
     */
    protected void onChannelActivity(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onChannelActivity(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelActivity(this, listener, controlHeader, errorCode);
    }

    /**
     * Called by WtcClientChannelManager.WtcClientChannel.talk(...)
     * @param channels
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer channelsPushToTalk(WtcpChannelIdList channels)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendChannelPushToTalk(channels);
    }

    protected void onChannelPushToTalk(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpChannelIdErrorDictionary channelIdErrors)
    {
        WtcLog.debug(TAG, "onChannelPushToTalk(stack, " + controlHeader + ", " + channelIdErrors + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelOperation(this, listener, controlHeader, channelIdErrors);
    }

    /**
     * Called by WtcClientChannelManager.WtcClientChannel.mute(...)
     * @param channels
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer channelsMute(WtcpChannelIdList channels)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendChannelMute(channels);
    }

    protected void onChannelMute(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpChannelIdErrorDictionary channelIdErrors)
    {
        WtcLog.debug(TAG, "onChannelMute(stack, " + controlHeader + ", " + channelIdErrors + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelOperation(this, listener, controlHeader, channelIdErrors);
    }

    /**
     * @param channelId
     * @param keys
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    protected Integer channelPropertiesGet(WtcInt32 channelId, WtcpStringList keys)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendChannelPropertiesGet(channelId, keys);
    }

    protected void onChannelPropertiesGet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    int channelId, WtcpKeyValueList keyValues)
    {
        WtcLog.debug(TAG, "onChannelPropertiesGet(stack, " + controlHeader + ", " + channelId + ", " + keyValues + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelPropertiesGet(this, listener, controlHeader, channelId, keyValues);
    }

    protected void onChannelPropertiesGet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    int channelId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onChannelPropertiesGet(stack, " + controlHeader + ", " + channelId + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelPropertiesGet(this, listener, controlHeader, channelId, errorCode);
    }

    /**
     * @param channelId
     * @param keyValues
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer channelPropertiesSet(WtcInt32 channelId, WtcpKeyValueList keyValues)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendChannelPropertiesSet(channelId, keyValues);
    }

    protected void onChannelPropertiesSet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    int channelId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onChannelPropertiesSet(stack, " + controlHeader + ", " + channelId + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        channelManager.onChannelPropertiesSet(this, listener, controlHeader, channelId, errorCode);
    }

    /**
     * @param keyValues
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointPropertiesSet(WtcpKeyValueList keyValues)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendEndpointPropertiesSet(keyValues);
    }

    protected void onEndpointPropertiesSet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onEndpointPropertiesSet(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onEndpointPropertiesSet(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode);
        }
    }

    /**
     * @param channelId
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupChannel(int channelId, String searchString)
    {
        return endpointLookupChannel(channelId, WtcpEndpointFlags.Visible, searchString);
    }

    /**
     * @param channelId
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupChannel(int channelId, int flagsInclude, String searchString)
    {
        return endpointLookupChannel(channelId, flagsInclude, WtcpEndpointFlags.None, searchString);
    }

    /**
     * @param channelId
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param flagsExclude Any combination of WtcpEndpointFlags
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupChannel(int channelId, int flagsInclude, int flagsExclude, String searchString)
    {
        return endpointLookupChannel(channelId, flagsInclude, flagsExclude, (byte) 0, (short) 0, searchString);
    }

    /**
     * @param channelId
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param flagsExclude Any combination of WtcpEndpointFlags
     * @param pageSize
     * @param pageNumber
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupChannel(int channelId, int flagsInclude, int flagsExclude, //
                    byte pageSize, short pageNumber, String searchString)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendEndpointLookup(WtcInt32.valueOf(channelId, true), //
                        WtcInt32.valueOf(flagsInclude, false), WtcInt32.valueOf(flagsExclude, false), //
                        new WtcInt8(pageSize), new WtcInt16(pageNumber), searchString);
    }

    /**
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupSystem(String searchString)
    {
        return endpointLookupChannel(WtcStack.ID_CHANNEL_SPC, searchString);
    }

    /**
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupSystem(int flagsInclude, String searchString)
    {
        return endpointLookupChannel(WtcStack.ID_CHANNEL_SPC, flagsInclude, searchString);
    }

    /**
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param flagsExclude Any combination of WtcpEndpointFlags
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupSystem(int flagsInclude, int flagsExclude, String searchString)
    {
        return endpointLookupChannel(WtcStack.ID_CHANNEL_SPC, flagsInclude, flagsExclude, searchString);
    }

    /**
     * @param flagsInclude Any combination of WtcpEndpointFlags
     * @param flagsExclude Any combination of WtcpEndpointFlags
     * @param pageSize
     * @param pageNumber
     * @param searchString
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointLookupSystem(int flagsInclude, int flagsExclude, byte pageSize, short pageNumber, String searchString)
    {
        return endpointLookupChannel(WtcStack.ID_CHANNEL_SPC, flagsInclude, flagsExclude, pageSize, pageNumber, searchString);
    }

    protected void onEndpointLookup(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    int channelId, short pageNumber, short numberOfPages, WtcpEndpointInfoList endpoints)
    {
        WtcLog.debug(TAG, "onEndpointLookup(stack, " + controlHeader + ", " + channelId + ", " + pageNumber + ", "
                        + numberOfPages + ", " + endpoints + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onEndpointLookup(this, controlHeader.getOpType(), controlHeader.transactionId, channelId, pageNumber,
                            numberOfPages, endpoints);
        }
    }

    protected void onEndpointLookup(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    int channelId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onEndpointLookup(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onEndpointLookupError(this, controlHeader.getOpType(), controlHeader.transactionId, channelId, errorCode);
        }
    }

    /**
     * @param endpointId
     * @param keys
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointPropertiesGet(String endpointId, String[] keys)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendEndpointPropertiesGet(endpointId, keys);
    }

    /**
     * @param endpointId
     * @param keys
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointPropertiesGet(String endpointId, WtcpStringList keys)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendEndpointPropertiesGet(endpointId, keys);
    }

    protected void onEndpointPropertiesGet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpEndpointProperties keyValues)
    {
        WtcLog.debug(TAG, "onEndpointPropertiesGet(stack, " + controlHeader + ", " + keyValues + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onEndpointPropertiesGet(this, controlHeader.getOpType(), controlHeader.transactionId, keyValues);
        }
    }

    protected void onEndpointPropertiesGet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onEndpointPropertiesGet(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onEndpointPropertiesGetError(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode);
        }
    }

    /**
     * @param filter
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer endpointPropertyFilterSet(String filter)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendEndpointPropertyFilterSet(WtcpEndpointFilterType.SemiColonSeparated, filter);
    }

    protected void onEndpointPropertyFilterSet(WtcStack stack, //
                    WtcpControlHeader controlHeader, //
                    WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onEndpointPropertyFilterSet(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onEndpointPropertyFilterSet(this, controlHeader.getOpType(), controlHeader.transactionId, errorCode);
        }
    }

    /**
     * @param stack
     * @param timeoutMs timeout requested
     * @param elapsedMs timeout actual
     * @param lastPingRequestTxId the previously Transmitted Ping Request PingId
     * @return the latest Ping Request transactionId
     */
    //@Override
    public short onPingRequestRxTimeout(WtcStack stack, long timeoutMs, long elapsedMs, short lastPingRequestTxId)
    {
        WtcLog.debug(TAG, "onPingRequestRxTimeout(stack, timeoutMs=" + timeoutMs + ", elapsedMs=" + elapsedMs
                        + ", lastPingRequestTxId=" + lastPingRequestTxId + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return -1;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            lastPingRequestTxId = listener.onPingRequestRxTimeout(this, timeoutMs, elapsedMs, lastPingRequestTxId);
        }

        return lastPingRequestTxId;
    }

    protected void onPing(WtcStack stack, WtcpControlHeader controlHeader, short pingId)
    {
        WtcLog.debug(TAG, "onPing(stack, " + controlHeader + ", " + pingId + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onPing(this, controlHeader.getOpType(), controlHeader.transactionId, pingId);
        }
    }

    //
    // Phone Lines
    //

    /**
     * NOTE: Currently only supports activating the <b>FIRST</b> phone line!
     * @param phoneLines
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer phoneLinesSetActive(String[] phoneLines)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneLineManager.activate(phoneLines);
    }

    /**
     * NOTE: Currently only supports activating the <b>FIRST</b> phone line!
     * @param phoneLines
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer phoneLinesSetActive(WtcpStringList phoneLines)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneLineManager.activate(phoneLines);
    }

    protected void onPhoneLinesSetActive(WtcStack stack, WtcpControlHeader controlHeader, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onPhoneLinesSetActive(stack, " + controlHeader + ", errorCode=" + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        if (phoneLineManager != null)
        {
            phoneLineManager.onPhoneLinesSetActive(controlHeader, errorCode);
        }
    }

    protected void onPhoneLineStatus(WtcStack stack, WtcpControlHeader controlHeader, String phoneLine, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onPhoneLineStatus(stack, " + controlHeader + ", " + WtcString.quote(phoneLine) + ", " + errorCode
                        + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        if (phoneLineManager != null)
        {
            phoneLineManager.onPhoneLineStatus(controlHeader, phoneLine, errorCode);
        }
    }

    //
    // Calls
    //

    /**
     * @deprecated Use {@link WtcClientPhoneLine#callEndpoint} or {@link WtcClientPhoneLine#callNumber}
     * @param callType One of WtcpCallType.*
     * @param source
     * @param destination
     * @param destinationName
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    //@Deprecated
    public Integer callMake(byte callType, String source, String destination, String destinationName)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneCallManager.call(callType, source, destination, destinationName);
    }

    protected void onCallMake(WtcStack stack, WtcpControlHeader controlHeader, WtcpCallInfo callInfo)
    {
        WtcLog.debug(TAG, "onCallMake(stack, " + controlHeader + ", " + callInfo + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallMake(controlHeader, callInfo);
        }
    }

    protected void onCallProgress(WtcStack stack, WtcpControlHeader controlHeader, WtcpCallProgress callProgress)
    {
        WtcLog.debug(TAG, "onCallProgress(stack, " + controlHeader + ", " + callProgress + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallProgress(controlHeader, callProgress);
        }
    }

    protected void onCallOffer(WtcStack stack, WtcpControlHeader controlHeader, WtcpCallOffer callOffer)
    {
        WtcLog.debug(TAG, "onCallOffer(stack, " + controlHeader + ", " + callOffer + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallOffer(controlHeader, callOffer);
        }
    }

    /**
     * @deprecated Use {@link WtcClientPhoneCall#answer}
     * @param callId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    //@Deprecated
    public Integer callAnswer(int callId)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneCallManager.answer(callId);
    }

    protected void onCallAnswer(WtcStack stack, WtcpControlHeader controlHeader, WtcpCallAnswer callAnswer)
    {
        WtcLog.debug(TAG, "onCallAnswer(stack, " + controlHeader + ", " + callAnswer + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallAnswer(controlHeader, callAnswer);
        }
    }

    /**
     * @deprecated Use {@link WtcClientPhoneCall#hangup}
     * @param callId
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    //@Deprecated
    public Integer callHangup(int callId)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneCallManager.hangup(callId);
    }

    protected void onCallHangup(WtcStack stack, WtcpControlHeader controlHeader, WtcpCallHangup callHangup)
    {
        WtcLog.debug(TAG, "onCallHangup(stack, " + controlHeader + ", " + callHangup + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallHangup(controlHeader, callHangup);
        }
    }

    /**
     * @deprecated Use {@link WtcClientPhoneCall#dtmf}
     * @param callId
     * @param digits
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    //@Deprecated
    public Integer callDtmf(int callId, String digits)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneCallManager.dtmf(callId, digits);
    }

    /**
     * Response success to callDtmf
     */
    protected void onCallDtmf(WtcStack stack, WtcpControlHeader controlHeader)
    {
        WtcLog.debug(TAG, "onCallDtmf(stack, " + controlHeader + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallDtmf(controlHeader, (WtcpErrorCode) null);
        }
    }

    /**
     * Error response to callDtmf
     */
    protected void onCallDtmf(WtcStack stack, WtcpControlHeader controlHeader, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onCallDtmf(stack, " + controlHeader + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallDtmf(controlHeader, errorCode);
        }
    }

    protected void onCallDtmf(WtcStack stack, WtcpControlHeader controlHeader, WtcpCallDtmf callDtmf)
    {
        WtcLog.debug(TAG, "onCallDtmf(stack, " + controlHeader + ", " + callDtmf + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallDtmf(controlHeader, callDtmf);
        }
    }

    /**
     * @deprecated Use {@link WtcClientPhoneCall#pushToTalk}
     * @param callId
     * @param on
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    //@Deprecated
    public Integer callPushToTalk(int callId, boolean on)
    {
        WtcClientPhoneLineManager phoneLineManager = this.phoneLineManager;
        return (phoneLineManager == null) ? null : phoneCallManager.pushToTalk(callId, on);
    }

    protected void onCallPushToTalkOn(WtcStack stack, WtcpControlHeader controlHeader, //
                    WtcInt32 callId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onCallPushToTalkOn(stack, " + controlHeader + ", " + callId + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallPushToTalkOn(controlHeader, callId, errorCode);
        }
    }

    protected void onCallPushToTalkOff(WtcStack stack, WtcpControlHeader controlHeader, //
                    WtcInt32 callId, WtcpErrorCode errorCode)
    {
        WtcLog.debug(TAG, "onCallPushToTalkOff(stack, " + controlHeader + ", " + callId + ", " + errorCode + ")");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientPhoneCallManager phoneCallManager = this.phoneCallManager;
        if (phoneCallManager != null)
        {
            phoneCallManager.onCallPushToTalkOff(controlHeader, callId, errorCode);
        }
    }

    //
    // Address Book
    //

    /**
     * @param version
     * @param filter
     * @return transactionId if the Request was successfully placed in the queue, otherwise null
     */
    public Integer getAddressBook(short version, String filter)
    {
        WtcStack stack = this.stack;
        return (stack == null) ? null : stack.sendAddressBookRequest(version, filter);
    }

    protected void onAddressBook(WtcStack stack, WtcpControlHeader controlHeader, //
                    WtcInt16 version, WtcpErrorCode errorCode, WtcpAddressBookInfoList addressBookInfoList)
    {
        WtcLog.debug(TAG, "onAddressBook(stack, " + controlHeader + ", " + version + ", " + errorCode
                        + ", addressBookInfoList(" + addressBookInfoList.size() + ")=...)");
        if (stack != this.stack)
        {
            WtcLog.warn(TAG, "Got event for non-current stack; ignoring");
            return;
        }

        WtcClientListener listener = this.listener;
        if (listener != null)
        {
            listener.onAddressBook(this, WtcpOpType.Response, -1, version, errorCode, addressBookInfoList);
        }
    }
}
