package com.twilio.voice;

import android.os.Handler;

class Registrar implements RegistrarListener {
    static final String RELATIVE_REGISTER_URL = "/v1/registrations?productId=voice";
    private static final String RELATIVE_UNREGISTER_URL = "?productId=voice";

    private final Handler handler;
    private final String accessToken;
    private final String channelType;
    private final String notificationToken;
    private String homeRegion;
    String urlString;

    private RegistrationListener registrationListener;
    private UnregistrationListener unregistrationListener;

    public Registrar(String accessToken, String channelType, String notificationToken) {
        this.accessToken = accessToken;
        this.channelType = channelType;
        this.notificationToken = notificationToken;
        this.handler = Utils.createHandler();
        try {
            this.homeRegion = new AccessTokenParser(accessToken).getHomeRegion();
        } catch (AccessTokenParseException e) {
            e.printStackTrace();
        }
        if (this.homeRegion == null) {
            urlString = Constants.getNotificationServiceUrl() + RELATIVE_REGISTER_URL;
        } else {
            urlString =
                    String.format("https://ers.%s.twilio.com" + RELATIVE_REGISTER_URL, homeRegion);
        }
    }

    public void register(RegistrationListener listener) {
        this.registrationListener = listener;
        String jsonPayloadString =
                new Registration(channelType, notificationToken).toJson().toString();
        HttpsRegistrar.register(accessToken, jsonPayloadString, urlString, this);
    }

    /*
     * RTD's registration/unregistration flow was designed for connected architecture, i.e. using
     * TwilioSock. To be able to unregister a binding RTD expects to know the registration server                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          Voice SDK is stateless
     * location. Voice SDK being a stateless SDK , it does not cache the registration location.
     * To make unregistration work, Voice SDK registers first. This overwrites the original
     * registration binding and returns the registration location that the SDK can use to unregister.
     */
    public void unregister(UnregistrationListener listener) {
        this.unregistrationListener = listener;
        String jsonPayloadString =
                new Registration(channelType, notificationToken).toJson().toString();

        boolean isUnregistrationRequest = true;
        HttpsRegistrar.register(
                accessToken,
                jsonPayloadString,
                urlString,
                isUnregistrationRequest,
                new RegistrarListener() {
                    @Override
                    public void onSuccess(String registrationLocation) {
                        String urlString = registrationLocation + RELATIVE_UNREGISTER_URL;
                        HttpsRegistrar.unregister(accessToken, urlString, Registrar.this);
                    }

                    @Override
                    public void onError(RegistrationException registrationException) {
                        Registrar.this.onError(registrationException);
                    }
                });
    }

    @Override
    public void onSuccess(String registrationLocation) {
        handler.post(
                new Runnable() {
                    @Override
                    public void run() {
                        if (registrationListener != null) {
                            registrationListener.onRegistered(accessToken, notificationToken);
                        } else {
                            unregistrationListener.onUnregistered(accessToken, notificationToken);
                        }
                    }
                });
    }

    @Override
    public void onError(final RegistrationException registrationException) {
        handler.post(
                new Runnable() {
                    @Override
                    public void run() {
                        if (registrationListener != null) {
                            registrationListener.onError(
                                    registrationException, accessToken, notificationToken);
                        } else {
                            unregistrationListener.onError(
                                    registrationException, accessToken, notificationToken);
                        }
                    }
                });
    }
}
