package com.twilio.voice;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

/** Represents parameters related to a log message. */
public class LogParameters {
    /** {@link LogModule} that generated this log message */
    @NonNull public LogModule module;

    /** {@link LogLevel} at which this log message was generated */
    @NonNull public LogLevel level;

    /** Local Timestamp in YY-MM-DD HH:MM:SS.MMM format as a string */
    @NonNull public String timestamp;

    /** Source file that generated this log message */
    @NonNull public String file;

    /** Method within which this log message was generated */
    @NonNull public String function;

    /** Line number of the log message within the source file */
    public int line;

    /** A customized string metadata that is defined by the log message generator */
    @NonNull public String tag;

    /** Log message text */
    @NonNull public String message;

    /** Thread name (or id) */
    @NonNull public String thread;

    /** A throwable object for logging stack traces. May be null. */
    @Nullable public Throwable tr;

    /**
     * Construct {@link LogParameters} via {@link LogParameters.Builder}
     *
     * @param builder {@link LogParameters.Builder} object
     */
    private LogParameters(@NonNull Builder builder) {
        Preconditions.checkNotNull(builder, "Builder must not be null");
        this.module = builder.module;
        this.level = builder.level;
        this.timestamp = builder.timestamp;
        this.file = builder.file;
        this.function = builder.function;
        this.line = builder.line;
        this.tag = builder.tag;
        this.message = builder.message;
        this.thread = builder.thread;
        this.tr = builder.tr;
    }

    public static class Builder {
        @NonNull private LogModule module = LogModule.PLATFORM;

        @NonNull private LogLevel level;

        @NonNull private String timestamp = getTimestamp();

        @NonNull public String file = "";

        @NonNull private String function = "";

        private int line = 0;

        @NonNull private String tag = "";

        @NonNull private String message = "";

        @NonNull private String thread = getThreadNameOrId();

        @Nullable private Throwable tr;

        /**
         * Use this Builder object to create a {@link LogParameters} object. Module defaults to
         * {@link LogModule#PLATFORM}. File and function parameters default to empty strings and
         * line number defaults to zero. The current time is used as the timestamp parameter and the
         * current thread's name is used as the thread parameter.
         *
         * @param level {@link LogLevel} at which this log message was generated. Must not be null.
         * @param tag String tag metadata. Must not be null.
         * @param message Log message text. Must not be null.
         */
        public Builder(@NonNull LogLevel level, @NonNull String tag, @NonNull String message) {
            Preconditions.checkNotNull(level, "Level must not be null");
            Preconditions.checkNotNull(tag, "Tag must not be null");
            Preconditions.checkNotNull(message, "Message must not be null");
            this.level = level;
            this.tag = tag;
            this.message = message;
        }

        /**
         * Set the {@link LogModule} that generated this log message. If not set, {@link
         * LogModule#PLATFORM} is used as the default.
         *
         * @param module {@link LogModule}. Must not be null.
         * @return {@link Builder}
         */
        @NonNull
        public Builder module(@NonNull LogModule module) {
            Preconditions.checkNotNull(module, "Module must not be null");
            this.module = module;
            return this;
        }

        /**
         * Set the local timestamp. If not set, the current timestamp is used as the default.
         *
         * @param timestamp in YY-MM-DD HH:MM:SS.MMM format as a string. Must not be null.
         * @return {@link Builder}
         */
        @NonNull
        public Builder timestamp(@NonNull String timestamp) {
            Preconditions.checkNotNull(timestamp, "Timestamp must not be null");
            this.timestamp = timestamp;
            return this;
        }

        /**
         * Set the source filename that generated this log message. If not set, empty string is set
         * as the default.
         *
         * @param file Source filename. Must not be null.
         * @return {@link Builder}
         */
        @NonNull
        public Builder file(@NonNull String file) {
            Preconditions.checkNotNull(file, "Filename must not be null");
            this.file = file;
            return this;
        }

        /**
         * Set the method name within which this log message was generated. If not set, empty string
         * is set as the default.
         *
         * @param function Method name. Must not be null.
         * @return {@link Builder}
         */
        @NonNull
        public Builder function(@NonNull String function) {
            Preconditions.checkNotNull(function, "Method name must not be null");
            this.function = function;
            return this;
        }

        /**
         * Set the line number of the log message in the source file. If not set, zero is used as
         * the default.
         *
         * @param line Line number
         * @return {@link Builder}
         */
        public Builder line(int line) {
            this.line = line;
            return this;
        }

        /**
         * Set the thread name or thread id where the log message is generated. If not set, current
         * thread name or id is used as the default.
         *
         * @param thread Thread name or id. Must not be null.
         * @return {@link Builder}
         */
        @NonNull
        public Builder thread(String thread) {
            Preconditions.checkNotNull(thread, "Thread name or id must not be null");
            this.thread = thread;
            return this;
        }

        /**
         * Set a throwable object for logging stack traces. If not set, default is null.
         *
         * @param tr A {@link Throwable} object. May be null.
         * @return {@link Builder}
         */
        @Nullable
        public Builder throwable(Throwable tr) {
            this.tr = tr;
            return this;
        }

        /**
         * Build the {@link LogParameters} object
         *
         * @return {@link LogParameters}
         */
        @NonNull
        public LogParameters build() {
            return new LogParameters(this);
        }
    }

    /**
     * DEPRECATED - Construct the {@link LogParameters} object using {@link LogParameters.Builder}
     * Construtct a {@link LogParameters} object.
     *
     * @param module {@link LogModule} that generated this log message. Must not be null.
     * @param level {@link LogLevel} at which this log message was generated. Must not be null.
     * @param timestamp Local Timestamp in YY-MM-DD HH:MM:SS.MMM format as a string. Must not be
     *     null. When empty, the SDK will set the current time as the timestamp.
     * @param file Source file that generated this log message. Must not be null.
     * @param function Method within which this log message was generated. Must not be null.
     * @param line Line number of the log message within the source file. Must not be null.
     * @param tag Metadata string. Must not be null.
     * @param message Log message text. Must not be null.
     * @param thread Thread name or id. Must not be null.
     */
    public LogParameters(
            @NonNull LogModule module,
            @NonNull LogLevel level,
            @NonNull String timestamp,
            @NonNull String file,
            @NonNull String function,
            int line,
            @NonNull String tag,
            @NonNull String message,
            @NonNull String thread) {
        Preconditions.checkNotNull(module, "Module must not be null");
        Preconditions.checkNotNull(level, "Level must not be null");
        Preconditions.checkNotNull(timestamp, "Timestamp must not be null");
        Preconditions.checkNotNull(file, "Filename must not be null");
        Preconditions.checkNotNull(function, "Method name must not be null");
        Preconditions.checkNotNull(tag, "Tag must not be null");
        Preconditions.checkNotNull(message, "Message must not be null");
        Preconditions.checkNotNull(thread, "Thread name or id must not be null");
        this.module = module;
        this.level = level;
        this.timestamp = timestamp.isEmpty() ? getTimestamp() : timestamp;
        this.file = file;
        this.function = function;
        this.line = line;
        this.tag = tag;
        this.message = message;
        this.thread = thread;
    }

    /**
     * DEPRECATED - Construct the {@link LogParameters} object using {@link LogParameters.Builder}
     * Construct a {@link LogParameters} object. This constructor requires the level, tag, and
     * message arguments. Module defaults to {@link LogModule#PLATFORM}. File and function
     * parameters default to empty strings and line number defaults to zero. The current time is
     * used as the timestamp parameter and the current thread's name is used as the thread
     * parameter.
     *
     * @param level {@link LogLevel} at which this log message was generated. Must not be null.
     * @param tag String tag metadata. Must not be null.
     * @param message Log message text. Must not be null.
     */
    public LogParameters(@NonNull LogLevel level, @NonNull String tag, @NonNull String message) {
        this(
                LogModule.PLATFORM,
                level,
                getTimestamp(),
                "",
                "",
                0,
                tag,
                message,
                getThreadNameOrId());
    }

    private static String getTimestamp() {
        SimpleDateFormat simpleformat = new SimpleDateFormat("yy-MM-dd hh:mm:ss.SSS", Locale.US);
        return simpleformat.format(Calendar.getInstance().getTime()).toString();
    }

    private static String getThreadNameOrId() {
        if (!Thread.currentThread().getName().isEmpty()) {
            return Thread.currentThread().getName();
        } else {
            return Long.toString(Thread.currentThread().getId());
        }
    }
}
