package com.twilio.voice;

import androidx.annotation.NonNull;

/**
 * Objects that represent messages that can be sent and received via {@link
 * Call#sendMessage(CallMessage)} and {@link Call.CallMessageListener} respectively. The size limit
 * of the message content is 10 KB. If a messages is attempted to be sent via {@link
 * Call#sendMessage(CallMessage)} with message content that exceeds the size limitation, the {@link
 * Call.CallMessageListener#onMessageFailure(String, String, VoiceException)} callback will be
 * triggered and no error will be logged in the Twilio developer console.
 */
public class CallMessage {

    /** Returns the message type string. */
    @NonNull
    public String getMessageType() {
        return messageType;
    }

    /**
     * Returns the MIME type for the message. Currently the only supported type is
     * "application/json", which is also the default value.
     */
    @NonNull
    public String getMessageContentType() {
        return contentType;
    }

    /**
     * Returns he content body of the message. The format of the message body must match the content
     * type. A {@link CallMessage} with content that does not match the content type will not result
     * in the {@link Call.CallMessageListener#onMessageFailure(String, String, VoiceException)}
     * callback being invoked and will instead generate an error in your Twilio developer console.
     * For example, a call message with content type "application/json" but with the content "Hello
     * World", which is not a valid JSON object, will result in such error.
     */
    @NonNull
    public String getContent() {
        return content;
    }

    /** Returns the unique identifier of the message. */
    @NonNull
    public String getVoiceEventSID() {
        return voiceEventSid;
    }

    /** Builds new {@link CallMessage}. */
    public static class Builder {
        /**
         * @param messageType The message type string. Should be a Twilio Programmable Voice
         *     supported message type found here:
         *     https://www.twilio.com/docs/voice/sdks/call-message-events
         */
        public Builder(@NonNull String messageType) {
            Preconditions.checkNotNull(messageType, "messageType must not be null");
            Preconditions.checkArgument(messageType.length() > 0, "messageType must not be empty");
            this.messageType = messageType;
            this.contentType = "application/json";
            this.content = "";
        }

        /**
         * Sets the message content MIME type. Currently the only supported type is
         * "application/json", which is also the default value.
         */
        public Builder contentType(@NonNull String contentType) {
            Preconditions.checkNotNull(contentType, "contentType must not be null");
            this.contentType = contentType;
            return this;
        }

        /** Sets the message content. Message content must be less than 10k in size. */
        public Builder content(@NonNull String content) {
            Preconditions.checkNotNull(content, "content must not be null");
            this.content = content;
            return this;
        }

        public CallMessage build() {
            return new CallMessage(messageType, contentType, content, "");
        }

        private String messageType;
        private String contentType;
        private String content;
    }

    private CallMessage(
            String messageType, String contentType, String content, String voiceEventSid) {
        this.messageType = messageType;
        this.contentType = contentType;
        this.content = content;
        this.voiceEventSid = voiceEventSid;
    }

    private final String messageType;
    private String voiceEventSid;
    private final String contentType;
    private final String content;
}
