package com.twilio.voice;

import android.util.Log;
import androidx.annotation.NonNull;

class DefaultLogger implements LoggerInterface {
    @Override
    public void log(@NonNull final LogParameters parameters) {
        Preconditions.checkNotNull(parameters, "LogParameters object must not be null");
        String message_prefix = "[" + convertModuleToString(parameters.module) + "] ";

        if (BuildConfig.DEBUG) {
            if (!parameters.thread.isEmpty()) {
                message_prefix += "(thread " + parameters.thread + ") ";
            }
            if (!parameters.file.isEmpty()) {
                // Remove the path before the filename
                int index = parameters.file.lastIndexOf('/');
                message_prefix += parameters.file.substring(index + 1) + " ";
                if (parameters.line > 0) {
                    message_prefix += "(line " + String.valueOf(parameters.line) + ") ";
                }
            }
            if (!parameters.function.isEmpty()) {
                message_prefix += parameters.function + ": ";
            }
        }

        String tag = parameters.tag.isEmpty() ? "Twilio" : "Twilio:" + parameters.tag;
        String message = message_prefix + parameters.message;
        if (parameters.tr != null) {
            message += "\n" + Log.getStackTraceString(parameters.tr);
        }

        switch (parameters.level) {
            case FATAL:
            case ERROR:
                Log.e(tag, message);
                break;
            case WARNING:
                Log.w(tag, message);
                break;
            case INFO:
                Log.i(tag, message);
                break;
            case DEBUG:
                Log.d(tag, message);
                break;
            case TRACE:
            case ALL:
                Log.v(tag, message);
                break;
            case OFF:
            default:
                break;
        }
    }

    private String convertModuleToString(LogModule module) {
        String mod = new String();
        switch (module) {
            case CORE:
                mod = "Core";
                break;
            case SIGNALING:
                mod = "Signaling";
                break;
            case WEBRTC:
                mod = "WebRTC";
                break;
            case PLATFORM:
                mod = "Platform";
                break;
            default:
                break;
        }
        return mod;
    }
}
