package com.twilio.voice;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * DefaultAudioDevice is the default audio device module SDK uses when a custom one is not provided.
 *
 * <p>By default, the Voice SDK uses `DefaultAudioDevice` to capture and render audio. The SDK
 * performs acoustic echo cancellation (AEC) and noise suppression (NS) using device hardware by
 * default. Using device hardware is more efficient, but some devices do not implement these audio
 * effects well. `DefaultAudioDevice` provides API `setUseHardwareAcousticEchoCanceler(...)` or
 * `setUseHardwareNoiseSuppressor(...)` to toggle the default behavior. If you are experiencing echo
 * or background noise on certain devices reference the following snippet for enabling software
 * implementations of AEC and NS anytime before invoking `Voice.connect(...)` or
 * `CallInvite.accept(...)`.
 *
 * <pre>
 * // Use software AEC
 * DefaultAudioDevice defaultAudioDevice=new DefaultAudioDevice();
 * defaultAudioDevice.setUseHardwareAcousticEchoCanceler(false);
 * Voice.setAudioDevice(defaultAudioDevice);
 *
 * // Use software NS
 * DefaultAudioDevice defaultAudioDevice=new DefaultAudioDevice();
 * defaultAudioDevice.setUseHardwareNoiseSuppressor(false);
 * Voice.setAudioDevice(defaultAudioDevice);
 *
 * </pre>
 */
public class DefaultAudioDevice implements AudioDevice {

    private boolean useHardwareAcousticEchoCanceler = true;
    private boolean useHardwareNoiseSuppressor = true;

    /**
     * Control if the built-in hardware AEC should be used or not. The default value is {@code
     * true}.
     *
     * @param useHardwareAcousticEchoCanceler true if audio device should use hardware AEC, false
     *     otherwise
     */
    public void setUseHardwareAcousticEchoCanceler(boolean useHardwareAcousticEchoCanceler) {
        this.useHardwareAcousticEchoCanceler = useHardwareAcousticEchoCanceler;
    }

    /**
     * Control if the built-in hardware noise suppressor should be used or not. The default value is
     * {@code true}.
     *
     * @param useHardwareNoiseSuppressor true if audio device should use hardware noise suppressor,
     *     false otherwise.
     */
    public void setUseHardwareNoiseSuppressor(boolean useHardwareNoiseSuppressor) {
        this.useHardwareNoiseSuppressor = useHardwareNoiseSuppressor;
    }

    /**
     * Indicates if the built-in hardware AEC is used or not. The default value is {@code true}.
     *
     * @return true if the built-in hardware AEC is used.
     */
    public boolean useHardwareAcousticEchoCanceler() {
        return this.useHardwareAcousticEchoCanceler;
    }

    /**
     * Indicates if the built-in hardware noise suppressor is used or not. The default value is
     * {@code true}.
     *
     * @return true if the built-in hardware noise suppressor is used.
     */
    public boolean useHardwareNoiseSuppressor() {
        return this.useHardwareNoiseSuppressor;
    }

    /** This method is a no-op that returns {@code null}. */
    @Override
    @Nullable
    public AudioFormat getCapturerFormat() {
        return null;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onInitCapturer() {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStartCapturing(@NonNull AudioDeviceContext audioDeviceContext) {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStopCapturing() {
        return false;
    }

    /** This method is a no-op that returns {@code null}. */
    @Override
    @Nullable
    public AudioFormat getRendererFormat() {
        return null;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onInitRenderer() {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStartRendering(@NonNull AudioDeviceContext audioDeviceContext) {
        return false;
    }

    /** This method is a no-op that returns {@code false}. */
    @Override
    public boolean onStopRendering() {
        return false;
    }
}
