package com.twilio.voice;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/** Represents options available when connecting to a {@link Call}. */
public abstract class CallOptions {
    private static final Set<Class> SUPPORTED_CODECS =
            new HashSet<Class>(
                    Arrays.asList(
                            // Audio codecs
                            OpusCodec.class, PcmuCodec.class));

    List<LocalAudioTrack> audioTracks;
    IceOptions iceOptions;
    boolean enableDscp;
    boolean enableIceGatheringOnAnyAddressPorts;
    List<AudioCodec> preferredAudioCodecs;
    PlatformInfo platformInfo;
    Call.CallMessageListener callMessageListener;
    AudioOptions audioOptions;

    static void checkAudioCodecs(@NonNull List<AudioCodec> audioCodecs) {
        for (AudioCodec audioCodec : audioCodecs) {
            Preconditions.checkNotNull(audioCodec);
            Preconditions.checkArgument(
                    SUPPORTED_CODECS.contains(audioCodec.getClass()),
                    String.format("Unsupported audio codec %s", audioCodec.getName()));
        }
    }

    @SuppressWarnings("unused")
    CallOptions() {
        // Package scoped constructor ensures the Javadoc does not show a default constructor
    }

    List<LocalAudioTrack> getAudioTracks() {
        return audioTracks;
    }

    /** Returns the custom ICE configuration used to connect to a Call. */
    @Nullable
    public IceOptions getIceOptions() {
        return iceOptions;
    }

    /** Returns the preferred audio codecs. */
    @Nullable
    public List<AudioCodec> getPreferredAudioCodecs() {
        return preferredAudioCodecs;
    }

    /**
     * Get the flag that indicates whether <a
     * href="https://tools.ietf.org/html/draft-ietf-tsvwg-rtcweb-qos-18#section-5">Differentiated
     * Services Field Code Point (DSCP)</a> is enabled with Expedited Forwarding (EF).
     */
    public boolean isDscpEnabled() {
        return enableDscp;
    }

    /**
     * Get the flag that indicates whether gathering of ICE candidates on "any address" ports is
     * enabled or not.
     */
    public boolean isIceGatheringOnAnyAddressPortsEnabled() {
        return enableIceGatheringOnAnyAddressPorts;
    }

    /** Returns the callMessageListener specified. */
    public Call.CallMessageListener getCallMessageListener() {
        return callMessageListener;
    }

    static void checkAudioTracksReleased(@Nullable List<LocalAudioTrack> audioTracks) {
        if (audioTracks != null) {
            for (LocalAudioTrack localAudioTrack : audioTracks) {
                Preconditions.checkState(
                        !localAudioTrack.isReleased(), "LocalAudioTrack cannot be released");
            }
        }
    }

    LocalAudioTrack[] getLocalAudioTracksArray() {
        LocalAudioTrack[] audioTracksArray = new LocalAudioTrack[0];
        if (audioTracks.size() > 0) {
            audioTracksArray = new LocalAudioTrack[audioTracks.size()];
            audioTracksArray = audioTracks.toArray(audioTracksArray);
        }
        return audioTracksArray;
    }

    AudioCodec[] getAudioCodecsArray() {
        AudioCodec[] audioCodecsArray = new AudioCodec[0];
        if (preferredAudioCodecs != null && !preferredAudioCodecs.isEmpty()) {
            audioCodecsArray = new AudioCodec[preferredAudioCodecs.size()];
            audioCodecsArray = preferredAudioCodecs.toArray(audioCodecsArray);
        }
        return audioCodecsArray;
    }

    /**
     * Returns the {@link AudioOptions} object that was set using the {@link AudioOptions.Builder}.
     *
     * @return {@link AudioOptions}
     */
    AudioOptions getAudioOptions() {
        return this.audioOptions;
    }

    /**
     * Build new {@link CallOptions}.
     *
     * <p>All methods are optional.
     */
    public abstract static class Builder {
        IceOptions iceOptions;
        List<LocalAudioTrack> audioTracks;
        List<AudioCodec> preferredAudioCodecs;
        boolean enableDscp = true;
        boolean enableIceGatheringOnAnyAddressPorts = false;
        Call.EventListener eventListener = null;
        Call.CallMessageListener callMessageListener = null;
        AudioOptions audioOptions;

        @SuppressWarnings("unused")
        Builder() {
            // Package scoped constructor ensures the Javadoc does not show a default constructor
        }

        Builder audioTracks(@NonNull List<LocalAudioTrack> audioTracks) {
            Preconditions.checkNotNull(audioTracks, "audioTracks must not be null");
            this.audioTracks = new ArrayList<>(audioTracks);
            return this;
        }

        Builder iceOptions(@NonNull IceOptions iceOptions) {
            Preconditions.checkNotNull(iceOptions, "iceOptions must not be null");
            this.iceOptions = iceOptions;
            return this;
        }

        Builder enableDscp(boolean enable) {
            this.enableDscp = enable;
            return this;
        }

        Builder enableIceGatheringOnAnyAddressPorts(boolean enable) {
            this.enableIceGatheringOnAnyAddressPorts = enable;
            return this;
        }

        Builder preferAudioCodecs(@NonNull List<AudioCodec> preferredAudioCodecs) {
            Preconditions.checkNotNull(
                    preferredAudioCodecs, "preferredAudioCodecs must not be null");
            this.preferredAudioCodecs = new ArrayList<>(preferredAudioCodecs);
            return this;
        }

        Builder callMessageListener(@NonNull Call.CallMessageListener callMessageListener) {
            this.callMessageListener = callMessageListener;
            return this;
        }

        Builder audioOptions(@NonNull AudioOptions audioOptions) {
            Preconditions.checkNotNull(audioOptions, "audioOptions must not be null.");
            this.audioOptions = audioOptions;
            return this;
        }
    }
}
