package com.twilio.voice;

/**
 * Audio options represent the audio processing operations that are performed on the local audio
 * track/source. These options can be set as a part of {@link ConnectOptions} when making outbound
 * calls and/or as a part of {@link AcceptOptions} when receiving incoming calls.
 */
public class AudioOptions {
    private final boolean echoCancellation;
    private final boolean autoGainControl;
    private final boolean noiseSuppression;
    private final boolean highpassFilter;
    private final int audioJitterBufferMaxPackets;
    private final int audioJitterBufferMinDelayMs;

    private AudioOptions(Builder builder) {
        echoCancellation = builder.echoCancellation;
        autoGainControl = builder.autoGainControl;
        noiseSuppression = builder.noiseSuppression;
        highpassFilter = builder.highpassFilter;
        audioJitterBufferMaxPackets = builder.audioJitterBufferMaxPackets;
        audioJitterBufferMinDelayMs = builder.audioJitterBufferMinDelayMs;
    }

    @Override
    public String toString() {
        return "AudioOptions{"
                + "echoCancellation="
                + echoCancellation
                + ", autoGainControl="
                + autoGainControl
                + ", noiseSuppression="
                + noiseSuppression
                + ", highpassFilter="
                + highpassFilter
                + ", audioJitterBufferMaxPackets="
                + audioJitterBufferMaxPackets
                + ", audioJitterBufferMinDelayMs="
                + audioJitterBufferMinDelayMs
                + '}';
    }

    /**
     * Returns whether echo cancellation is enabled.
     *
     * @return boolean
     */
    public boolean isEchoCancellationEnabled() {
        return this.echoCancellation;
    }

    /**
     * Returns whether automatic gain control is enabled.
     *
     * @return boolean
     */
    public boolean isAutoGainControlEnabled() {
        return this.autoGainControl;
    }

    /**
     * Returns whether noise suppression is enabled.
     *
     * @return boolean
     */
    public boolean isNoiseSuppressionEnabled() {
        return this.noiseSuppression;
    }

    /**
     * Returns whether removal of background noise of lower frequencies is enabled.
     *
     * @return boolean
     */
    public boolean isHighpassFilterEnabled() {
        return this.highpassFilter;
    }

    /**
     * Returns the audio receiver jitter buffer (NetEq) max capacity in number of packets.
     *
     * @return int
     */
    public int getAudioJitterBufferMaxPackets() {
        return this.audioJitterBufferMaxPackets;
    }

    /**
     * Returns the audio receiver jitter buffer (NetEq) minimum target delay in milliseconds.
     *
     * @return int
     */
    public int getAudioJitterBufferMinDelayMs() {
        return this.audioJitterBufferMinDelayMs;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if ((o == null) || !(o instanceof AudioOptions)) {
            return false;
        }
        final AudioOptions audioOptions = (AudioOptions) o;
        return (this.echoCancellation == audioOptions.echoCancellation
                && this.autoGainControl == audioOptions.autoGainControl
                && this.noiseSuppression == audioOptions.noiseSuppression
                && this.highpassFilter == audioOptions.highpassFilter
                && this.audioJitterBufferMaxPackets == audioOptions.audioJitterBufferMaxPackets
                && this.audioJitterBufferMinDelayMs == audioOptions.audioJitterBufferMinDelayMs);
    }

    /** Builds new {@link AudioOptions}. */
    public static final class Builder {
        private boolean echoCancellation = true;
        private boolean autoGainControl = true;
        private boolean noiseSuppression = true;
        private boolean highpassFilter = true;
        private int audioJitterBufferMaxPackets = 200;
        private int audioJitterBufferMinDelayMs = 0;

        public Builder() {}

        /**
         * Audio processing that attempts to filter away the output signal from later inbound
         * pickup. Default is set to true.
         *
         * @param echoCancellation - true enables echo cancellation and false disables it.
         * @return {@link AudioOptions.Builder}
         */
        public Builder echoCancellation(boolean echoCancellation) {
            this.echoCancellation = echoCancellation;
            return this;
        }

        /**
         * Audio processing that adjusts the sensitivity of the local microphone dynamically.
         * Default is set to true.
         *
         * @param autoGainControl - true enables automatic gain control and false disables it.
         * @return {@link AudioOptions.Builder}
         */
        public Builder autoGainControl(boolean autoGainControl) {
            this.autoGainControl = autoGainControl;
            return this;
        }

        /**
         * Enable audio processing that filters out background noise. Default is set to true.
         *
         * @param noiseSuppression - true filters out background noise and false disables the
         *     filtering.
         * @return {@link AudioOptions.Builder}
         */
        public Builder noiseSuppression(boolean noiseSuppression) {
            this.noiseSuppression = noiseSuppression;
            return this;
        }

        /**
         * Enable audio processing that removes background noise of lower frequencies. Default is
         * set to true.
         *
         * @param highpassFilter - true removes lower frequency background noise and false disables
         *     the removal.
         * @return {@link AudioOptions.Builder}
         */
        public Builder highpassFilter(boolean highpassFilter) {
            this.highpassFilter = highpassFilter;
            return this;
        }

        /**
         * Audio receiver jitter buffer (NetEq) max capacity in number of packets. Minimum allowed
         * value is 20. Default is set at 200 packets.
         *
         * @param maxPackets - Max capacity in number of packets.
         * @return {@link AudioOptions.Builder}
         */
        public Builder audioJitterBufferMaxPackets(int maxPackets) {
            maxPackets = (maxPackets < 20) ? 20 : maxPackets;
            this.audioJitterBufferMaxPackets = maxPackets;
            return this;
        }

        /**
         * Audio receiver jitter buffer (NetEq) minimum target delay in milliseconds. Minimum
         * allowed value is zero msec. Default is set at zero msec.
         *
         * @param minDelayMs - Minimum target delay in milliseconds.
         * @return {@link AudioOptions.Builder}
         */
        public Builder audioJitterBufferMinDelayMs(int minDelayMs) {
            minDelayMs = (minDelayMs < 0) ? 0 : minDelayMs;
            this.audioJitterBufferMinDelayMs = minDelayMs;
            return this;
        }

        public AudioOptions build() {
            return new AudioOptions(this);
        }
    }
}
