package com.twilio.video;

import static java.lang.annotation.RetentionPolicy.SOURCE;

import android.Manifest;
import android.content.Context;
import android.hardware.Camera;
import android.os.Handler;
import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.lang.annotation.Retention;
import java.util.Arrays;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import tvi.webrtc.Camera1Capturer;
import tvi.webrtc.Camera1Enumerator;
import tvi.webrtc.Camera1Session;
import tvi.webrtc.CameraVideoCapturer;
import tvi.webrtc.CapturerObserver;
import tvi.webrtc.SurfaceTextureHelper;
import tvi.webrtc.ThreadUtils;

/**
 * The CameraCapturer class is used to provide video frames for a {@link LocalVideoTrack} from a
 * given {@link #cameraId}. The frames are provided via the preview API of {@link
 * android.hardware.Camera}.
 *
 * <p>This class represents an implementation of a {@link VideoCapturer} interface. Although public,
 * these methods are not meant to be invoked directly.
 *
 * <p><b>Note</b>: This capturer can be reused, but cannot be shared across multiple {@link
 * LocalVideoTrack}s simultaneously.
 */
public class CameraCapturer implements VideoCapturer {
    /*
     * Some devices take up to three seconds before the camera resource is released and WebRTC
     * notifies this class that the camera session is closed.
     */
    private static final int CAMERA_CLOSED_TIMEOUT_MS = 3000;
    private static final String ERROR_MESSAGE_CAMERA_SERVER_DIED = "Camera server died!";
    private static final String ERROR_MESSAGE_UNKNOWN = "Camera error:";
    private static final Logger logger = Logger.getLogger(CameraCapturer.class);

    @Retention(SOURCE)
    @IntDef({
        ERROR_CAMERA_FREEZE,
        ERROR_CAMERA_SERVER_STOPPED,
        ERROR_UNSUPPORTED_SOURCE,
        ERROR_CAMERA_PERMISSION_NOT_GRANTED,
        ERROR_CAMERA_SWITCH_FAILED,
        ERROR_UNKNOWN
    })
    public @interface Error {}

    public static final int ERROR_CAMERA_FREEZE = 0;
    public static final int ERROR_CAMERA_SERVER_STOPPED = 1;
    public static final int ERROR_UNSUPPORTED_SOURCE = 2;
    public static final int ERROR_CAMERA_PERMISSION_NOT_GRANTED = 3;
    public static final int ERROR_CAMERA_SWITCH_FAILED = 5;
    public static final int ERROR_UNKNOWN = 6;

    /*
     * State definitions used to control interactions with the public API
     */
    private enum State {
        IDLE,
        STARTING,
        RUNNING,
        STOPPING
    }

    // These fields are used to safely transition between CameraCapturer states
    private final Object stateLock = new Object();
    private State state = State.IDLE;

    private final Context context;
    private final Camera1Enumerator camera1Enumerator;
    private final AtomicBoolean parameterUpdatePending = new AtomicBoolean(false);
    private CameraCapturer.Listener listener;
    private final Handler handler;
    @VisibleForTesting tvi.webrtc.Camera1Capturer webRtcCameraCapturer;
    private String cameraId;
    private String pendingCameraId;
    private Camera1Session camera1Session;
    private CapturerObserver capturerObserver;
    private final tvi.webrtc.CapturerObserver observerAdapter =
            new tvi.webrtc.CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean success) {
                    if (success) {
                        synchronized (stateLock) {
                            if (state == State.STARTING) {
                                setSession();
                                state = State.RUNNING;
                                /*
                                 * The user requested a camera parameter update while the capturer was
                                 * not running. We need to apply these parameters after the capturer
                                 * is started to ensure consistency on the camera capturer instance.
                                 */
                                if (cameraParameterUpdater != null) {
                                    updateCameraParametersOnCameraThread(cameraParameterUpdater);
                                    cameraParameterUpdater = null;
                                }
                            } else {
                                logger.w("Attempted to transition from " + state + " to RUNNING");
                            }
                        }
                    } else {
                        logger.e("Failed to start capturer");
                    }

                    capturerObserver.onCapturerStarted(success);
                }

                @Override
                public void onCapturerStopped() {
                    capturerObserver.onCapturerStopped();
                }

                @Override
                public void onFrameCaptured(tvi.webrtc.VideoFrame videoFrame) {
                    capturerObserver.onFrameCaptured(videoFrame);
                }
            };
    private CameraParameterUpdater cameraParameterUpdater;

    private CountDownLatch cameraClosed;
    private final CameraVideoCapturer.CameraEventsHandler cameraEventsHandler =
            new CameraVideoCapturer.CameraEventsHandler() {
                @Override
                public void onCameraError(String errorMsg) {
                    if (listener != null) {
                        if (errorMsg.equals(ERROR_MESSAGE_CAMERA_SERVER_DIED)) {
                            logger.e("Camera server stopped.");
                            listener.onError(CameraCapturer.ERROR_CAMERA_SERVER_STOPPED);
                        } else if (errorMsg.contains(ERROR_MESSAGE_UNKNOWN)) {
                            logger.e("Unknown camera error occurred.");
                            listener.onError(CameraCapturer.ERROR_UNKNOWN);
                        }
                    }
                    synchronized (stateLock) {
                        state = State.IDLE;
                    }
                }

                @Override
                public void onCameraFreezed(String s) {
                    logger.e("Camera froze.");
                    if (listener != null) {
                        listener.onError(CameraCapturer.ERROR_CAMERA_FREEZE);
                    }
                    synchronized (stateLock) {
                        state = State.IDLE;
                    }
                }

                @Override
                public void onCameraOpening(String message) {
                    // Ignore this event for now
                }

                @Override
                public void onFirstFrameAvailable() {
                    setSession();
                    if (listener != null) {
                        listener.onFirstFrameAvailable();
                    }
                }

                @Override
                public void onCameraClosed() {
                    synchronized (stateLock) {
                        if (state == State.STOPPING) {
                            // Null out the camera session because it is no longer usable
                            CameraCapturer.this.camera1Session = null;

                            // We are awaiting the camera being freed in stopCapture
                            cameraClosed.countDown();
                        }
                    }
                }

                @Override
                public void onCameraDisconnected() {
                    // TODO: do we need to handle this?
                }
            };

    private final CameraVideoCapturer.CameraSwitchHandler cameraSwitchHandler =
            new CameraVideoCapturer.CameraSwitchHandler() {
                @Override
                public void onCameraSwitchDone(boolean isFrontCamera) {
                    synchronized (CameraCapturer.this) {
                        cameraId = pendingCameraId;
                        pendingCameraId = null;
                    }
                    if (listener != null) {
                        listener.onCameraSwitched(cameraId);
                    }
                }

                @Override
                public void onCameraSwitchError(String errorMessage) {
                    logger.e("Failed to switch to camera source " + cameraId);
                    if (listener != null) {
                        listener.onError(ERROR_CAMERA_SWITCH_FAILED);
                    }
                    synchronized (stateLock) {
                        state = State.IDLE;
                    }
                }
            };

    /**
     * Constructs a CameraCapturer instance.
     *
     * @param context application context
     * @param cameraId unique identifier of the camera device to open that must be specified in
     *     {@link tvi.webrtc.Camera1Enumerator#getDeviceNames()}
     */
    public CameraCapturer(@NonNull Context context, @NonNull String cameraId) {
        this(context, cameraId, null);
    }

    /**
     * Constructs a CameraCapturer instance.
     *
     * @param context application context
     * @param cameraId unique identifier of the camera device to open that must be specified in
     *     {@link tvi.webrtc.Camera1Enumerator#getDeviceNames()}
     * @param listener an optional listener of camera capturer events
     */
    public CameraCapturer(
            @NonNull Context context, @NonNull String cameraId, @Nullable Listener listener) {
        this(context, cameraId, listener, Util.createCallbackHandler());
    }

    /*
     * Package scope constructor that allows passing in a mocked handler for unit tests.
     */
    @VisibleForTesting
    CameraCapturer(
            @NonNull Context context,
            @NonNull String cameraId,
            @Nullable Listener listener,
            @NonNull Handler handler) {
        Preconditions.checkNotNull(context, "Context must not be null");
        Preconditions.checkNotNull(cameraId, "Camera source must not be null");
        Preconditions.checkArgument(!cameraId.isEmpty(), "Camera ID must not be empty");
        this.context = context;
        this.camera1Enumerator = new Camera1Enumerator(false);
        this.cameraId = cameraId;
        this.listener = listener;
        this.handler = handler;
    }

    /** Indicates that the camera capturer is not a screen cast. */
    @Override
    public boolean isScreencast() {
        return false;
    }

    @Override
    public void initialize(
            @NonNull SurfaceTextureHelper surfaceTextureHelper,
            @NonNull Context context,
            @NonNull CapturerObserver capturerObserver) {
        this.capturerObserver = capturerObserver;
        this.webRtcCameraCapturer =
                (Camera1Capturer) camera1Enumerator.createCapturer(cameraId, cameraEventsHandler);
        this.webRtcCameraCapturer.initialize(surfaceTextureHelper, context, observerAdapter);
    }

    @Override
    public void startCapture(int width, int height, int framerate) {
        checkCapturerState();
        synchronized (stateLock) {
            if (state == State.IDLE || state == State.STOPPING) {
                logger.i("startCapture");
                state = State.STARTING;
                webRtcCameraCapturer.startCapture(width, height, framerate);
            } else if (state == State.STARTING || state == State.RUNNING) {
                logger.w("attempted to start capturing when capturer is starting or running");
            }
        }
    }

    /**
     * Stops all frames being captured. The {@link android.hardware.Camera} interface should be
     * available for use upon completion.
     *
     * <p><b>Note</b>: This method is not meant to be invoked directly.
     */
    @Override
    public void stopCapture() {
        if (webRtcCameraCapturer != null) {
            synchronized (stateLock) {
                state = State.STOPPING;
                cameraClosed = new CountDownLatch(1);
            }
            webRtcCameraCapturer.stopCapture();

            /*
             * Wait until the camera closed event has fired. This event indicates
             * that CameraCapturer stopped capturing and that the camera resource is released. If
             * the event is not received then log an error and developer will be notified via
             * onError callback.
             */
            if (!ThreadUtils.awaitUninterruptibly(cameraClosed, CAMERA_CLOSED_TIMEOUT_MS)) {
                logger.e("Camera closed event not received");
            }
            synchronized (stateLock) {
                cameraClosed = null;
                state = State.IDLE;
            }
        }
    }

    @Override
    public void dispose() {
        if (webRtcCameraCapturer != null) {
            webRtcCameraCapturer.dispose();
            webRtcCameraCapturer = null;
        }
    }

    /** Returns the currently set cameraID. */
    @NonNull
    public synchronized String getCameraId() {
        return cameraId;
    }

    /**
     * Switches the current {@link #cameraId}. This method can be invoked while capturing frames or
     * not.
     *
     * @param newCameraId the new camera id.
     */
    public synchronized void switchCamera(@NonNull final String newCameraId) {
        Preconditions.checkNotNull(newCameraId, "Camera ID must not be null");
        Preconditions.checkArgument(!newCameraId.isEmpty(), "Camera ID must not be empty");
        Preconditions.checkArgument(
                !newCameraId.equals(cameraId),
                "Camera ID must be different " + "from current camera ID");
        Preconditions.checkArgument(
                cameraIdSupported(camera1Enumerator, newCameraId),
                "Camera ID %s is not supported or could not be validated",
                newCameraId);
        synchronized (stateLock) {
            if (state != State.IDLE) {
                if (pendingCameraId == null) {
                    pendingCameraId = newCameraId;
                    webRtcCameraCapturer.switchCamera(cameraSwitchHandler, newCameraId);
                } else {
                    handler.post(() -> listener.onError(ERROR_CAMERA_SWITCH_FAILED));
                }
            } else {
                cameraId = newCameraId;
                listener.onCameraSwitched(cameraId);
            }
        }
    }

    /**
     * Schedules a camera parameter update. The current camera's {@link
     * android.hardware.Camera.Parameters} will be provided for modification via {@link
     * CameraParameterUpdater#apply(Camera.Parameters)}. Any changes to the parameters will be
     * applied after the invocation of this callback. This method can be invoked while capturing
     * frames or not.
     *
     * <p>The following snippet demonstrates how to turn on the flash of a camera while capturing.
     *
     * <pre><code>
     *     // Create camera capturer
     *     CameraCapturer cameraCapturer = new CameraCapturer(context, cameraId, null);
     *
     *     // Start camera capturer
     *     LocalVideoTrack cameraVideoTrack = LocalVideoTrack.create(context, true, cameraCapturer);
     *
     *     // Schedule camera parameter update
     *     cameraCapturer.updateCameraParameters(new CameraParameterUpdater() {
     *        {@literal @}Override
     *         public void apply(Camera.Parameters cameraParameters) {
     *             // Ensure camera supports flash and turn on
     *             if (cameraParameters.getFlashMode() != null) {
     *                  cameraParameters.setFlashMode(Camera.Parameters.FLASH_MODE_TORCH);
     *             }
     *         }
     *     });
     * </code></pre>
     *
     * @param cameraParameterUpdater camera parameter updater that receives current camera
     *     parameters for modification.
     * @return true if update was scheduled or false if an update is pending or could not be
     *     scheduled.
     */
    public synchronized boolean updateCameraParameters(
            @NonNull final CameraParameterUpdater cameraParameterUpdater) {
        synchronized (stateLock) {
            if (state == State.RUNNING) {
                if (!parameterUpdatePending.get()) {
                    parameterUpdatePending.set(true);
                    return webRtcCameraCapturer
                            .getHandler()
                            .post(
                                    () ->
                                            updateCameraParametersOnCameraThread(
                                                    cameraParameterUpdater));
                } else {
                    logger.w("Parameters will not be applied with parameter update pending");
                    return false;
                }
            } else {
                logger.i(
                        "Camera capturer is not running. Parameters will be applied when "
                                + "camera capturer is resumed");
                this.cameraParameterUpdater = cameraParameterUpdater;

                return true;
            }
        }
    }

    private void setSession() {
        logger.d("Camera capture session configured");
        camera1Session = (Camera1Session) webRtcCameraCapturer.getCameraSession();
    }

    private void checkCapturerState() {
        Preconditions.checkState(
                Util.permissionGranted(context, Manifest.permission.CAMERA),
                "CAMERA permission must be granted to start capturer");
        Preconditions.checkState(
                cameraIdSupported(camera1Enumerator, cameraId),
                "Camera ID %s is not supported or could not be validated",
                cameraId);
    }

    private static boolean cameraIdSupported(
            @NonNull Camera1Enumerator camera1Enumerator, @NonNull String targetCameraId) {
        return Arrays.asList(camera1Enumerator.getDeviceNames()).contains(targetCameraId);
    }

    private void updateCameraParametersOnCameraThread(
            @NonNull final CameraParameterUpdater cameraParameterUpdater) {
        synchronized (stateLock) {
            if (state == State.RUNNING) {
                // Grab camera parameters and forward to updater
                Camera camera = camera1Session.getCamera();
                Camera.Parameters cameraParameters = camera.getParameters();
                logger.i("Applying camera parameters");
                cameraParameterUpdater.apply(cameraParameters);

                // Stop preview and clear internal camera buffer to avoid camera freezes
                camera.stopPreview();
                camera.setPreviewCallbackWithBuffer(null);

                // Apply the parameters
                camera.setParameters(cameraParameters);

                // Reinitialize the preview callback and buffer.
                Camera1Session.initializeCallbackBuffer(camera1Session.getCaptureFormat(), camera);
                camera1Session.listenForBytebufferFrames();

                // Resume preview
                camera.startPreview();
            } else {
                logger.w(
                        "Attempted to update camera parameters while camera capturer is "
                                + "not running");
            }

            // Clear the parameter updating flag
            parameterUpdatePending.set(false);
        }
    }

    /** Interface that provides events and errors related to {@link CameraCapturer}. */
    public interface Listener {
        /** Indicates when the first frame has been captured from the camera. */
        void onFirstFrameAvailable();

        /**
         * Notifies when a camera switch is complete.
         *
         * @param newCameraId the camera ID after camera switch is complete.
         */
        void onCameraSwitched(@NonNull String newCameraId);

        /**
         * Reports an error that occurred in {@link CameraCapturer}.
         *
         * @param errorCode the code that describes the error that occurred.
         */
        void onError(@CameraCapturer.Error int errorCode);
    }
}
