package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/** A remote data track publication represents a {@link RemoteDataTrack}. */
public class RemoteDataTrackPublication implements DataTrackPublication {
    private final String sid;
    private final String name;
    private RemoteDataTrack remoteDataTrack;
    private boolean subscribed;
    private boolean enabled;
    private TrackPriority publishPriority;

    RemoteDataTrackPublication(
            boolean subscribed,
            boolean enabled,
            @NonNull String sid,
            @NonNull String name,
            @NonNull TrackPriority publishPriority) {
        this.enabled = enabled;
        this.subscribed = subscribed;
        this.sid = sid;
        this.name = name;
        this.publishPriority = publishPriority;
    }

    /**
     * Returns the remote data track's server identifier. This value uniquely identifies the remote
     * data track within the scope of a {@link Room}.
     */
    @NonNull
    @Override
    public String getTrackSid() {
        return sid;
    }

    /**
     * @return Returns the published remote data track. {@code null} is returned if the track is not
     *     subscribed to.
     */
    @Override
    @Nullable
    public synchronized DataTrack getDataTrack() {
        return remoteDataTrack;
    }

    /**
     * Returns the name of the published data track. An empty string is returned if no track name
     * was specified.
     */
    @Override
    @NonNull
    public String getTrackName() {
        return name;
    }

    /** Returns true if the published data track is enabled or false otherwise. */
    @Override
    public boolean isTrackEnabled() {
        return enabled;
    }

    /** Check if the remote data track is subscribed to by the {@link LocalParticipant}. */
    public synchronized boolean isTrackSubscribed() {
        return subscribed;
    }

    /**
     * Returns the {@link TrackPriority} set by the publisher for the corresponding {@link
     * RemoteDataTrack}. Returns {@link TrackPriority#STANDARD} if no priority was specified by the
     * publisher.
     */
    public synchronized @NonNull TrackPriority getPublishPriority() {
        return publishPriority;
    }

    /**
     * Returns the published remote data track. {@code null} is returned if the track is not
     * subscribed to.
     */
    @Nullable
    public synchronized RemoteDataTrack getRemoteDataTrack() {
        return remoteDataTrack;
    }

    /*
     * Set by remote participant listener proxy.
     */
    synchronized void setSubscribed(boolean subscribed) {
        this.subscribed = subscribed;
    }

    /*
     * Set by remote participant listener proxy.
     */
    synchronized void setPublishPriority(TrackPriority publishPriority) {
        this.publishPriority = publishPriority;
    }

    /*
     * Called from JNI layer when a track has been subscribed to.
     */
    @SuppressWarnings("unused")
    synchronized void setRemoteDataTrack(RemoteDataTrack remoteDataTrack) {
        this.remoteDataTrack = remoteDataTrack;
    }
}
