package com.twilio.video;

import androidx.annotation.NonNull;

/** A remote audio track represents a remote audio source. */
public class RemoteAudioTrack extends AudioTrack {
    private static final Logger logger = Logger.getLogger(RemoteAudioTrack.class);

    private long nativeRemoteAudioTrackHandle;
    private final String sid;
    private boolean playbackEnabled;

    RemoteAudioTrack(
            long nativeRemoteAudioTrackHandle,
            @NonNull String sid,
            @NonNull String name,
            boolean isEnabled) {
        super(nativeRemoteAudioTrackHandle, isEnabled, name);
        this.nativeRemoteAudioTrackHandle = nativeRemoteAudioTrackHandle;
        this.sid = sid;
        this.playbackEnabled = true;
    }

    /**
     * Returns the remote audio track's server identifier. This value uniquely identifies the remote
     * audio track within the scope of a {@link Room}.
     */
    @NonNull
    public String getSid() {
        return sid;
    }

    /**
     * Enables playback of remote audio track. When playback is disabled the audio is muted.
     *
     * @param enable the desired playback state of the remote audio track.
     */
    public synchronized void enablePlayback(boolean enable) {
        this.playbackEnabled = enable;
        if (!isReleased()) {
            nativeEnablePlayback(nativeRemoteAudioTrackHandle, enable);
        } else {
            logger.w(
                    "Cannot enable playback of remote audio track that is no longer "
                            + "subscribed to");
        }
    }

    /**
     * Check if playback on the remote audio track is enabled.
     *
     * <p>When the value is false, the remote audio track is muted. When the value is true the
     * remote audio track is playing.
     *
     * @return true if remote audio is enabled.
     */
    public synchronized boolean isPlaybackEnabled() {
        return playbackEnabled;
    }

    @Override
    synchronized void release() {
        if (!isReleased()) {
            nativeRelease(nativeRemoteAudioTrackHandle);
            nativeRemoteAudioTrackHandle = 0;
        }
    }

    @Override
    boolean isReleased() {
        return nativeRemoteAudioTrackHandle == 0;
    }

    private native void nativeEnablePlayback(long nativeAudioTrackHandle, boolean enable);

    private native void nativeRelease(long nativeAudioTrackHandle);
}
