package com.twilio.video;

import androidx.annotation.VisibleForTesting;
import java.util.Arrays;

/**
 * AudioFormat describes 16-bit signed Linear Pulse Code Modulation (LPCM) audio. This class allows
 * developers to customize the audio format used for audio capturing or rendering.
 *
 * <p>Supported sample rate values are 8000, 16000, 24000, 32000, 44100 and 48000 Hz. Only
 * mono(number of channels 1) and stereo(number of channels 2) are supported by the SDK
 */
public class AudioFormat {

    public static final int AUDIO_SAMPLE_RATE_8000 = 8000;
    public static final int AUDIO_SAMPLE_RATE_16000 = 16000;
    public static final int AUDIO_SAMPLE_RATE_24000 = 24000;
    public static final int AUDIO_SAMPLE_RATE_32000 = 32000;
    public static final int AUDIO_SAMPLE_RATE_44100 = 44100;
    public static final int AUDIO_SAMPLE_RATE_48000 = 48000;

    public static int AUDIO_SAMPLE_MONO = 1;
    public static final int AUDIO_SAMPLE_STEREO = 2;

    @VisibleForTesting
    static final Integer validSampleRates[] =
            new Integer[] {
                Integer.valueOf(AUDIO_SAMPLE_RATE_8000),
                Integer.valueOf(AUDIO_SAMPLE_RATE_16000),
                Integer.valueOf(AUDIO_SAMPLE_RATE_24000),
                Integer.valueOf(AUDIO_SAMPLE_RATE_32000),
                Integer.valueOf(AUDIO_SAMPLE_RATE_44100),
                Integer.valueOf(AUDIO_SAMPLE_RATE_48000)
            };

    @VisibleForTesting
    static final Integer validChannelCounts[] =
            new Integer[] {
                Integer.valueOf(AUDIO_SAMPLE_MONO), Integer.valueOf(AUDIO_SAMPLE_STEREO)
            };

    private final int sampleRate;
    private final int channelCount;

    public AudioFormat(int sampleRate, int channelCount) {
        Preconditions.checkArgument(
                Arrays.asList(validSampleRates).contains(Integer.valueOf(sampleRate)),
                String.format("Unsupported sample rate %s", sampleRate));
        Preconditions.checkArgument(
                Arrays.asList(validChannelCounts).contains(Integer.valueOf(channelCount)),
                String.format("Unsupported channel count %s", channelCount));
        this.sampleRate = sampleRate;
        this.channelCount = channelCount;
    }

    /**
     * Returns the sample rate of the audio stream. Valid values for this property are 8000, 16000,
     * 24000, 32000, 44100 and 48000 Hz.
     *
     * @return sampleRate
     */
    public int getSampleRate() {
        return sampleRate;
    }

    /**
     * Returns the number of channels in the audio stream.
     *
     * @return channelCount
     */
    public int getChannelCount() {
        return channelCount;
    }
}
