package com.twilio.video;

import static com.twilio.video.VideoSinkHintsConsumer.CUSTOM_SINK_HINTS_ID;
import static com.twilio.video.VideoSinkHintsConsumer.MANUAL_SINK_HINTS_ID;
import static com.twilio.video.VideoSinkHintsConsumer.NO_ATTACHED_SINK_HINTS_ID;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.util.Map;
import tvi.webrtc.VideoSink;

/** A remote video track represents a remote video source. */
public class RemoteVideoTrack extends VideoTrack {
    private static final Logger logger = Logger.getLogger(RemoteVideoTrack.class);
    private static final String SWITCH_ON_SWITCH_OFF_ERROR =
            "RemoteVideoTrack.switchOn() and RemoteVideoTrack.switchOff() can only be called after setting ClientTrackSwitchOffControl.MANUAL and not setting maxTracks in VideoBandwidthProfileOptions.\";";
    private static final String SET_CONTENT_PREFERENCES_ERROR =
            "RemoteVideoTrack.setContentPreferences() can only be called after setting VideoContentPreferencesMode.MANUAL and not setting renderDimensions in VideoBandwidthProfileOptions.\";";

    private final String sid;
    private boolean isSwitchedOff;
    private @Nullable TrackPriority priority;
    private long nativeRemoteVideoTrackContext;
    private final @Nullable VideoBandwidthProfileOptions videoBandwidthProfileOptions;
    @VisibleForTesting final @Nullable VideoSinkHintsConsumer videoSinkHintsConsumer;
    private SinkHints cachedSinkHints = new SinkHints(MANUAL_SINK_HINTS_ID, null, null);

    RemoteVideoTrack(
            @NonNull tvi.webrtc.VideoTrack webRtcVideoTrack,
            @NonNull String sid,
            @NonNull String name,
            boolean enabled,
            boolean isSwitchedOff,
            @Nullable TrackPriority priority,
            @Nullable VideoBandwidthProfileOptions videoBandwidthProfileOptions,
            long nativeRemoteVideoTrackContext) {
        super(webRtcVideoTrack, enabled, name);
        this.sid = sid;
        this.isSwitchedOff = isSwitchedOff;
        this.priority = priority;
        this.nativeRemoteVideoTrackContext = nativeRemoteVideoTrackContext;
        this.videoBandwidthProfileOptions = videoBandwidthProfileOptions;
        if (videoBandwidthProfileOptions != null) {
            this.videoSinkHintsConsumer =
                    new VideoSinkHintsConsumer(
                            isInVideoContentPreferencesAutoMode() && isRenderDimensionsNotSet(),
                            isInClientTrackSwitchOffAutoMode() && isMaxTracksNotSet(),
                            this::addSinkHints);
        } else {
            this.videoSinkHintsConsumer = null;
        }
    }

    /**
     * Returns the remote video track's server identifier. This value uniquely identifies the remote
     * video track within the scope of a {@link Room}.
     */
    @NonNull
    public String getSid() {
        return sid;
    }

    @Override
    public synchronized void addSink(@NonNull VideoSink videoSink) {
        super.addSink(videoSink);
        if (!isReleased()) {
            if ((isInClientTrackSwitchOffAutoMode() && isMaxTracksNotSet())
                    || (isInVideoContentPreferencesAutoMode() && isRenderDimensionsNotSet())) {
                long sinkHintsId = videoSinkHintsConsumer.getNextSinkHintsId();
                if (videoSink instanceof VideoTextureView) {
                    ((VideoTextureView) videoSink)
                            .setupVideoSinkHintsProducer(videoSinkHintsConsumer, sinkHintsId);
                } else if (videoSink instanceof VideoView) {
                    ((VideoView) videoSink)
                            .setupVideoSinkHintsProducer(videoSinkHintsConsumer, sinkHintsId);
                } else {
                    // Custom sinks leave track switched on, unless maxTracks is set
                    @Nullable Boolean isEnabled = true;
                    if (!isMaxTracksNotSet()) {
                        isEnabled = null;
                    }
                    addSinkHints(new SinkHints(CUSTOM_SINK_HINTS_ID, isEnabled, null));
                }
                logger.d(
                        "Add a new VideoSink using the auto switch off policy with id = "
                                + sinkHintsId);
            }
        }
    }

    @Override
    public synchronized void removeSink(@NonNull VideoSink videoSink) {
        super.removeSink(videoSink);
        if (!isReleased()) {
            if ((isInClientTrackSwitchOffAutoMode() && isMaxTracksNotSet())
                    || (isInVideoContentPreferencesAutoMode() && isRenderDimensionsNotSet())) {
                checkSinkAttachments();
                logger.d("Removing VideoSinkHintsProducer");
                if (videoSink instanceof VideoTextureView) {
                    VideoTextureView view = (VideoTextureView) videoSink;
                    removeSinkHints(view.getSinkHintsId());
                    view.removeVideoSinkHintsProducer();
                } else if (videoSink instanceof VideoView) {
                    VideoView view = (VideoView) videoSink;
                    removeSinkHints(view.getSinkHintsId());
                    view.removeVideoSinkHintsProducer();
                } else {
                    // Custom sink hints only removed if there are no sinks
                    if (getSinks().isEmpty()) {
                        removeSinkHints(CUSTOM_SINK_HINTS_ID);
                    }
                }
            }
        }
    }

    /**
     * Request to switch on a {@link RemoteVideoTrack}, if it is currently switched off. This method
     * must be called only if the {@link ClientTrackSwitchOffControl} is set to {@link
     * ClientTrackSwitchOffControl#MANUAL} in video bandwidth profile options. Tracks may still
     * remain switched off when available bandwidth is limited.
     *
     * @throws IllegalStateException if the following preconditions exist:</b>
     *     <ol>
     *       <li>{@link BandwidthProfileOptions} has not been set.
     *       <li>{@link ClientTrackSwitchOffControl#AUTO} is set.
     *       <li>{@link VideoBandwidthProfileOptions.Builder#maxTracks(Long)} is set.
     *     </ol>
     */
    public void switchOn() {
        checkClientTrackSwitchOffParameters();
        logger.d("Manually switching on track");
        SinkHints sinkHints =
                new SinkHints(
                        MANUAL_SINK_HINTS_ID, true, cachedSinkHints.getVideoContentPreferences());
        cachedSinkHints = sinkHints;
        addSinkHints(sinkHints);
    }

    /**
     * Request to switch off a {@link RemoteVideoTrack}, if it is currently switched on. This method
     * must be called only if the {@link ClientTrackSwitchOffControl} is set to {@link
     * ClientTrackSwitchOffControl#MANUAL} in video bandwidth profile options.
     *
     * @throws IllegalStateException if the following preconditions exist:</b>
     *     <ol>
     *       <li>{@link BandwidthProfileOptions} has not been set.
     *       <li>{@link ClientTrackSwitchOffControl#AUTO} is set.
     *       <li>{@link VideoBandwidthProfileOptions.Builder#maxTracks(Long)} is set.
     *     </ol>
     */
    public void switchOff() {
        checkClientTrackSwitchOffParameters();
        logger.d("Manually switching off track");
        SinkHints sinkHints =
                new SinkHints(
                        MANUAL_SINK_HINTS_ID, false, cachedSinkHints.getVideoContentPreferences());
        cachedSinkHints = sinkHints;
        addSinkHints(sinkHints);
    }

    /**
     * Sets the {@link VideoContentPreferences} for the {@link RemoteVideoTrack}. This method must
     * be called only if the {@link VideoContentPreferencesMode} is set to {@link
     * VideoContentPreferencesMode#MANUAL} in video bandwidth profile options.
     *
     * @param videoContentPreferences The {@link VideoContentPreferences} for this {@link
     *     RemoteVideoTrack}.
     * @throws IllegalStateException if the following preconditions exist:</b>
     *     <ol>
     *       <li>{@link BandwidthProfileOptions} has not been set.
     *       <li>{@link VideoContentPreferencesMode#AUTO} is set.
     *       <li>{@link VideoBandwidthProfileOptions.Builder#renderDimensions(Map)} is set.
     *     </ol>
     */
    public void setContentPreferences(VideoContentPreferences videoContentPreferences) {
        checkSetContentPreferencesParameters();
        SinkHints sinkHints =
                new SinkHints(
                        MANUAL_SINK_HINTS_ID, cachedSinkHints.isEnabled(), videoContentPreferences);
        cachedSinkHints = sinkHints;
        addSinkHints(sinkHints);
    }

    /** Return whether the track is switched off. */
    public boolean isSwitchedOff() {
        return isSwitchedOff;
    }

    /**
     * Get the subscriber's {@link TrackPriority} for this {@link RemoteVideoTrack}. The default
     * subscriber priority is {@code null}, which indicates that the subscriber has not set a
     * priority for this {@link RemoteVideoTrack}.
     */
    public @Nullable TrackPriority getPriority() {
        return priority;
    }

    /**
     * Set the subscriber's {@link TrackPriority} for this {@link RemoteVideoTrack}. Providing
     * {@code null} clears the subscriber's {@link TrackPriority} for this {@link RemoteVideoTrack}.
     * This method is a no-op if the {@link RemoteVideoTrack} has been unsubscribed from.
     *
     * @param priority The priority to be set.
     */
    public void setPriority(@Nullable TrackPriority priority) {
        this.priority = priority;
        if (!isReleased()) {
            nativeSetPriority(nativeRemoteVideoTrackContext, priority);
        }
    }

    void setSwitchedOff(boolean isSwitchedOff) {
        this.isSwitchedOff = isSwitchedOff;
    }

    @Override
    synchronized void release() {
        if (!isReleased()) {
            removeVideoSinkHintsProducers();
            super.release();
            nativeRelease(nativeRemoteVideoTrackContext);
            nativeRemoteVideoTrackContext = 0;
        }
    }

    @Override
    synchronized boolean isReleased() {
        return nativeRemoteVideoTrackContext == 0 && super.isReleased();
    }

    void checkSinkAttachments() {
        if (isInClientTrackSwitchOffAutoMode() && isMaxTracksNotSet()) {
            if (getSinks().isEmpty()) {
                SinkHints sinkHints = new SinkHints(NO_ATTACHED_SINK_HINTS_ID, false, null);
                addSinkHints(sinkHints);
            } else {
                removeSinkHints(NO_ATTACHED_SINK_HINTS_ID);
            }
        }
    }

    private void removeVideoSinkHintsProducers() {
        for (VideoSink videoSink : getVideoSinks()) {
            if (videoSink instanceof VideoTextureView) {
                ((VideoTextureView) videoSink).removeVideoSinkHintsProducer();
            }
            if (videoSink instanceof VideoView) {
                ((VideoView) videoSink).removeVideoSinkHintsProducer();
            }
        }
    }

    private boolean isInVideoContentPreferencesAutoMode() {
        return videoBandwidthProfileOptions != null
                && (videoBandwidthProfileOptions.getVideoContentPreferencesMode()
                                == VideoContentPreferencesMode.AUTO
                        || videoBandwidthProfileOptions.getVideoContentPreferencesMode() == null);
    }

    private boolean isInClientTrackSwitchOffAutoMode() {
        return videoBandwidthProfileOptions != null
                && (videoBandwidthProfileOptions.getClientTrackSwitchOffControl()
                                == ClientTrackSwitchOffControl.AUTO
                        || videoBandwidthProfileOptions.getClientTrackSwitchOffControl() == null);
    }

    private void addSinkHints(SinkHints sinkHints) {
        if (!isReleased()) {
            logger.d("Add SinkHints = " + sinkHints);
            nativeAddSinkHints(nativeRemoteVideoTrackContext, sinkHints);
        }
    }

    private void removeSinkHints(long sinkId) {
        logger.d("Removing SinkHints for sink id: " + sinkId);
        nativeRemoveSinkHints(nativeRemoteVideoTrackContext, sinkId);
    }

    private void checkClientTrackSwitchOffParameters() {
        Preconditions.checkState(
                videoBandwidthProfileOptions != null
                        && isMaxTracksNotSet()
                        && videoBandwidthProfileOptions.getClientTrackSwitchOffControl()
                                == ClientTrackSwitchOffControl.MANUAL,
                SWITCH_ON_SWITCH_OFF_ERROR);
    }

    private void checkSetContentPreferencesParameters() {
        Preconditions.checkState(
                videoBandwidthProfileOptions != null
                        && videoBandwidthProfileOptions.getVideoContentPreferencesMode()
                                == VideoContentPreferencesMode.MANUAL
                        && isRenderDimensionsNotSet(),
                SET_CONTENT_PREFERENCES_ERROR);
    }

    private boolean isMaxTracksNotSet() {
        return videoBandwidthProfileOptions != null
                && videoBandwidthProfileOptions.getMaxTracks() == null;
    }

    private boolean isRenderDimensionsNotSet() {
        return videoBandwidthProfileOptions != null
                && videoBandwidthProfileOptions.getRenderDimensions().isEmpty();
    }

    private native void nativeSetPriority(
            long nativeRemoteVideoTrackContext, TrackPriority priority);

    @VisibleForTesting
    native void nativeRelease(long nativeRemoteVideoTrackContext);

    @VisibleForTesting
    native void nativeAddSinkHints(long nativeRemoteVideoTrackContext, SinkHints sinkHints);

    @VisibleForTesting
    native void nativeRemoveSinkHints(long nativeRemoteVideoTrackContext, long sinkId);
}
