package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import tvi.webrtc.VideoDecoderFactory;
import tvi.webrtc.VideoEncoderFactory;

/*
 * Provides options for creating a test MediaFactory instance. Useful for simulating media scenarios
 * on a device.
 */
@VisibleForTesting(otherwise = VisibleForTesting.NONE)
class MediaOptions {
    @SuppressWarnings("unused")
    @Nullable
    private final String audioFilePath;

    @Nullable final VideoEncoderFactory videoEncoderFactory;
    @Nullable final VideoDecoderFactory videoDecoderFactory;

    private MediaOptions(Builder builder) {
        this.audioFilePath = builder.audioFilePath;
        this.videoEncoderFactory = builder.videoEncoderFactory;
        this.videoDecoderFactory = builder.videoDecoderFactory;
    }

    static class Builder {
        @Nullable private String audioFilePath;
        @Nullable private VideoEncoderFactory videoEncoderFactory;
        @Nullable private VideoDecoderFactory videoDecoderFactory;

        /*
         * Provide a path to an audio file. Providing an audio file path configures the
         * MediaFactory instance to use an audio device module that captures audio from the
         * given file.
         */
        Builder audioFilePath(@NonNull String audioFilePath) {
            Preconditions.checkNotNull("audioFilePath should not be null", audioFilePath);
            this.audioFilePath = audioFilePath;
            return this;
        }

        Builder videoEncoderFactory(VideoEncoderFactory videoEncoderFactory) {
            this.videoEncoderFactory = videoEncoderFactory;
            return this;
        }

        Builder videoDecoderFactory(VideoDecoderFactory videoDecoderFactory) {
            this.videoDecoderFactory = videoDecoderFactory;
            return this;
        }

        MediaOptions build() {
            return new MediaOptions(this);
        }
    }
}
