/*
 * Copyright (C) 2020 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;

/**
 * BandwidthProfileOptions enables you to configure how available bandwidth is shared among the
 * Tracks in a Group Room.
 *
 * @see <a href="https://www.twilio.com/docs/video/tutorials/using-bandwidth-profile-api">Using
 *     Bandwidth Profile API</a>
 */
public class BandwidthProfileOptions {
    @NonNull private final VideoBandwidthProfileOptions videoBandwidthProfileOptions;

    /** Construct instance with SDK defaults. */
    public BandwidthProfileOptions() {
        this.videoBandwidthProfileOptions = new VideoBandwidthProfileOptions.Builder().build();
    }

    /**
     * Construct instance with provided {@link VideoBandwidthProfileOptions}
     *
     * @param videoBandwidthProfileOptions Parameter to configure how available bandwidth is shared
     *     among the {@link RemoteVideoTrack}s in a Group room.
     */
    public BandwidthProfileOptions(
            @NonNull VideoBandwidthProfileOptions videoBandwidthProfileOptions) {
        Preconditions.checkNotNull(
                videoBandwidthProfileOptions, "VideoBandwidthProfileOptions should not be null");
        this.videoBandwidthProfileOptions = videoBandwidthProfileOptions;
    }

    /** Get the video options specified in the {@link BandwidthProfileOptions} constructor. */
    @NonNull
    public VideoBandwidthProfileOptions getVideoBandwidthProfileOptions() {
        return videoBandwidthProfileOptions;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof BandwidthProfileOptions)) return false;

        BandwidthProfileOptions that = (BandwidthProfileOptions) o;

        return getVideoBandwidthProfileOptions().equals(that.getVideoBandwidthProfileOptions());
    }

    @Override
    public int hashCode() {
        return getVideoBandwidthProfileOptions().hashCode();
    }
}
