/*
 * Copyright (C) 2017 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;

/** Represents options when adding a {@link LocalAudioTrack}. */
public class AudioOptions {
    public final boolean echoCancellation;
    public final boolean autoGainControl;
    public final boolean noiseSuppression;
    public final boolean highpassFilter;
    public final boolean stereoSwapping;
    public final boolean audioJitterBufferFastAccelerate;
    public final boolean typingDetection;

    private AudioOptions(@NonNull Builder builder) {
        echoCancellation = builder.echoCancellation;
        autoGainControl = builder.autoGainControl;
        noiseSuppression = builder.noiseSuppression;
        highpassFilter = builder.highpassFilter;
        stereoSwapping = builder.stereoSwapping;
        audioJitterBufferFastAccelerate = builder.audioJitterBufferFastAccelerate;
        typingDetection = builder.typingDetection;
    }

    @NonNull
    @Override
    public String toString() {
        return "AudioOptions{"
                + "echoCancellation="
                + echoCancellation
                + ", autoGainControl="
                + autoGainControl
                + ", noiseSuppression="
                + noiseSuppression
                + ", highpassFilter="
                + highpassFilter
                + ", stereoSwapping="
                + stereoSwapping
                + ", audioJitterBufferFastAccelerate="
                + audioJitterBufferFastAccelerate
                + ", typingDetection="
                + typingDetection
                + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof AudioOptions)) return false;

        AudioOptions that = (AudioOptions) o;

        if (echoCancellation != that.echoCancellation) return false;
        if (autoGainControl != that.autoGainControl) return false;
        if (noiseSuppression != that.noiseSuppression) return false;
        if (highpassFilter != that.highpassFilter) return false;
        if (stereoSwapping != that.stereoSwapping) return false;
        if (audioJitterBufferFastAccelerate != that.audioJitterBufferFastAccelerate) return false;
        return typingDetection == that.typingDetection;
    }

    @Override
    public int hashCode() {
        int result = (echoCancellation ? 1 : 0);
        result = 31 * result + (autoGainControl ? 1 : 0);
        result = 31 * result + (noiseSuppression ? 1 : 0);
        result = 31 * result + (highpassFilter ? 1 : 0);
        result = 31 * result + (stereoSwapping ? 1 : 0);
        result = 31 * result + (audioJitterBufferFastAccelerate ? 1 : 0);
        result = 31 * result + (typingDetection ? 1 : 0);
        return result;
    }

    /** Builds new {@link AudioOptions}. */
    public static final class Builder {
        private boolean echoCancellation = true;
        private boolean autoGainControl = true;
        private boolean noiseSuppression = true;
        private boolean highpassFilter = true;
        private boolean stereoSwapping = false;
        private boolean audioJitterBufferFastAccelerate = false;
        private boolean typingDetection = true;

        public Builder() {}

        /**
         * Attempts to filter away the output signal from later inbound pickup. Defaults to `true`.
         */
        @NonNull
        public Builder echoCancellation(boolean echoCancellation) {
            this.echoCancellation = echoCancellation;
            return this;
        }

        /** Adjust the sensitivity of the local mic dynamically. Defaults to `true`. */
        @NonNull
        public Builder autoGainControl(boolean autoGainControl) {
            this.autoGainControl = autoGainControl;
            return this;
        }

        /** Filter out background noise. Defaults to `true`. */
        @NonNull
        public Builder noiseSuppression(boolean noiseSuppression) {
            this.noiseSuppression = noiseSuppression;
            return this;
        }

        /** Remove background noise of lower frequencies. Defaults to `true`. */
        @NonNull
        public Builder highpassFilter(boolean highpassFilter) {
            this.highpassFilter = highpassFilter;
            return this;
        }

        /** Swap left and right audio channels. Defaults to `false`. */
        @NonNull
        public Builder stereoSwapping(boolean stereoSwapping) {
            this.stereoSwapping = stereoSwapping;
            return this;
        }

        /** Enables fast accelerate mode of jitter buffer. Defaults to `false`. */
        @NonNull
        public Builder audioJitterBufferFastAccelerate(boolean audioJitterBufferFastAccelerate) {
            this.audioJitterBufferFastAccelerate = audioJitterBufferFastAccelerate;
            return this;
        }

        /** Enables typing detection. Defaults to `true`. */
        @NonNull
        public Builder typingDetection(boolean typingDetection) {
            this.typingDetection = typingDetection;
            return this;
        }

        @NonNull
        public AudioOptions build() {
            return new AudioOptions(this);
        }
    }
}
