/*
 * Copyright (C) 2017 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import static java.lang.annotation.RetentionPolicy.SOURCE;

import android.Manifest;
import android.content.Context;
import android.hardware.Camera;
import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.lang.annotation.Retention;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicBoolean;
import tvi.webrtc.Camera1Capturer;
import tvi.webrtc.Camera1Session;
import tvi.webrtc.CameraVideoCapturer;
import tvi.webrtc.CapturerObserver;
import tvi.webrtc.SurfaceTextureHelper;
import tvi.webrtc.ThreadUtils;

/**
 * The CameraCapturer class is used to provide video frames for a {@link LocalVideoTrack} from a
 * given {@link CameraSource}. The frames are provided via the preview API of {@link
 * android.hardware.Camera}.
 *
 * <p>This class represents an implementation of a {@link VideoCapturer} interface. Although public,
 * these methods are not meant to be invoked directly.
 *
 * <p><b>Note</b>: This capturer can be reused, but cannot be shared across multiple {@link
 * LocalVideoTrack}s simultaneously.
 */
public class CameraCapturer implements VideoCapturer {
    /*
     * Some devices take up to three seconds before the camera resource is released and WebRTC
     * notifies this class that the camera session is closed.
     */
    private static final int CAMERA_CLOSED_TIMEOUT_MS = 3000;
    private static final String CAMERA_CLOSED_FAILED = "Failed to close camera";
    private static final String ERROR_MESSAGE_CAMERA_SERVER_DIED = "Camera server died!";
    private static final String ERROR_MESSAGE_UNKNOWN = "Camera error:";
    private static final Logger logger = Logger.getLogger(CameraCapturer.class);

    @Retention(SOURCE)
    @IntDef({
        ERROR_CAMERA_FREEZE,
        ERROR_CAMERA_SERVER_STOPPED,
        ERROR_UNSUPPORTED_SOURCE,
        ERROR_CAMERA_PERMISSION_NOT_GRANTED,
        ERROR_CAMERA_SWITCH_FAILED,
        ERROR_UNKNOWN
    })
    public @interface Error {}

    public static final int ERROR_CAMERA_FREEZE = 0;
    public static final int ERROR_CAMERA_SERVER_STOPPED = 1;
    public static final int ERROR_UNSUPPORTED_SOURCE = 2;
    public static final int ERROR_CAMERA_PERMISSION_NOT_GRANTED = 3;
    public static final int ERROR_CAMERA_SWITCH_FAILED = 5;
    public static final int ERROR_UNKNOWN = 6;

    /** Camera source types. */
    public enum CameraSource {
        FRONT_CAMERA,
        BACK_CAMERA
    }

    /*
     * State definitions used to control interactions with the public API
     */
    private enum State {
        IDLE,
        STARTING,
        RUNNING,
        STOPPING
    }

    // These fields are used to safely transition between CameraCapturer states
    private final Object stateLock = new Object();
    private State state = State.IDLE;

    private final Context context;
    private final CameraCapturerFormatProvider formatProvider;
    private final AtomicBoolean parameterUpdatePending = new AtomicBoolean(false);
    private CameraCapturer.Listener listener;
    @VisibleForTesting tvi.webrtc.Camera1Capturer webRtcCameraCapturer;
    private CameraSource cameraSource;
    private Camera1Session camera1Session;
    private CapturerObserver capturerObserver;
    private final tvi.webrtc.CapturerObserver observerAdapter =
            new tvi.webrtc.CapturerObserver() {
                @Override
                public void onCapturerStarted(boolean success) {
                    capturerObserver.onCapturerStarted(success);

                    // Transition the camera capturer to running state
                    synchronized (stateLock) {
                        /*
                         * We only transition from STARTING to RUNNING
                         */
                        if (state == State.STARTING) {
                            CameraCapturer.this.camera1Session =
                                    (Camera1Session) webRtcCameraCapturer.getCameraSession();

                            state = State.RUNNING;
                            /*
                             * The user requested a camera parameter update while the capturer was
                             * not running. We need to apply these parameters after the capturer
                             * is started to ensure consistency on the camera capturer instance.
                             */
                            if (cameraParameterUpdater != null) {
                                updateCameraParametersOnCameraThread(cameraParameterUpdater);
                                cameraParameterUpdater = null;
                            }
                        } else {
                            logger.w("Attempted to transition from " + state + " to RUNNING");
                        }
                    }
                }

                @Override
                public void onCapturerStopped() {
                    capturerObserver.onCapturerStopped();
                }

                @Override
                public void onFrameCaptured(tvi.webrtc.VideoFrame videoFrame) {
                    capturerObserver.onFrameCaptured(videoFrame);
                }
            };
    private CameraParameterUpdater cameraParameterUpdater;

    private CountDownLatch cameraClosed;
    private final CameraVideoCapturer.CameraEventsHandler cameraEventsHandler =
            new CameraVideoCapturer.CameraEventsHandler() {
                @Override
                public void onCameraError(String errorMsg) {
                    if (listener != null) {
                        if (errorMsg.equals(ERROR_MESSAGE_CAMERA_SERVER_DIED)) {
                            logger.e("Camera server stopped.");
                            listener.onError(CameraCapturer.ERROR_CAMERA_SERVER_STOPPED);
                        } else if (errorMsg.contains(ERROR_MESSAGE_UNKNOWN)) {
                            logger.e("Unknown camera error occurred.");
                            listener.onError(CameraCapturer.ERROR_UNKNOWN);
                        }
                    }
                    synchronized (stateLock) {
                        state = State.IDLE;
                    }
                }

                @Override
                public void onCameraFreezed(String s) {
                    logger.e("Camera froze.");
                    if (listener != null) {
                        listener.onError(CameraCapturer.ERROR_CAMERA_FREEZE);
                    }
                    synchronized (stateLock) {
                        state = State.IDLE;
                    }
                }

                @Override
                public void onCameraOpening(String message) {
                    // Ignore this event for now
                }

                @Override
                public void onFirstFrameAvailable() {
                    if (listener != null) {
                        listener.onFirstFrameAvailable();
                    }
                }

                @Override
                public void onCameraClosed() {
                    synchronized (stateLock) {
                        if (state == State.STOPPING) {
                            // Null out the camera session because it is no longer usable
                            CameraCapturer.this.camera1Session = null;

                            // We are awaiting the camera being freed in stopCapture
                            cameraClosed.countDown();
                        }
                    }
                }

                @Override
                public void onCameraDisconnected() {
                    // TODO: do we need to handle this?
                }
            };

    private final CameraVideoCapturer.CameraSwitchHandler cameraSwitchHandler =
            new CameraVideoCapturer.CameraSwitchHandler() {
                @Override
                public void onCameraSwitchDone(boolean isFrontCamera) {
                    synchronized (CameraCapturer.this) {
                        cameraSource =
                                isFrontCamera
                                        ? CameraSource.FRONT_CAMERA
                                        : CameraSource.BACK_CAMERA;
                    }
                    if (listener != null) {
                        listener.onCameraSwitched();
                    }
                }

                @Override
                public void onCameraSwitchError(String errorMessage) {
                    logger.e("Failed to switch to camera source " + cameraSource);
                    if (listener != null) {
                        listener.onError(ERROR_CAMERA_SWITCH_FAILED);
                    }
                    synchronized (stateLock) {
                        state = State.IDLE;
                    }
                }
            };

    /**
     * Indicates if a camera source is available on the device.
     *
     * @param cameraSource the camera source
     * @return true if source is available on device and false otherwise.
     */
    public static boolean isSourceAvailable(@NonNull CameraSource cameraSource) {
        Preconditions.checkNotNull(cameraSource, "Camera source must not be null");
        CameraCapturerFormatProvider cameraCapturerFormatProvider =
                new CameraCapturerFormatProvider();

        return isSourceAvailable(cameraCapturerFormatProvider, cameraSource);
    }

    static boolean isSourceAvailable(
            @NonNull CameraCapturerFormatProvider cameraCapturerFormatProvider,
            @NonNull CameraSource cameraSource) {
        return cameraCapturerFormatProvider.getCameraId(cameraSource) != -1;
    }

    public CameraCapturer(@NonNull Context context, @NonNull CameraSource cameraSource) {
        this(context, cameraSource, null);
    }

    public CameraCapturer(
            @NonNull Context context,
            @NonNull CameraSource cameraSource,
            @Nullable Listener listener) {
        this(context, cameraSource, listener, new CameraCapturerFormatProvider());
    }

    /*
     * Visible for tests so we can provide a mocked format provider to emulate cases where
     * an error occurs connecting to camera service.
     */
    @VisibleForTesting(otherwise = VisibleForTesting.PRIVATE)
    CameraCapturer(
            @NonNull Context context,
            @NonNull CameraSource cameraSource,
            @Nullable Listener listener,
            @NonNull CameraCapturerFormatProvider formatProvider) {
        Preconditions.checkNotNull(context, "Context must not be null");
        Preconditions.checkNotNull(cameraSource, "Camera source must not be null");
        this.context = context;
        this.cameraSource = cameraSource;
        this.listener = listener;
        this.formatProvider = formatProvider;
    }

    /** Indicates that the camera capturer is not a screen cast. */
    @Override
    public boolean isScreencast() {
        return false;
    }

    @Override
    public void initialize(
            @NonNull SurfaceTextureHelper surfaceTextureHelper,
            @NonNull Context context,
            @NonNull CapturerObserver capturerObserver) {
        this.capturerObserver = capturerObserver;
        boolean capturerCreated = createWebRtcCameraCapturer();
        if (capturerCreated) {
            webRtcCameraCapturer.initialize(surfaceTextureHelper, context, observerAdapter);
        }
    }

    @Override
    public void startCapture(int width, int height, int framerate) {
        if (webRtcCameraCapturer != null) {
            synchronized (stateLock) {
                state = State.STARTING;
            }

            webRtcCameraCapturer.startCapture(width, height, framerate);
        } else {
            logger.e("Failed to startCapture");
        }
    }

    /**
     * Stops all frames being captured. The {@link android.hardware.Camera} interface should be
     * available for use upon completion.
     *
     * <p><b>Note</b>: This method is not meant to be invoked directly.
     */
    @Override
    public void stopCapture() {
        if (webRtcCameraCapturer != null) {
            synchronized (stateLock) {
                state = State.STOPPING;
                cameraClosed = new CountDownLatch(1);
            }
            webRtcCameraCapturer.stopCapture();

            /*
             * Wait until the camera closed event has fired. This event indicates
             * that CameraCapturer stopped capturing and that the camera resource is released. If
             * the event is not received then log an error and developer will be notified via
             * onError callback.
             */
            if (!ThreadUtils.awaitUninterruptibly(cameraClosed, CAMERA_CLOSED_TIMEOUT_MS)) {
                logger.e("Camera closed event not received");
            }
            synchronized (stateLock) {
                cameraClosed = null;
                state = State.IDLE;
            }
        }
    }

    @Override
    public void dispose() {
        if (webRtcCameraCapturer != null) {
            webRtcCameraCapturer.dispose();
            webRtcCameraCapturer = null;
        }
    }

    /** Returns the currently specified camera source. */
    @NonNull
    public synchronized CameraSource getCameraSource() {
        return cameraSource;
    }

    /**
     * Switches the current {@link CameraSource}. This method can be invoked while capturing frames
     * or not.
     */
    public synchronized void switchCamera() {
        CameraSource nextCameraSource =
                cameraSource == CameraSource.FRONT_CAMERA
                        ? CameraSource.BACK_CAMERA
                        : CameraSource.FRONT_CAMERA;
        boolean nextCameraSourceSupported = isSourceAvailable(formatProvider, nextCameraSource);

        if (!nextCameraSourceSupported) {
            logger.w(
                    String.format(
                            "Cannot switch to unsupported camera source %s", nextCameraSource));
            return;
        }

        synchronized (stateLock) {
            if (state != State.IDLE) {
                webRtcCameraCapturer.switchCamera(cameraSwitchHandler);
            } else {
                cameraSource = nextCameraSource;
                if (listener != null) {
                    listener.onCameraSwitched();
                }
            }
        }
    }

    /**
     * Schedules a camera parameter update. The current camera's {@link
     * android.hardware.Camera.Parameters} will be provided for modification via {@link
     * CameraParameterUpdater#apply(Camera.Parameters)}. Any changes to the parameters will be
     * applied after the invocation of this callback. This method can be invoked while capturing
     * frames or not.
     *
     * <p>The following snippet demonstrates how to turn on the flash of a camera while capturing.
     *
     * <pre><code>
     *     // Create camera capturer
     *     CameraCapturer cameraCapturer = new CameraCapturer(context,
     *          CameraCapturer.CameraSource.BACK_CAMERA, null);
     *
     *     // Start camera capturer
     *     LocalVideoTrack cameraVideoTrack = LocalVideoTrack.create(context, true, cameraCapturer);
     *
     *     // Schedule camera parameter update
     *     cameraCapturer.updateCameraParameters(new CameraParameterUpdater() {
     *        {@literal @}Override
     *         public void apply(Camera.Parameters cameraParameters) {
     *             // Ensure camera supports flash and turn on
     *             if (cameraParameters.getFlashMode() != null) {
     *                  cameraParameters.setFlashMode(Camera.Parameters.FLASH_MODE_TORCH);
     *             }
     *         }
     *     });
     * </code></pre>
     *
     * @param cameraParameterUpdater camera parameter updater that receives current camera
     *     parameters for modification.
     * @return true if update was scheduled or false if an update is pending or could not be
     *     scheduled.
     */
    public synchronized boolean updateCameraParameters(
            @NonNull final CameraParameterUpdater cameraParameterUpdater) {
        synchronized (stateLock) {
            if (state == State.RUNNING) {
                if (!parameterUpdatePending.get()) {
                    parameterUpdatePending.set(true);
                    return webRtcCameraCapturer
                            .getHandler()
                            .post(
                                    () ->
                                            updateCameraParametersOnCameraThread(
                                                    cameraParameterUpdater));
                } else {
                    logger.w("Parameters will not be applied with parameter update pending");
                    return false;
                }
            } else {
                logger.i(
                        "Camera capturer is not running. Parameters will be applied when "
                                + "camera capturer is resumed");
                this.cameraParameterUpdater = cameraParameterUpdater;

                return true;
            }
        }
    }

    private boolean createWebRtcCameraCapturer() {
        if (!Util.permissionGranted(context, Manifest.permission.CAMERA)) {
            logger.e("CAMERA permission must be granted to start capturer");
            if (listener != null) {
                listener.onError(ERROR_CAMERA_PERMISSION_NOT_GRANTED);
            }
            synchronized (stateLock) {
                state = State.IDLE;
            }
            return false;
        }
        int cameraId = formatProvider.getCameraId(cameraSource);
        String deviceName = formatProvider.getDeviceName(cameraId);

        if (cameraId < 0 || deviceName == null) {
            logger.e("Failed to find camera source");
            if (listener != null) {
                listener.onError(ERROR_UNSUPPORTED_SOURCE);
            }
            synchronized (stateLock) {
                state = State.IDLE;
            }
            return false;
        }
        webRtcCameraCapturer = new Camera1Capturer(deviceName, cameraEventsHandler, false);

        return true;
    }

    private void updateCameraParametersOnCameraThread(
            @NonNull final CameraParameterUpdater cameraParameterUpdater) {
        synchronized (stateLock) {
            if (state == State.RUNNING) {
                // Grab camera parameters and forward to updater
                Camera camera = camera1Session.getCamera();
                Camera.Parameters cameraParameters = camera.getParameters();
                logger.i("Applying camera parameters");
                cameraParameterUpdater.apply(cameraParameters);

                // Stop preview and clear internal camera buffer to avoid camera freezes
                camera.stopPreview();
                camera.setPreviewCallbackWithBuffer(null);

                // Apply the parameters
                camera.setParameters(cameraParameters);

                // Reinitialize the preview callback and buffer.
                Camera1Session.initializeCallbackBuffer(camera1Session.getCaptureFormat(), camera);
                camera1Session.listenForBytebufferFrames();

                // Resume preview
                camera.startPreview();
            } else {
                logger.w(
                        "Attempted to update camera parameters while camera capturer is "
                                + "not running");
            }

            // Clear the parameter updating flag
            parameterUpdatePending.set(false);
        }
    }

    /** Interface that provides events and errors related to {@link CameraCapturer}. */
    public interface Listener {
        /** Indicates when the first frame has been captured from the camera. */
        void onFirstFrameAvailable();

        /** Notifies when a camera switch is complete. */
        void onCameraSwitched();

        /**
         * Reports an error that occurred in {@link CameraCapturer}.
         *
         * @param errorCode the code that describes the error that occurred.
         */
        void onError(@CameraCapturer.Error int errorCode);
    }
}
