/*
 * Copyright (C) 2018 Twilio, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.twilio.video;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.VisibleForTesting;
import java.util.List;

/*
 * Provides options for creating a test MediaFactory instance. Useful for simulating media scenarios
 * on a device.
 */
@VisibleForTesting(otherwise = VisibleForTesting.NONE)
class MediaOptions {
    /*
     * Fields are read from native media factory layer.
     */
    @Nullable private final List<VideoCodec> fakeVideoCodecs;

    @SuppressWarnings("unused")
    @Nullable
    private final String audioFilePath;

    private MediaOptions(Builder builder) {
        this.fakeVideoCodecs = builder.fakeVideoCodecs;
        this.audioFilePath = builder.audioFilePath;
    }

    static class Builder {
        @Nullable private List<VideoCodec> fakeVideoCodecs;
        @Nullable private String audioFilePath;

        /*
         * Provide a list of codecs that results in a media factory that uses
         * fake encoder and decoder factories with the specified supported video codecs. This
         * method should be used if validating codec specific behavior. By default, this value is
         * null which results in the real encoder and decoder factories being used.
         */
        Builder fakeVideoCodecs(@Nullable List<VideoCodec> fakeVideoCodecs) {
            this.fakeVideoCodecs = fakeVideoCodecs;
            return this;
        }

        /*
         * Provide a path to an audio file. Providing an audio file path configures the
         * MediaFactory instance to use an audio device module that captures audio from the
         * given file.
         */
        Builder audioFilePath(@NonNull String audioFilePath) {
            Preconditions.checkNotNull("audioFilePath should not be null", audioFilePath);
            this.audioFilePath = audioFilePath;
            return this;
        }

        MediaOptions build() {
            return new MediaOptions(this);
        }
    }
}
