package com.tts.android.aopmonitor.tools.crash;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.Application;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;
import android.os.Looper;
import android.view.Gravity;
import android.widget.Toast;

import com.blankj.utilcode.util.CacheDiskUtils;
import com.google.gson.Gson;
import com.tts.android.aopmonitor.RecordService;
import com.tts.android.aopmonitor.bean.ViewClickAction;
import com.tts.android.aopmonitor.net.bean.RecordRequestBean;
import com.tts.android.aopmonitor.tools.CtwUtils;
import com.tts.android.aopmonitor.tools.LogUtils;
import com.tts.android.aopmonitor.tools.TimeUtils;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Field;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 当程序发生Uncaught异常的时候,由该类来接管程序,并记录发送错误报告.
 * 需要在Application中注册，为了要在程序启动器就监控整个程序。
 * @author shijianting
 * 2019/1/12 8:38 AM
 */
public class CrashHandler implements Thread.UncaughtExceptionHandler {

    public static final String TAG = "CrashHandler";

    /**
     * 自定义toast
     */
    private static Toast mCustomToast;

    /**
     * 提示文字
     */
    private static String mCrashTip = "很抱歉,程序出现异常,即将退出.";
    private static String mCrashTip2 = "很抱歉,程序出现异常,即将重启.";

    //系统默认的UncaughtException处理类
    private Thread.UncaughtExceptionHandler mDefaultHandler;

    //CrashHandler实例
    private static CrashHandler mCrashHandler;

    //程序的App对象
    public Application mApplication;

    //生命周期监听
    MyActivityLifecycleCallbacks mMyActivityLifecycleCallbacks = new MyActivityLifecycleCallbacks();

    //用来存储设备信息和异常信息
    private Map<String, String> infos = new HashMap();

    //用于格式化日期,作为日志文件名的一部分
    private DateFormat formatter = new SimpleDateFormat("yyyy-MM-dd-HH-mm-ss");

    //是否是Debug模式
    private boolean mIsDebug;

    //是否展示toast
    private boolean mIsShowToast;

    //是否重启APP
    private boolean mIsRestartApp;

    //重启APP时间
    private long mRestartTime;

    //重启后的第一个Activity class文件
    private Class mClassOfFirstActivity;

    //是否已经toast
    private boolean hasToast;

    /**
     * 私有构造函数
     */
    public CrashHandler() {

    }

    /**
     * 获取CrashHandler实例 ,单例模式
     *
     * @return
     * @since V1.0
     */
    public static CrashHandler getInstance() {
        if (mCrashHandler == null) {
            mCrashHandler = new CrashHandler();
        }
        return mCrashHandler;
    }

    /**
     * 初始化crash
     * @param application
     * @param isDebug 调试，开启后会发送崩溃日志信息
     * @param isShowToast 是否展示toast
     * @param isRestartApp 是否重启
     * @param restartTime 重启时间间隔
     * @param classOfFirstActivity 要重启的activity
     */
    public void init(Application application, boolean isDebug, boolean isShowToast, boolean isRestartApp, long restartTime, Class classOfFirstActivity) {
        mIsShowToast = isShowToast;
        mIsRestartApp = isRestartApp;
        mRestartTime = restartTime;
        mClassOfFirstActivity = classOfFirstActivity;
        initCrashHandler(application, isDebug);
    }

    public void init(Application application, boolean isDebug) {
        initCrashHandler(application, isDebug);
    }

    /**
     * 初始化
     *
     * @since V1.0
     */
    @TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
    private void initCrashHandler(Application application, boolean isDebug) {
        mIsDebug = isDebug;
        mApplication = application;
        mApplication.registerActivityLifecycleCallbacks(mMyActivityLifecycleCallbacks);
        // 设置该CrashHandler为程序的默认处理器
        Thread.setDefaultUncaughtExceptionHandler(this);
        // 获取系统默认的UncaughtException处理器
        mDefaultHandler = Thread.getDefaultUncaughtExceptionHandler();

    }

    /**
     * 当UncaughtException发生时会转入该函数来处理
     */
    @Override
    public void uncaughtException(Thread thread, Throwable ex) {
        boolean isHandle = handleException(ex);
        if (!isHandle && mDefaultHandler != null) {
            // 如果我们没有处理则让系统默认的异常处理器来处理
            mDefaultHandler.uncaughtException(thread, ex);
        } else {
            try {
                //给Toast留出时间
                Thread.sleep(2800);
            } catch (InterruptedException e) {
                LogUtils.loge("uncaughtException() InterruptedException:" + e);
            }

            if (mIsRestartApp) {
                //利用系统时钟进行重启任务
                AlarmManager mgr = (AlarmManager) mApplication.getSystemService(Context.ALARM_SERVICE);
                try {
                    Intent intent = new Intent(mApplication, mClassOfFirstActivity);
                    intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
                    PendingIntent restartIntent = PendingIntent.getActivity(mApplication, 0, intent, PendingIntent.FLAG_ONE_SHOT);
                    // x秒钟后重启应用
                    mgr.set(AlarmManager.RTC, System.currentTimeMillis() + mRestartTime, restartIntent);
                } catch (Exception e) {
                    LogUtils.loge("first class error:" + e);
                }
            }

            mMyActivityLifecycleCallbacks.removeAllActivities();
            android.os.Process.killProcess(android.os.Process.myPid());
            System.exit(1);
            System.gc();

        }
    }

    /**
     * 自定义错误处理,收集错误信息 发送错误报告等操作均在此完成.
     *
     * @param ex
     * @return true:如果处理了该异常信息;否则返回false.
     */
    private boolean handleException(Throwable ex) {

        if (mIsShowToast) {

            if (!hasToast) {
                new Thread(new Runnable() {
                    @Override
                    public void run() {
                        try {
                            Looper.prepare();
                            Toast toast;
                            if (mCustomToast == null) {
                                if (mIsRestartApp) {
                                    toast = Toast.makeText(mApplication, mCrashTip2, Toast.LENGTH_LONG);
                                }
                                else {
                                    toast = Toast.makeText(mApplication, mCrashTip, Toast.LENGTH_LONG);
                                }
                                toast.setGravity(Gravity.CENTER, 0, 0);
                            } else {
                                toast = mCustomToast;
                            }
                            toast.show();
                            Looper.loop();
                            hasToast = true;
                        } catch (Exception e) {
                            LogUtils.loge("handleException Toast error" + e);
                        }

                    }
                }).start();
            }
        }


        if (ex == null) {
            return false;
        }

        if (mIsDebug) {
            LogUtils.logd("save crash info in disk, after restart app,resend");
            ViewClickAction action = new ViewClickAction();
            action.setWidgetId(ex.toString());
            action.setPageId("crash");
            action.setExtend(getCrashInfo(ex));
            action.setTimeStamp(TimeUtils.getTimeStamp());
            List<ViewClickAction> actions = new ArrayList<>();
            actions.add(action);
            RecordRequestBean recordRequestBean = new RecordRequestBean();
            recordRequestBean = RecordService.initBaseRecordBean(recordRequestBean);
            recordRequestBean.setActions(actions);
            //对象转String，加入基础数据
            String actionStr = new Gson().toJson(CtwUtils.initCtwRecordBean(recordRequestBean));
            CacheDiskUtils.getInstance().put("crash", actionStr);
        }

        return true;
    }




    /**
     * 得到程序崩溃的详细信息
     */
    public String getCrashInfo(Throwable ex) {
        Writer result = new StringWriter();
        PrintWriter printWriter = new PrintWriter(result);
        ex.setStackTrace(ex.getStackTrace());
        ex.printStackTrace(printWriter);
        printWriter.close();
        return result.toString();
    }

}