package com.tts.android.aopmonitor.aspect;

import android.app.Fragment;
import android.content.Context;

import com.blankj.utilcode.util.DeviceUtils;
import com.tts.android.aopmonitor.annotation.NeedPermission;
import com.tts.android.aopmonitor.annotation.PermissionDenied;
import com.tts.android.aopmonitor.tools.LogUtils;
import com.yanzhenjie.permission.Action;
import com.yanzhenjie.permission.AndPermission;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.List;

/**
 * 权限验证AOP
 * @author shijianting
 * 2018/11/27 9:00 AM
 */
@Aspect
public class NeedPermissionAspect {

    private static final int ANDROID_M = 23;

    @Pointcut("execution(@com.tts.android.aopmonitor.annotation.NeedPermission * *(..)) && @annotation(needPermission)")
    public void needPermission(NeedPermission needPermission){}

    @Around("needPermission(needPermission)")
    public void checkPermission(final ProceedingJoinPoint point, NeedPermission needPermission) {
        Context context = null;
        final Object object = point.getThis();
        if (object instanceof Context) {
            context = (Context) object;
        } else if (object instanceof Fragment) {
            context = ((Fragment) object).getActivity();
        } else if (object instanceof android.support.v4.app.Fragment) {
            context = ((android.support.v4.app.Fragment) object).getActivity();
        }
        if (context == null || needPermission == null) {
            LogUtils.loge("the method is not belong to a activity or fragment, " +
                    "or NeedPermission annotation not found");
            try {
                point.proceed();
            } catch (Throwable throwable) {
                throwable.printStackTrace();
            }
            return;
        }
        /**
         * 小于23的版本不需要动态权限验证
         */
        if (DeviceUtils.getSDKVersionCode() < ANDROID_M) {
            try {
                point.proceed();
            } catch (Throwable throwable) {
                throwable.printStackTrace();
            }
            return;
        }
        //开始检查并获取权限
        AndPermission.with(context)
                .runtime()
                .permission(needPermission.value())
                .onGranted(new Action<List<String>>() {
                    @Override
                    public void onAction(List<String> data) {
                        try {
                            point.proceed();
                        } catch (Throwable throwable) {
                            throwable.printStackTrace();
                        }
                    }
                })
                .onDenied(new Action<List<String>>() {
                    @Override
                    public void onAction(List<String> data) {
                        Class<?> cls = object.getClass();
                        Method[] methods = cls.getDeclaredMethods();
                        if (methods == null || methods.length == 0) {
                            return;
                        }
                        for (Method method : methods) {
                            //过滤不含自定义注解PermissionDenied的方法
                            boolean isHasAnnotation = method.isAnnotationPresent(PermissionDenied.class);
                            if (isHasAnnotation) {
                                method.setAccessible(true);
                                //获取方法参数类型
                                Class<?>[] types = method.getParameterTypes();
                                //获取方法上的注解
                                PermissionDenied aInfo = method.getAnnotation(PermissionDenied.class);
                                if (aInfo == null) {
                                    return;
                                }
                                try {
                                    if (null != types && types.length == 1 ) {
                                        method.invoke(object, data);
                                    }
                                    else {
                                        method.invoke(object);
                                    }
                                } catch (IllegalAccessException e) {
                                    e.printStackTrace();
                                } catch (InvocationTargetException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                    }
                })
                .start();
    }
}
