package com.testvagrant.optimuscloud.remote;

import com.testvagrant.optimuscloud.IOptimusCloudDriver;
import com.testvagrant.optimuscloud.clients.AkiraClient;
import com.testvagrant.optimuscloud.driver.DriverCreatorFactory;
import com.testvagrant.optimuscloud.driver.SessionManager;
import com.testvagrant.optimuscloud.entities.MandatoryCaps;
import com.testvagrant.optimuscloud.entities.MobileDriverDetails;
import com.testvagrant.optimuscloud.entities.SessionDetails;
import com.testvagrant.optimuscloud.entities.SessionState;
import com.testvagrant.optimuscloud.exceptions.EmptySessionUrlException;
import com.testvagrant.optimuscloud.messaging.SessionQueueReader;
import com.testvagrant.optimuscloud.utils.CapToManCapConverter;
import com.testvagrant.optimuscloud.utils.RetryEngine;
import com.testvagrant.optimuscloud.utils.TestFeedToDesiredCapConverter;
import io.appium.java_client.MobileDriver;
import org.awaitility.Duration;
import org.openqa.selenium.NoSuchSessionException;
import org.openqa.selenium.SessionNotCreatedException;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.springframework.retry.RetryCallback;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;

import static org.awaitility.Awaitility.await;

public class OptimusCloudDriver implements IOptimusCloudDriver {

    private AkiraClient akiraClient;

    public OptimusCloudDriver() {
        this.akiraClient = new AkiraClient();
    }

    @Override
    public MobileDriverDetails createDriver(String buildNo, DesiredCapabilities desiredCapabilities) throws Exception {
        System.out.printf("Creating driver for build %s with capabilities %s",buildNo,desiredCapabilities.toJson());
        MandatoryCaps mandatoryCaps = CapToManCapConverter.convert(buildNo, desiredCapabilities);
        SessionDetails sessionDetails = getSessionDetailsFromCloud(mandatoryCaps);
        System.out.printf("Session details %s",sessionDetails.toString());
        if(akiraClient.isWfsmbox(buildNo,mandatoryCaps.getPlatformName())) {
            sessionDetails = getSessionDetailsIfSessionIsUnAssigned(buildNo, sessionDetails);
        }
        return initDriver(desiredCapabilities,sessionDetails);
    }

    @Override
    public MobileDriverDetails createDriver(DesiredCapabilities desiredCapabilities) {
        System.out.printf("Creating driver with capabilities %s",desiredCapabilities.toJson());
        MandatoryCaps mandatoryCaps = CapToManCapConverter.convert(desiredCapabilities);
        SessionDetails sessionDetails = getSessionDetailsFromCloud(mandatoryCaps);
        System.out.printf("Session details %s",sessionDetails.toString());
        return initDriver(desiredCapabilities,sessionDetails);
    }

    private SessionDetails getSessionDetailsFromCloud(MandatoryCaps mandatoryCaps) {
        AtomicReference<SessionDetails> sessionDetails = new AtomicReference<>(new SessionDetails());
        RetryCallback retryCallback = retry -> {
            try {
                sessionDetails.set(akiraClient.getSessionDetails(mandatoryCaps));
                System.out.println("Mandatory Caps "+mandatoryCaps.toString());
                System.out.println("Session Details "+sessionDetails.get().toString());
                if(sessionDetails.get().getSessionUrl()==null) throw new EmptySessionUrlException();
                SessionManager sessionManager = new SessionManager();
                boolean sessionUp = sessionManager.isSessionUp(sessionDetails.get().getSessionUrl());
                if(!sessionUp) {
                    akiraClient.terminateSession(sessionDetails.get().getSessionUrl());
                    throw new EmptySessionUrlException();
                }
                return sessionDetails;
            } catch (Exception e) {
                System.out.println("Retrying session");
                throw e;
            }

        };

        RetryEngine retryEngine = new RetryEngine(EmptySessionUrlException.class);
        retryEngine.execute(retryCallback);
        return sessionDetails.get();
    }

    private MobileDriverDetails initDriver(DesiredCapabilities desiredCapabilities, SessionDetails sessionDetails) {
        try {
            return getDriverDetails(desiredCapabilities, sessionDetails);
        } catch (SessionNotCreatedException | NoSuchSessionException e) {
            akiraClient.terminateSession(sessionDetails.getSessionUrl());
            throw e;
        } catch (MalformedURLException e) {
            e.printStackTrace();
        }
        throw new RuntimeException("Cannot initialize driver");
    }

    private MobileDriverDetails getDriverDetails(DesiredCapabilities desiredCapabilities, SessionDetails sessionDetails) throws MalformedURLException {
        sessionDetails.getDesiredCapabilities().entrySet().stream()
                .forEach(entry -> desiredCapabilities.setCapability(entry.getKey(), entry.getValue()));
        final MobileDriver[] mobileDriver = new MobileDriver[1];
        RetryCallback retryCallback = doWithRetry -> {
            mobileDriver[0] = DriverCreatorFactory.getInstance(sessionDetails.getSessionUrl(), desiredCapabilities).getDriver();
            return mobileDriver[0];
        };
        RetryEngine retryEngine = new RetryEngine(SessionNotCreatedException.class);
        retryEngine.execute(retryCallback);
        return getMobileDriverDetails(sessionDetails, desiredCapabilities, mobileDriver[0]);
    }

    public synchronized SessionDetails getSessionDetailsIfSessionIsUnAssigned(String buildNo, SessionDetails sessionDetails) throws IOException, TimeoutException {
        String sessionUrl  = "";
        if(sessionDetails.getSessionUrl()==null) {
            sessionUrl = waitTillSessionIsAvailable(buildNo);
        } else {
            return sessionDetails;
        }
        LinkedHashMap<String, String> sessionCapabilities = akiraClient.getSessionCapabilities(sessionUrl);
        SessionDetails sessionDetailsUpdated = new SessionDetails();
        sessionDetails.setSessionUrl(sessionUrl);
        sessionDetails.setDesiredCapabilities(sessionCapabilities);
        sessionDetails.setReserved(true);
        sessionDetails.setSessionState(SessionState.AVAILABLE);
        return sessionDetailsUpdated;
    }

    private synchronized String waitTillSessionIsAvailable(String buildNo) throws IOException, TimeoutException {
        String sessionUrl = "";
        SessionQueueReader sessionQueueReader = new SessionQueueReader();
        boolean isMessageRead = false;
        while(!isMessageRead) {
            await()
                    .pollInterval(Duration.FIVE_SECONDS)
                    .atMost(Duration.FIVE_MINUTES.multiply(5))
                    .until(() -> sessionQueueReader.queueHasAMessage(buildNo));
            sessionUrl = sessionQueueReader.consumeMessage(buildNo);
            if(sessionUrl!=null || !sessionUrl.isEmpty()) {
                isMessageRead=true;
                akiraClient.engageSession(sessionUrl);
            }
        }
        return sessionUrl;
    }

    @Override
    public MobileDriverDetails createDriver(String buildNo, Map<String, String> desiredCapabilities) throws Exception {
        DesiredCapabilities desiredCapabilities1 = getDesiredCapabilities(desiredCapabilities);
        return createDriver(buildNo,desiredCapabilities1);
    }

    @Override
    public MobileDriverDetails createDriver(Map<String, String> desiredCapabilities) {
        return null;
    }

    @Override
    public MobileDriverDetails createDriver(String buildNo, String testFeed) throws Exception {
        List<DesiredCapabilities> desiredCapabilitiesList = new TestFeedToDesiredCapConverter(testFeed).convert();
        return createDriver(buildNo,desiredCapabilitiesList.get(0));
    }

    @Override
    public MobileDriverDetails createDriver(String buildNo, String testFeed, String appPath) throws Exception {
        List<DesiredCapabilities> desiredCapabilitiesList = new TestFeedToDesiredCapConverter(testFeed).convert();
        DesiredCapabilities desiredCapabilities = desiredCapabilitiesList.get(0);
        desiredCapabilities.setCapability("app",appPath);
        return createDriver(buildNo,desiredCapabilities);
    }

    private DesiredCapabilities getDesiredCapabilities(Map<String, String> desiredCapabilities) {
        DesiredCapabilities desiredCapabilities1 = new DesiredCapabilities();
        desiredCapabilities.entrySet().stream().forEach(entry -> {
            desiredCapabilities1.setCapability(entry.getKey(),entry.getValue());
        });
        return desiredCapabilities1;
    }

    private MobileDriverDetails getMobileDriverDetails(SessionDetails sessionDetails, DesiredCapabilities desiredCapabilities, MobileDriver mobileDriver) {
        MobileDriverDetails mobileDriverDetails = new MobileDriverDetails();
        mobileDriverDetails.setSessionUrl(sessionDetails.getSessionUrl());
        mobileDriverDetails.setMobileDriver(mobileDriver);
        mobileDriverDetails.setUdid(sessionDetails.getDesiredCapabilities().get("udid"));
        mobileDriverDetails.setDesiredCapabilities(desiredCapabilities);
        return mobileDriverDetails;
    }

}
