package com.tenqube.visual_third.manager;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.support.v4.app.TaskStackBuilder;

import com.tenqube.visual_third.Constants;
import com.tenqube.visual_third.R;
import com.tenqube.visual_third.entity.VisualNotification;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.util.Utils;

import java.util.Calendar;

import static com.tenqube.visual_third.manager.PrefManager.DEEP_LINK;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_COLOR;
import static com.tenqube.visual_third.manager.PrefManager.NOTI_ICON_RES;
import static com.tenqube.visual_third.ui.VisualWebActivity.ARG_PATH;
import static com.tenqube.visual_third.util.Utils.DECIMAL_FORMAT;
import static com.tenqube.visual_third.util.Utils.getYMD;
import static com.tenqube.visual_third.util.Utils.threeComma;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualNotificationManager {

    public static final String TAG = VisualNotificationManager.class.getSimpleName();
    public static final int NOTI_ID = 1080;
    public static final String NOTI_CHANNEL = "VISUAL_CHANNEL";
    private VisualRepository repository;
    private PrefManager prefManager;
    private Context context;
    private static VisualNotificationManager mInstance = null;

    public static VisualNotificationManager getInstance(Context ctx) {
        synchronized (VisualNotificationManager.class) {
            if (mInstance == null) {
                mInstance = new VisualNotificationManager(ctx.getApplicationContext());
            }
        }
        return mInstance;
    }

    private VisualNotificationManager(Context context) {
        this.context = context;
        repository = VisualRepository.getInstance(context);
        prefManager = PrefManager.getInstance(context);
    }

    public void showNotification(VisualNotification notification) {

        String[] dates = getDateRanges(notification); // 날짜 범위

        double sum = repository.getSum(dates[0], dates[1]);
        notifyNotification(notification.getTicker().contains("%") ?
                        String.format(notification.getTicker(), threeComma(sum))
                :
                notification.getTicker(),
                notification.getTitle(),
                notification.getContent().contains("%") ?
                        String.format(notification.getContent(), threeComma(sum))
                        :
                        notification.getContent(),
                getStackBuilder(getPath(notification.getName())));

    }

    private String getPath(String name) {
        switch (name) {
            case Constants.NotificationType.MONTHLY:

                return "&monthly=true";

            case Constants.NotificationType.WEEKLY_FRI:
            case Constants.NotificationType.WEEKLY_MON:

                return "&weekly=true ";

            default:
                return "&daily=true";
        }
    }

    private String[] getDateRanges(VisualNotification notification) {
        String fromAt = "";
        String toAt = "";

        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.HOUR, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);

        long currentTime = calendar.getTimeInMillis();

        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DATE);

        int fromYear, fromMonth;

        switch (notification.getName()) {
            case "monthly":
                fromYear = year;
                fromMonth = month - 1;

                if(fromMonth == 0) {
                    fromYear = year - 1;
                    fromMonth = 12;
                }
                fromAt = fromYear + "-" + DECIMAL_FORMAT.format(fromMonth) + "-01";// 지난달 1일 'ymd'
                toAt = year + "-" + DECIMAL_FORMAT.format(month) + "-01";// 지난달 1일 'ymd'
                break;

            case "weekly_mon":
                // 지난 주 월 ~ 이번주 월

                calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONTH);
                if(currentTime < calendar.getTimeInMillis()) {
                    calendar.add(Calendar.WEEK_OF_YEAR, -2);
                } else if (currentTime == calendar.getTimeInMillis()) {
                    calendar.add(Calendar.WEEK_OF_YEAR, -1);
                }

                fromAt = getYMD(calendar); // 지난주 월 'ymd'
                calendar.add(Calendar.WEEK_OF_YEAR, 1);
                toAt = getYMD(calendar); // 이번주 월 'ymd'

                break;

            case "weekly_fri":
                calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONTH);
                if(currentTime < calendar.getTimeInMillis()) {
                    calendar.add(Calendar.WEEK_OF_YEAR, -1);
                }
                fromAt = getYMD(calendar); // 지난주 월 'ymd'

                calendar.add(Calendar.WEEK_OF_YEAR, 1);
                toAt = getYMD(calendar); // 이번주 월 'ymd'

                break;

            case "daily":
                fromAt = Utils.getYMD(calendar);
                calendar.add(Calendar.DATE, 1);
                toAt = Utils.getYMD(calendar);
                break;
        }

        LOGI(TAG, "type: " + notification.getName() + "\nfromAt:" + fromAt + "toAt" + toAt, mIsDebug);
        return new String []{fromAt, toAt};
    }

    private void createNotificationChannel() {
        // Create the NotificationChannel, but only on API 26+ because
        // the NotificationChannel class is new and not in the support library
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            int importance = NotificationManager.IMPORTANCE_DEFAULT;
            NotificationChannel channel = new NotificationChannel((prefManager.loadStringValue(NOTI_CHANNEL, NOTI_CHANNEL)), "굿리치", importance);
            // Register the channel with the system; you can't change the importance
            // or other notification behaviors after this
            NotificationManager notificationManager = context.getSystemService(NotificationManager.class);
            if(notificationManager != null) notificationManager.createNotificationChannel(channel);
        }
    }

    private void notifyNotification(String ticker,
                                    String title,
                                    String content,
                                    TaskStackBuilder stackBuilder) {

        PendingIntent pi = stackBuilder.getPendingIntent(NOTI_ID, PendingIntent.FLAG_UPDATE_CURRENT);

        createNotificationChannel();

        int color = prefManager.loadIntValue(NOTI_COLOR, 0);

        NotificationCompat.Builder mBuilder =
                new NotificationCompat.Builder(context, prefManager.loadStringValue(NOTI_CHANNEL, NOTI_CHANNEL))
                        .setSmallIcon(prefManager.loadIntValue(NOTI_ICON_RES, R.drawable.ic_withdraw_shopping))
                        .setContentTitle(title)
                        .setContentText(content)
                        .setTicker(ticker)
                        .setAutoCancel(true)
                        .setContentIntent(pi)
                        .setStyle(new NotificationCompat.BigTextStyle()
                                .bigText(content));

        if(color != 0) mBuilder.setColor(color);

        NotificationManagerCompat notificationManager = NotificationManagerCompat.from(context);
        notificationManager.notify(NOTI_ID, mBuilder.build());

    }

    private TaskStackBuilder getStackBuilder(String path) {

        Intent intent = new Intent(Intent.ACTION_VIEW);
        intent.setData(Uri.parse(prefManager.loadStringValue(DEEP_LINK, "")));
        intent.putExtra(ARG_PATH, path);
        TaskStackBuilder stackBuilder = TaskStackBuilder.create(context);
        stackBuilder.addNextIntent(intent);
        return stackBuilder;
    }

}
