package com.tenqube.visual_third;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v7.app.AppCompatActivity;
import android.text.TextUtils;

import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.VisualActivityFragment;
import com.tenqube.visual_third.util.ActivityUtils;

import tenqube.parser.core.ParserService;

import static com.tenqube.visual_third.Constants.DEV;
import static com.tenqube.visual_third.Constants.PROD;
import static com.tenqube.visual_third.manager.AnswerManager.initFabric;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.Validator.notZero;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();
    private Context context;

    private ParserService parserService;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;

    private boolean isDoingStartVisual;

    public VisualServiceImpl(@NonNull Context context, String apiKey, String qualifier) throws ParameterException {

        if(TextUtils.isEmpty(apiKey)) throw new ParameterException("apiKey is null");

        if(TextUtils.isEmpty(qualifier) || !(Constants.DEV.equals(qualifier) || Constants.PROD.equals(qualifier)))
            throw new ParameterException("check qualifier value (" + DEV + ", " + PROD + ")");

        this.context = context;
        initFabric(context);
        alarmManager = VisualAlarmManager.getInstance(context);
        parserService = ParserService.getInstance(context);
        repository = VisualRepository.getInstance(context);
        repository.saveSDKInfo(apiKey, qualifier);
    }

    @Override
    public void startVisual(@NonNull final AppCompatActivity activity,
                            final int containerId,
                            final boolean isBackStack,
                            @NonNull String uid,
                            @NonNull String path) throws ParameterException, SecurityException {
        LOGI(TAG, "웹 가계부 호출", mIsDebug);

        try {
            if(isDoingStartVisual) return;

            isDoingStartVisual = true;
            notNull(activity);
            notNull(path);
            notZero(containerId);
            checkSMSPermission();

            AnswerManager.onKeyMetric(new LogRequest("startVisual"));
            ActivityUtils.addFragmentToActivity(activity.getSupportFragmentManager(),
                    VisualActivityFragment.newInstance(uid, path),
                    containerId,
                    VisualActivityFragment.TAG,
                    isBackStack);

        } catch (ParameterException | SecurityException e) {
            throw e;
        } finally {
            isDoingStartVisual = false;
        }

    }

    @Override
    public void setDeepLink(String url) {
        repository.setDeepLink(url);
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.shouldShowTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    private void checkSMSPermission() throws SecurityException {
        if (!repository.isExecutedBulk() &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.READ_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.RECEIVE_SMS) != PackageManager.PERMISSION_GRANTED &&
                ActivityCompat.checkSelfPermission(context, Manifest.permission.RECEIVE_MMS) != PackageManager.PERMISSION_GRANTED) {

            throw new SecurityException("Please grant permissions " +
                    "(Manifest.permission.READ_SMS, " +
                    "Manifest.permission.RECEIVE_SMS, " +
                    "Manifest.permission.RECEIVE_MMS)");
        }
    }

    @Override
    public void setDebugMode(boolean isDebug) {
        LOGI(TAG, "setDebugMode", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.setDebugMode(isDebug);
        }
    }

    @Override
    public void initSDK() {
        LOGI(TAG, "initSDK", mIsDebug);

        if(isActive() && parserService != null) {
            parserService.initDb();
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);
    }

    @Override
    public void settingNotification(int smallIcon) {
        repository.settingNotification(smallIcon);
    }

    @Override
    public void setMonthlyNoti(boolean isActive) {
        repository.setActiveNoti("monthly", isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveMonthlyNoti() {
        return repository.isActiveNoti("monthly");
    }

    @Override
    public void setWeeklyNoti(boolean isActive) {
        repository.setActiveNoti("weekly", isActive);
        alarmManager.setAlarms();

    }

    @Override
    public boolean isActiveWeeklyNoti() {
        return repository.isActiveNoti("weekly");
    }

    @Override
    public void setDailyNoti(boolean isActive) {
        repository.setActiveNoti("daily", isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveDailyNoti() {
        return repository.isActiveNoti("daily");
    }

    @Override
    public void setUserName(String userName) {
        repository.setUserName(userName);
    }

    private boolean isActive() {
        return context != null;
    }

}
