package com.tenqube.visual_third;

import android.app.Activity;
import android.content.Context;
import androidx.annotation.NonNull;

import com.tenqube.visual_third.analysis.AnalysisHelper;
import com.tenqube.visual_third.analysis.AnalysisService;
import com.tenqube.visual_third.analysis.AnalysisServiceImpl;
import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.analysis.Analysis;
import com.tenqube.visual_third.model.analysis.Transaction;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.model.js.TransactionRequest;
import com.tenqube.visual_third.model.service.ApiInfo;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.AppExecutors;
import com.tenqube.visual_third.util.Validator;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;

import static com.tenqube.visual_third.util.Utils.sumBy;
import static com.tenqube.visual_third.util.Validator.notNull;
import static com.tenqube.visual_third.util.Validator.notZero;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private Context context;
    private AnalysisService service;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;
    private AppExecutors appExecutors = new AppExecutors();

    public static boolean logger = false;

    public static class Builder {

        Context context;
        String apiKey;
        ApiInfo apiInfo;
        String qualifier;
        String authority;

        int smallIcon;
        String channel;
        int color;

        boolean logger;

        public Builder withContext(final Context context) {
            this.context = context;
            return this;
        }

        public Builder apiKey(final String apiKey) {
            this.apiKey = apiKey;
            return this;
        }

        public Builder apiInfo(final ApiInfo apiInfo) {
            this.apiInfo = apiInfo;

            return this;
        }

        public Builder qualifier(final String qualifier) {
            this.qualifier = qualifier;
            return this;
        }

        public Builder authority(final String authority) {
            this.authority = authority;
            return this;
        }

        public Builder notification(int smallIcon, @NonNull String channel, int color) {

            this.smallIcon = smallIcon;
            this.channel = channel;
            this.color = color;

            return this;
        }

        public Builder logger(boolean logger) {
            this.logger = logger;
            return this;
        }

        public VisualService build() throws ParameterException {

            return new VisualServiceImpl(this);
        }
    }

    private VisualServiceImpl(final Builder builder) throws ParameterException {

        Validator.notNull(builder.context);

        Validator.isNotEmpty(builder.apiKey);

        Validator.isNotEmpty(builder.authority);

        notZero(builder.smallIcon);
        notNull(builder.channel);
        notNull(builder.apiInfo);

        Validator.in(builder.qualifier, Constants.DEV, Constants.PROD);

        this.context = builder.context;

        logger = builder.logger;

        repository = RepositoryHolder.getInstance(context).getVisualRepository();

        repository.saveSDKInfo(builder.apiKey, builder.qualifier, builder.authority, builder.apiInfo);

        repository.settingNotification(builder.smallIcon, builder.channel, builder.color);

        alarmManager = VisualAlarmManager.getInstance(context);

        service = new AnalysisServiceImpl(context, builder.qualifier);

    }

    @Override
    public void initialize(@NonNull String serviceId, @NonNull VisualUserInfo userInfo, final @NonNull OnResultListener resultListener
    ) throws ParameterException {
        Validator.isNotEmpty(serviceId);
        Validator.notNull(userInfo);
        repository.signUp(serviceId, userInfo,  new OnResultListener() {
            @Override
            public void onResult(int signUpResult, String msg) {
                resultListener.onResult(signUpResult, msg);
            }
        });
    }

    @Override
    public void startVisual(final @NonNull Activity activity,
                            final @NonNull String serviceId,
                            final @NonNull VisualUserInfo visualUserInfo,
                            final @NonNull VisualCallback visualCallback,
                            final @NonNull OnResultListener resultListener
                            ) throws ParameterException {

        Validator.notNull(activity);
        Validator.isNotEmpty(serviceId);
        Validator.notNull(resultListener);
        Validator.notNull(visualUserInfo);
        Validator.notNull(visualCallback);


        repository.setVisualCallback(visualCallback)
        ;
        repository.signUp(serviceId, visualUserInfo,  new OnResultListener() {
            @Override
            public void onResult(int signUpResult, String msg) {
                try {
                    //가입 성공시에만 isAgree 값에 맞게 설정정보 업데이트 처리
                    switch (signUpResult) {
                        case Constants.SignupResult.SUCCESS:
                        case Constants.SignupResult.ALREADY_JOINED:
                            VisualWebActivity.startActivity(activity, "");
                            break;
                    }
                } catch (AuthException e) {
                    signUpResult = Constants.SignupResult.SERVER_ERROR;
                    msg = e.toString();
                } finally {
                    resultListener.onResult(signUpResult, msg);
                }
            }
        });
    }

    @Override
    public String getNotificationServiceName() {
        return "com.tenqube.visual_third.parser.catcher.NotiCatcher";
    }

    @Override
    public void signOut(Callback<Boolean> callback) {
        repository.signOut(new Callback<Boolean>() {
            @Override
            public void onDataLoaded(Boolean value) {
                alarmManager.setAlarms();
            }
        });
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.isActiveTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    @Override
    public void setReportAlarm(Constants.ReportAlarmType type, boolean isActive) {
        repository.setActiveNoti(type.name().toLowerCase(), isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveReportAlarm(Constants.ReportAlarmType type) {
        return repository.isActiveNoti(type.name().toLowerCase());
    }

    @Override
    public void setReportTest(Constants.ReportAlarmType type, int second) {
        alarmManager.setReportTest(type.name().toLowerCase(), second);
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);
        if(!enabled) {
            repository.setNotiAll(false);
            alarmManager.setAlarms();
        }
    }

    @Override
    public boolean isJoined() {
        return repository.isJoined();
    }

    @Override
    public void getVisualSummary(final Callback<VisualSummary> callback) {

        // 초기화 체크
        appExecutors.diskIO().execute(new Runnable() {
            @Override
            public void run() {
                if (!repository.isJoined()) { // 가입되지 않은 경우 null 을 리턴합니다.
                    appExecutors.mainThread().execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onDataLoaded(null);
                        }
                    });

                } else {

                    Calendar calendar = Calendar.getInstance();
                    int year = calendar.get(Calendar.YEAR);
                    int month = calendar.get(Calendar.MONTH) + 1;

                    TransactionRequest request = new TransactionRequest(year, month, 3, true, "");
                    request.setDwType(tenqube.parser.constants.Constants.DWType.WITHDRAW.ordinal());
                    request.setExceptType(0);

                    ArrayList<Transaction> transactions = repository.loadAnalysisTransactions(request);

                    // empty 허용하지 않습니다.
                    final ArrayList<Analysis> monthly = service.loadAnalysisList(transactions, repository.getStartDay(), true);
                    final VisualSummary visualSummary;

                    double sum = getCurrentMonthSum(year, month);
                    if (monthly.isEmpty()) {
                        visualSummary = new VisualSummary(sum, String.format("이번 달 지출은 %1$s 입니다.", AnalysisHelper.getLv0Currency(sum)));
                    } else {
                        Collections.shuffle(monthly);
                        Analysis analysis = monthly.get(0);
                        visualSummary = new VisualSummary(sum, analysis.getlContent());
                    }

                    // 랜덤으로 하나 뽑아서 visual summary 에 할당합니다.
                    appExecutors.mainThread().execute(new Runnable() {
                        @Override
                        public void run() {
                            callback.onDataLoaded(visualSummary);
                        }
                    });
                }
            }
        });
    }

    private double getCurrentMonthSum(int year, int month) {

        TransactionRequest currentRequest = new TransactionRequest(year, month, 0, true, "");
        currentRequest.setDwType(tenqube.parser.constants.Constants.DWType.WITHDRAW.ordinal());
        currentRequest.setExceptType(0);
        ArrayList<Transaction> currentMonthTransactions = repository.loadAnalysisTransactions(currentRequest);

        return sumBy(currentMonthTransactions);
    }

    @Override
    public void setAppNoti(boolean isActive) {
        repository.setAppNoti(isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveAppNoti() {
        return repository.isAppNoti();
    }

    private boolean isActive() {
        return context != null;
    }
}
