package com.tenqube.visual_third.parser.loader;


import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;
import android.os.Message;

import com.tenqube.visual_third.Callback;
import com.tenqube.visual_third.manager.PrefManager;
import com.tenqube.visual_third.model.api.SearchCompanyRequest;
import com.tenqube.visual_third.model.api.SearchCompanyResponse;
import com.tenqube.visual_third.model.parser.SyncTransaction;
import com.tenqube.visual_third.util.Mapper;
import com.tenqube.visual_third.util.Utils;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import tenqube.parser.BulkSmsAdapter;
import tenqube.parser.OnNetworkResultListener;
import tenqube.parser.constants.Constants;
import tenqube.parser.model.SMS;
import tenqube.parser.model.Transaction;

import static com.tenqube.visual_third.VisualServiceImpl.logger;
import static com.tenqube.visual_third.ui.VisualWebActivity.FINISH;
import static com.tenqube.visual_third.ui.VisualWebActivity.PROGRESS;
import static com.tenqube.visual_third.util.Utils.getConvertedDate;
import static com.tenqube.visual_third.util.Utils.isEmpty;
import static tenqube.parser.core.ParserService.mIsDebug;
import static tenqube.parser.util.LogUtil.LOGI;

public class BulkLoader extends BaseLoader implements BulkSmsAdapter {

    private static final String TAG = BulkLoader.class.getSimpleName();
    private static final int BULK_MONTH = -3;
    private Cursor mCursor;
    private final Handler mHandler;
    private final OnSmsLoadFinished onSmsLoadFinished;

    private long lastSmsDate;

    /**
     * @param context 컨텍스트
     * @param handler 진행상태 메세지 전송 handler
     */
    public BulkLoader(Context context,
                      Handler handler,
                      OnSmsLoadFinished onSmsLoadFinished) {
        super(context);
        this.mHandler = handler;
        this.onSmsLoadFinished = onSmsLoadFinished;
    }

    public void doParsing() {

        // 마지막 날짜이후 파싱 진행
//        sync();

        // cursor 초기화
        initCursor();

        // syncParsingRule()
        syncParsingRule(new Callback<Boolean>() {
            @Override
            public void onDataLoaded(Boolean value) {
                // parseBulk
                parserService.parseBulk(BulkLoader.this);
            }
        });
    }



    private long getLastSmsDate(List<SyncTransaction> transactions) {
        long maxTime = 0;

        long today = new Date().getTime();

        for(SyncTransaction tran : transactions) {

            long time = Utils.toTime(tran.spentDate);
            if(time <= today) {
                if(time > maxTime) {
                    maxTime = time;
                }
            }

        }
        return maxTime;
    }

    private void initCursor() {

        try {
            LOGI(TAG, "initCursor: start", logger);

            Uri uri = Uri.parse("content://sms/inbox");
            mCursor = mContext.getContentResolver().query(uri, null, getWhere(), null, "date asc");
            if (mCursor != null) {
                mCursor.moveToFirst();
            }
        } catch (Exception e) {
            LOGI(TAG, "initCursor: " + e.toString(), logger);
            onError(1);
        }

    }

    private String getWhere() {
        Calendar calendar = Calendar.getInstance();
        long time = prefManager.loadLongValue(PrefManager.SIGN_UP_TIME, 0);
        if(time == 0) {
            time = calendar.getTimeInMillis();
        } else {
            calendar.setTimeInMillis(time);
        }
        LOGI(TAG, "BulkTime toDate: " + Utils.getStringDateAsYYYYMMddHHmmss(calendar), logger);

        int month = calendar.get(Calendar.MONTH);
        calendar.set(Calendar.MONTH, month);
        calendar.set(Calendar.DATE, 1);

        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);

        calendar.add(Calendar.MONTH, BULK_MONTH);

        LOGI(TAG, "BulkTime fromDate: " + Utils.getStringDateAsYYYYMMddHHmmss(calendar), mIsDebug);

        String where;
        if(lastSmsDate == 0) {
            where = "date >= " + calendar.getTimeInMillis() + " and date <= " + time;
        } else {
            where = "date > " + lastSmsDate + " and date <= " + time;
        }
        LOGI(TAG, "getWhere: " +where, logger);

        return where;
    }

    @Override
    public int getSmsCount() {
        int cnt = mCursor == null ? 0 : mCursor.getCount();
        LOGI(TAG, "getSmsCount: " + cnt, logger);

        return cnt;
    }

    @Override
    public SMS getSmsAt(int n) {
        return getSMS();
    }

    private SMS getSMS() {
        SMS sms = null;
        if (mCursor != null) {
            sms = new SMS(mCursor.getInt(mCursor.getColumnIndex("_id")),
                mCursor.getString(mCursor.getColumnIndex("body")),
                mCursor.getString(mCursor.getColumnIndex("address")),
                mCursor.getString(mCursor.getColumnIndex("address")),
                getConvertedDate(mCursor.getLong(mCursor.getColumnIndex("date"))),
                Constants.SMSType.SMS.ordinal());
            mCursor.moveToNext();
        }

        return sms;
    }

    /**
     * @param now 현재 내역 수
     * @param total 전체 내역 수
     */
    @Override
    public void onProgress(int now, int total) {
        //ui 변경
        int percent;
        if(lastSmsDate == 0) {
            percent = ((now * 100) / total) + 1;

        } else {
            percent = 10 + (((now * 90) / total) + 1);
        }

        Bundle data = new Bundle();
        data.putInt("percent", percent);
        data.putInt("now", now);
        data.putInt("total", total);


        sendHandlerMsg(data,
                PROGRESS,
                percent);


    }

    /**
     * 구버전의 경우 v1형태에 맞게 구성후 검색서버 반영되면 변경 처리
     * doGrouping
     * @param parsedTransactions 파싱된 지출내역 목록
     * @param callback 성공여부 콜백 메소드
     */
    @Override
    public void sendToServerTransactions(final ArrayList<Transaction> parsedTransactions, final OnNetworkResultListener callback) {

        try {
            LOGI(TAG, "sendToServerTransactions: " + parsedTransactions.size(), logger);

            // 31개가 오는경우가있음 parser 확인
            ArrayList<Transaction> subParsedTransactions = parsedTransactions.size() > 30 ? (ArrayList<Transaction>) parsedTransactions.subList(0, 30) : parsedTransactions;
            ArrayList<com.tenqube.visual_third.model.parser.Transaction> transactions = Mapper.toParserTransactions(subParsedTransactions);
            setCurrency(transactions);

            repository.mergeTransactions(transactions);

            ArrayList<SearchCompanyRequest.Transaction> apiTransactions = Mapper.toSearchTransaction(transactions);
            if(!isEmpty(apiTransactions)) {
                SearchCompanyResponse response = repository.searchCompany(new SearchCompanyRequest(apiTransactions));
                if(response != null) {
                    repository.updateTransactions(response);
                }
            }
            repository.syncTransactions(callback);
        } catch (Exception e) {
            callback.onResult(true);
        }
    }

    @Override
    public void onCompleted() {

        sendHandlerMsg(null, FINISH, 0);

        if(mCursor != null)
            mCursor.close();

        if(onSmsLoadFinished != null)
            onSmsLoadFinished.onFinished(false);
    }

    @Override
    public void onError(final int resultCode) {

        //파싱룰 동기화가 필요한경우
        sendHandlerMsg(null, FINISH, 0);

        if(mCursor != null)
            mCursor.close();


        if(onSmsLoadFinished != null)
            onSmsLoadFinished.onFinished(true);
    }

    private void sendHandlerMsg(Bundle data, int what, int arg1) {

        if(mHandler != null) {
            Message hMsg = mHandler.obtainMessage();
            if(data != null) hMsg.setData(data);
            hMsg.what = what;
            hMsg.arg1 = arg1;
            mHandler.sendMessage(hMsg);
        }
    }

    public interface OnSmsLoadFinished {
        void onFinished(boolean isError);
    }

}
