package com.tenqube.visual_third;

import android.app.Activity;
import android.content.Context;
import android.service.notification.StatusBarNotification;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.tenqube.visual_third.exception.AuthException;
import com.tenqube.visual_third.exception.ParameterException;
import com.tenqube.visual_third.manager.AnswerManager;
import com.tenqube.visual_third.manager.VisualAlarmManager;
import com.tenqube.visual_third.model.js.LogRequest;
import com.tenqube.visual_third.parser.catcher.NotiManager;
import com.tenqube.visual_third.repository.RepositoryHolder;
import com.tenqube.visual_third.repository.VisualRepository;
import com.tenqube.visual_third.ui.OnResultListener;
import com.tenqube.visual_third.ui.VisualWebActivity;
import com.tenqube.visual_third.util.Validator;

import tenqube.parser.core.ParserService;

public class VisualServiceImpl implements VisualService {

    public static final String TAG = VisualServiceImpl.class.getSimpleName();

    private Context context;

    private ParserService parserService;
    private VisualRepository repository;
    private VisualAlarmManager alarmManager;
    private NotiManager notiManager;

    public static class Builder {

        Context context;
        VisualCallback visualCallback;

        String apiKey;
        String qualifier;
        String authority;

        int smallIcon;
        String channel;
        int color;

        boolean logger;

        public Builder withContext(final Context context) {
            this.context = context;
            return this;
        }

        public Builder visualCallback(final VisualCallback visualCallback) {
            this.visualCallback = visualCallback;
            return this;
        }

        public Builder apiKey(final String apiKey) {
            this.apiKey = apiKey;
            return this;
        }

        public Builder qualifier(final String qualifier) {
            this.qualifier = qualifier;
            return this;
        }

        public Builder authority(final String authority) {
            this.authority = authority;
            return this;
        }

        public Builder notification(int smallIcon, @NonNull String channel, int color) {

            this.smallIcon = smallIcon;
            this.channel = channel;
            this.color = color;

            return this;
        }

        public Builder logger(boolean logger) {
            this.logger = logger;
            return this;
        }

        public VisualService build() throws ParameterException {

            return new VisualServiceImpl(this);
        }
    }

    private VisualServiceImpl(final Builder builder) throws ParameterException {

        Validator.notNull(builder.context);

        Validator.isNotEmpty(builder.apiKey);

        Validator.isNotEmpty(builder.authority);

        Validator.in(builder.qualifier, Constants.DEV, Constants.PROD);

        this.context = builder.context;

        repository = RepositoryHolder.getInstance(context).getVisualRepository();

        repository.saveSDKInfo(builder.apiKey, builder.qualifier, builder.authority);

        repository.settingNotification(builder.smallIcon, builder.channel, builder.color);

        parserService = ParserService.getInstance(context);

        alarmManager = VisualAlarmManager.getInstance(context);

        notiManager = NotiManager.getInstance(context.getApplicationContext());

    }

    @Override
    public void parseNoti(StatusBarNotification sbn) {
        notiManager.parseNoti(sbn);
    }

    @Override
    public void signUp(@NonNull String uid, @NonNull OnResultListener onResultListener) throws ParameterException {

        Validator.isNotEmpty(uid);
        Validator.notNull(onResultListener);
        repository.signUp(uid, onResultListener);
    }

    @Override
    public void startVisual(final @Nullable Activity activity,
                            final @NonNull String uid,
                            final @NonNull String path,
                            final @NonNull OnResultListener resultListener) throws ParameterException {

        Validator.isNotEmpty(uid);
        Validator.notNull(resultListener);
        Validator.notNull(path);

        repository.signUp(uid, new OnResultListener() {
            @Override
            public void onResult(Constants.SignUpResponse signUpResult, String msg) {
                try {
                    //가입 성공시에만 isAgree 값에 맞게 설정정보 업데이트 처리
                    if (signUpResult == Constants.SignUpResponse.SUCCESS) {
                        AnswerManager.onKeyMetric(new LogRequest("startVisual"));
                        if (activity == null) {
                            VisualWebActivity.startActivity(context, path);
                        } else {
                            VisualWebActivity.startActivity(activity, path);
                        }
                    }

                } catch (AuthException e) {
                    signUpResult = Constants.SignUpResponse.FAIL;
                    msg = e.toString();
                } finally {
                    resultListener.onResult(signUpResult, msg);
                }
            }
        });

    }

    @Override
    public void signOut(Callback<Boolean> callback) {
        repository.signOut(new Callback<Boolean>() {
            @Override
            public void onDataLoaded(Boolean value) {
                alarmManager.setAlarms();

            }
        });
    }

    @Override
    public boolean isActiveTranPopup() {
        AnswerManager.onKeyMetric(new LogRequest("isActiveTranPopup"));
        return repository.isActiveTranPopup();
    }

    @Override
    public void setTranPopup(boolean isActive) {
        AnswerManager.onKeyMetric(new LogRequest("setTranPopup"));
        repository.setTranPopup(isActive);
    }

    @Override
    public void setReportAlarm(Constants.ReportAlarmType type, boolean isActive) {
        repository.setActiveNoti(type.name().toLowerCase(), isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveReportAlarm(Constants.ReportAlarmType type) {
        return repository.isActiveNoti(type.name().toLowerCase());
    }

    @Override
    public void setReportTest(Constants.ReportAlarmType type, int second) {
        alarmManager.setReportTest(type.name().toLowerCase(), second);
    }

    @Override
    public void setLogger(boolean isActive) {
        if(isActive() && parserService != null) {
            parserService.setDebugMode(isActive);
        }
    }

    @Override
    public void initSDK() {
        if(isActive() && parserService != null) {
            parserService.initDb();
        }
    }

    @Override
    public void setEnabled(boolean enabled) {
        repository.setEnabled(enabled);
        if(!enabled) {
            repository.setNotiAll(false);
            alarmManager.setAlarms();
        }
    }

    @Override
    public boolean isJoined() {
        return repository.isJoined();
    }

    @Override
    public void setAppNoti(boolean isActive) {
        repository.setAppNoti(isActive);
        alarmManager.setAlarms();
    }

    @Override
    public boolean isActiveAppNoti() {
        return repository.isAppNoti();
    }

    private boolean isActive() {
        return context != null;
    }
}
