package com.tencent.wecast.utils;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.view.Choreographer;

import com.tencent.wecast.WeCastAppContext;

import java.util.Map;
import java.util.concurrent.TimeUnit;

public class BlockDetectUtil {
    private static final long DETECT_MAIN_THREAD_DEAD_DURATION = 4 * 60 * 1000L; // 主线程卡死保底时间
    private static final int PRINT_THREAD_MAX_COUNT = 50;
    private static int sPrintThreadCount = 0;

    public static void setBlockListener(IBlockListener listener) {
        BlockMonitor.sInstance.setBlockListener(listener);
    }

    private static Choreographer.FrameCallback choreographerCb = new Choreographer.FrameCallback() {
        long lastFrameTimeNanos = 0;
        long currentFrameTimeNanos = 0;

        long frameNum = 0;

        @Override
        public void doFrame(long frameTimeNanos) {
            frameNum++;
            if (lastFrameTimeNanos == 0) {
                lastFrameTimeNanos = frameTimeNanos;
            }
            currentFrameTimeNanos = frameTimeNanos;

            long diffMs = TimeUnit.MILLISECONDS.convert(currentFrameTimeNanos - lastFrameTimeNanos, TimeUnit.NANOSECONDS);
            lastFrameTimeNanos = currentFrameTimeNanos;

            if (diffMs > 500) {
                Logger.t("BlockDetect").d("render flush frame time %d ms", diffMs);
            }


            if (frameNum >= 5) {
                frameNum = 0;
                BlockMonitor.sInstance.removeMonitor();
                BlockMonitor.sInstance.startMonitor();
            }
            Choreographer.getInstance().postFrameCallback(this);
        }
    };

    public static void start() {
        Choreographer.getInstance().postFrameCallback(choreographerCb);
        MainLoopBlockCheck.sInstance.start();
    }

    public static void stop() {
        Choreographer.getInstance().removeFrameCallback(choreographerCb);
    }

    private static class BlockMonitor {
        private static BlockMonitor sInstance = new BlockMonitor();
        private HandlerThread mBlockThread = new HandlerThread("block_monitor");
        private Handler mBlockHandler;
        private volatile boolean mIsStartMonitor = false;
        private long mLastHandleMs = 0;          // 启动时间
        private long mBlockCheckStepTime = 1000L;

        private IBlockListener mBlockListener = null;

        private Runnable mBlockHandlerRunnable = new Runnable() {
            @Override
            public void run() {
                // 限制在2s~8s卡顿时长，减少log量
                if (mBlockCheckStepTime >= 2000 && mBlockCheckStepTime <= 16000) {
                    printAllThread();
                }

                if (mBlockCheckStepTime >= 4000) {
                    Logger.t("BlockMonitor").i("mBlockCheckStepTime = " + mBlockCheckStepTime);
                }


                // 主线程卡死保底操作（超过一定时间）
                if (mBlockCheckStepTime >= DETECT_MAIN_THREAD_DEAD_DURATION) {
                    Logger.t("BlockMonitor").i("detect main thread dead, block time = " + mBlockCheckStepTime);
                    doDetectMainThreadDead();
                }

                mBlockHandler.postDelayed(mBlockHandlerRunnable, mBlockCheckStepTime);
                mBlockCheckStepTime += mBlockCheckStepTime;
            }
        };

        private Runnable mPrintAllThreadRunnable = new Runnable() {
            @Override
            public void run() {
                autoHandleBlock();
                mLastHandleMs = System.currentTimeMillis();
            }
        };

        private BlockMonitor() {
            mBlockThread.start();
            mBlockHandler = new Handler(mBlockThread.getLooper());
        }

        void setBlockListener(IBlockListener listener) {
            mBlockListener = listener;
        }

        void startMonitor() {
            mBlockCheckStepTime = 1000L;
            mBlockHandler.postDelayed(mBlockHandlerRunnable, mBlockCheckStepTime);
            mIsStartMonitor = true;
        }

        void removeMonitor() {
            mBlockHandler.post(mPrintAllThreadRunnable);
            if (mIsStartMonitor) {
                mBlockHandler.removeCallbacks(mBlockHandlerRunnable);
            }
            mIsStartMonitor = false;
        }

        private void autoHandleBlock() {
            if (mLastHandleMs > 0) {
                long stuckDurationMs = System.currentTimeMillis() - mLastHandleMs;
                if (stuckDurationMs >= 1000) {
                    Logger.t("BlockDetect").i("autoHandleBlock stuckDurationMs = " + stuckDurationMs);
                    printAllThread();
                }
                long stuckDuration = stuckDurationMs / 1000;
                if (stuckDuration >= 1) {
                    if (mBlockListener != null) {
                        mBlockListener.onBlock(stuckDuration);
                    }
                }
            }
        }

        private void printAllThread() {
            if (sPrintThreadCount < PRINT_THREAD_MAX_COUNT) {
                sPrintThreadCount++;

                Map<Thread, StackTraceElement[]> allStackTraces = Thread.getAllStackTraces();
                Thread mainThread = Looper.getMainLooper().getThread();
                allStackTraces.put(mainThread, mainThread.getStackTrace());
                StringBuilder sb = new StringBuilder();
                for (Thread thread : allStackTraces.keySet()) {
                    StackTraceElement[] trace = thread.getStackTrace();
                    sb.append("threadName=")
                            .append(thread.getName())
                            .append("|Priority=")
                            .append(thread.getPriority())
                            .append(": ");
                    for (int i = 0; i < trace.length; i++) {
                        sb.append(trace[i].toString()).append("\n\t");
                    }
                    int endIndex = sb.length();
                    if (endIndex > 2) {
                        endIndex = endIndex - 2;
                    }
                    Logger.t("BlockDetect").i(sb.substring(0, endIndex));
                    sb.setLength(0);
                }
            }
        }


        // 主线程卡死处理
        private void doDetectMainThreadDead() {
            // 1、上报事件
            ReportUtils.AddEventReport(ReportUtils.EVENT_PROBLEM_MAIN_THREAD_DEAD, 0, "");

            // 2、5秒后强杀进程
            mBlockHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    WeCastUtil.killWeCast(WeCastAppContext.get());
                    android.os.Process.killProcess(android.os.Process.myPid());
                }
            }, 5000L);
        }
    }

    private static class MainLoopBlockCheck {
        private static MainLoopBlockCheck sInstance = new MainLoopBlockCheck();

        private Handler mHandler = new Handler(Looper.getMainLooper());
        private long mLastTime = 0L;
        private Runnable mRefreshTask = new Runnable() {
            @Override
            public void run() {
                int diff = (int) ((System.currentTimeMillis() - mLastTime) / 1000);
                if (mLastTime > 0 && diff >= 2) {
                    ReportUtils.AddEventReport(ReportUtils.EVENT_PROBLEM_SYSTEM_STUCK_NEW, diff, "");
                }
                if (diff > 5) {
                    Logger.t("BlockDetect").i("refresh delay time: " + diff);
                }
                mLastTime = System.currentTimeMillis();
                mHandler.postDelayed(mRefreshTask, 1000L);
            }
        };

        void start() {
            mHandler.post(mRefreshTask);
        }
    }

    public interface IBlockListener {
        // 单位s
        void onBlock(long blockDuration);
    }
}
