package com.tencent.wecast.sender.cloud.activity

import android.app.Dialog
import android.content.Context
import android.content.res.Configuration
import android.graphics.Color
import android.os.Bundle
import android.os.Handler
import android.os.Looper
import android.os.Message
import android.support.v7.app.AlertDialog
import android.text.TextUtils
import android.widget.TextView
import android.widget.Toast
import com.tencent.wecast.WeCastBaseActivity
import com.tencent.wecast.WeCastConfigManager
import com.tencent.wecast.WeCastEnv
import com.tencent.wecast.sender.cloud.R
import com.tencent.wecast.sender.cloud.utils.CommonUtils
import com.tencent.wecast.utils.Logger


open class BaseActivity : WeCastBaseActivity() {

    companion object {
        private val TAG = "WeCastBaseActivity"
    }

    /**
     * 构造一个主线程的Handler，用于投递消息
     */
    protected var mBaseHandler: Handler? = null

    protected var mLoadingDialog: Dialog? = null

    /**
     * 环境切换标志
     */
    private var mLastChangeEnvTapTimeMs: Long = 0
    private var mChangeEnvTapTimes: Int = 0

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        supportActionBar?.setHomeButtonEnabled(true)
        supportActionBar?.setDisplayHomeAsUpEnabled(true)
        mBaseHandler = object : Handler(Looper.getMainLooper()) {
            override fun handleMessage(msg: Message) {
                baseHandleMessage(msg)
            }
        }
    }

    override fun onResume() {
        Logger.t(TAG).d("onResume > $localClassName")
        super.onResume()
    }

    override fun onPause() {
        Logger.t(TAG).d("onPause > $localClassName")
        super.onPause()
    }

    override fun onDestroy() {
        Logger.t(TAG).d("onDestroy > $localClassName")
        super.onDestroy()
        mLoadingDialog?.dismiss()
        mLoadingDialog = null

        mBaseHandler?.removeCallbacksAndMessages(null)
        mBaseHandler = null
    }


    /**
     * 界面公共部分
     */
    fun showLoading(text: String) {
        if (mLoadingDialog == null) {
            mLoadingDialog = CommonUtils.createLoadingDialog(this)
        }

        val textView = mLoadingDialog?.findViewById(R.id.loading_dialog_text) as TextView?
        textView?.text = if (TextUtils.isEmpty(text)) "" else text
        mLoadingDialog?.show()
    }

    fun hideLoading() {
        mLoadingDialog?.dismiss()
        mLoadingDialog = null
    }

    /**
     * 子类重写此方法，进行MSG处理
     *
     * @param msg
     */
    fun baseHandleMessage(msg: Message) {

    }

    fun getConfValue(id: Int, defaultValue: Boolean): Boolean {
        return try {
            resources.getBoolean(id)
        } catch (e: Exception) {
            Logger.t(TAG).e("getConfValue e =" + e)
            defaultValue
        }
    }

    override fun attachBaseContext(newBase: Context?) {
        val config = Configuration(newBase?.resources?.configuration)
        config.setLocale(WeCastConfigManager.getInstance().languageLocale)
        super.attachBaseContext(newBase?.createConfigurationContext(config))
    }

    fun showAlertDialog(title: String, message: String, positiveText: String?, positiveFunc: (() -> Unit)?
                        , negativeText: String?, negativeFunc: (() -> Unit)?) {
        runOnUiThread {
            val alertDialogBuilder = AlertDialog.Builder(this@BaseActivity)
                    .setTitle(title)
                    .setCancelable(false)
            if (positiveText != null) {
                alertDialogBuilder.setPositiveButton(getString(R.string.wecast_confirm_text)
                ) { _, _ ->
                    positiveFunc?.invoke()
                }
            }
            if (negativeText != null) {
                alertDialogBuilder.setNegativeButton(getString(R.string.wecast_cancel_text)
                ) { _, _ ->
                    negativeFunc?.invoke()
                }
            }
            val alertDialog = alertDialogBuilder.create()
            alertDialog?.setMessage(message)
            alertDialog?.show()
            alertDialog?.getButton(Dialog.BUTTON_POSITIVE)?.setTextColor(Color.BLUE)
            alertDialog?.getButton(Dialog.BUTTON_NEGATIVE)?.setTextColor(Color.GRAY)
        }
    }

    fun tryChangeEnv() {
        /**
         * 5秒10下切换环境
         */
        val currentTimeMs = System.currentTimeMillis()
        if (currentTimeMs - mLastChangeEnvTapTimeMs > 5 * 1000) {
            mLastChangeEnvTapTimeMs = currentTimeMs
            mChangeEnvTapTimes = 0
        } else {
            mChangeEnvTapTimes++
        }

        if (mChangeEnvTapTimes >= 9) {
            mLastChangeEnvTapTimeMs = 0
            mChangeEnvTapTimes = 0

            val dialog = AlertDialog.Builder(this)
                    .setTitle("提示").setMessage("确认切换后台环境吗？切换后可能导致投屏不可用").setPositiveButton("确认切换") { _, _ ->
                        Thread(Runnable {
                            WeCastEnv.changeNextEnv()
                            val resultStr = WeCastEnv.getCurrentEnvDesc()
                            mBaseHandler?.post {
                                Toast.makeText(this, resultStr, Toast.LENGTH_LONG).show()
                            }
                        }).start()
                    }.setNegativeButton("取消") { _, _ ->

                    }.setCancelable(false).create()
            dialog.show()
            dialog.getButton(Dialog.BUTTON_POSITIVE)?.setTextColor(Color.BLUE)
            dialog.getButton(Dialog.BUTTON_NEGATIVE)?.setTextColor(Color.BLUE)
        }
    }
}