package com.tencent.wecast.utils;

import android.util.Log;

import com.tencent.wecast.jni.JniHelper;

import java.util.Locale;

public class Logger {
    private final static int LOG_DEBUG = 1;
    private final static int LOG_INFO = 2;
    private final static int LOG_WARNING = 3;
    private final static int LOG_ERROR = 4;

    private static Printer printer = new LoggerPrinter();

    private Logger() {
        //no instance
    }

    public static Printer t(String tag) {
        return printer.t(tag);
    }

    public static void d(String message, Object... args) {
        printer.d(message, args);
    }

    public static void d(Object object) {
        printer.d(object);
    }

    public static void e(String message, Object... args) {
        printer.e(null, message, args);
    }

    public static void e(Throwable throwable, String message, Object... args) {
        printer.e(throwable, message, args);
    }

    public static void i(String message, Object... args) {
        printer.i(message, args);
    }

    public static void w(String message, Object... args) {
        printer.w(message, args);
    }

    // 关键节点
    public static Printer kp() {
        return printer.kp();
    }


    public interface Printer {
        Printer t(String tag);

        void d(String message, Object... args);

        void d(Object object);

        void e(String message, Object... args);

        void e(Throwable throwable, String message, Object... args);

        void w(String message, Object... args);

        void i(String message, Object... args);

        Printer kp();
    }

    public static class LoggerPrinter implements Printer {
        private static final String DEFAULT_TAG = "TAG";
        private static String TAG = DEFAULT_TAG;

        @Override
        public Printer t(String tag) {
            TAG = tag;
            return this;
        }

        @Override
        public void d(String message, Object... args) {
            nativeInfo(TAG, message, args);
            TAG = DEFAULT_TAG;
        }

        @Override
        public void d(Object object) {
            nativeInfo(TAG, "%s", object.toString());
            TAG = DEFAULT_TAG;
        }

        @Override
        public void e(String message, Object... args) {
            nativeError(TAG, message, args);
            TAG = DEFAULT_TAG;
        }

        @Override
        public void e(Throwable throwable, String message, Object... args) {
            nativeWarning(TAG, message + " e:" + throwable.getMessage(), args);
            TAG = DEFAULT_TAG;
        }

        @Override
        public void w(String message, Object... args) {
            nativeWarning(TAG, message, args);
            TAG = DEFAULT_TAG;
        }

        @Override
        public void i(String message, Object... args) {
            nativeInfo(TAG, message, args);
            TAG = DEFAULT_TAG;
        }

        @Override
        public Printer kp() {
            TAG = "KeyPoint";
            return this;
        }
    }

    private static void nativeDebug(String tag, String msgFormat, Object... args) {
        nativeLog(LOG_DEBUG, String.format("%s%s", tag, format(msgFormat, args)));
    }

    private static void nativeInfo(String tag, String msgFormat, Object... args) {
        nativeLog(LOG_INFO, String.format("%s%s", tag, format(msgFormat, args)));
    }

    private static void nativeWarning(String tag, String msgFormat, Object... args) {
        nativeLog(LOG_WARNING, String.format("%s%s", tag, format(msgFormat, args)));
    }

    private static void nativeError(String tag, String msgFormat, Object... args) {
        nativeLog(LOG_ERROR, String.format("%s%s", tag, format(msgFormat, args)));
    }

    private static void nativeLog(int logType, String content) {
        try {
            Object ret = JniHelper.callJniFunc("log", logType, content);
            if (ret == null) {
                switch (logType) {
                    case LOG_DEBUG:
                        Log.d("Log", content);
                        break;
                    default:
                        Log.w("Log", content);
                        break;

                }
            }
        } catch (UnsatisfiedLinkError ignore) {
        }
    }

    protected static String format(String format, Object... args) {
        String funcTag = getFuncTag();
        if (format == null) {
            return funcTag + "msg is null";
        }

        if (args == null || args.length == 0) {
            return funcTag + format;
        }
        return funcTag + String.format(Locale.US, format, args);
    }


    private final static int MIN_STACK_OFFSET = 5;

    private static int getStackOffset(StackTraceElement[] trace, Class cla) {
        for (int i = 0; i < trace.length; i++) {
            StackTraceElement e = trace[i];
            String name = e.getClassName();
            if (cla.equals(Logger.class) && i < trace.length - 1 && trace[i + 1].getClassName()
                    .equals(Logger.class.getName())) {
                continue;
            }
            if (name.equals(cla.getName())) {
                return i + 1;
            }
        }
        return -1;
    }

    private static StackTraceElement getCurrentStackTrace() {
        StackTraceElement[] trace = Thread.currentThread().getStackTrace();
        int stackOffset = getStackOffset(trace, LoggerPrinter.class);
        if (stackOffset == -1) {
            return null;
        }

        return trace[stackOffset];
    }

    private static String getFuncTag() {
        final StackTraceElement stackTrace = getCurrentStackTrace();
        if (stackTrace != null) {
            String fileName = stackTrace.getFileName();
            if (fileName == null)
                fileName = "";  // It is necessary if you want to use proguard obfuscation.
            return "(" + fileName + ":" + stackTrace.getLineNumber() + ") ";
        }
        return "";
    }
}
