package com.tencent.wecast.jni;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

public class ParamsPackage {
    public static final byte PKG_MAGIC_NUMBER = (byte) 0xEF;

    private static final byte TYPE_BOOLEAN = 0;
    private static final byte TYPE_BYTE = 1;
    private static final byte TYPE_CHAR = 2;
    private static final byte TYPE_SHORT = 3;
    private static final byte TYPE_INT = 4;
    private static final byte TYPE_FLOAT = 5;
    private static final byte TYPE_LONG = 6;
    private static final byte TYPE_DOUBLE = 7;
    private static final byte TYPE_STRING = 8;


    public static ParamsFromCpp fromBytes(byte[] params) throws IOException {
        if (params != null && params[0] == PKG_MAGIC_NUMBER) {
            return new ParamsFromCpp(params);
        }
        return null;
    }

    // 来自jni层的参数
    public static class ParamsFromCpp {
        private ByteArrayInputStream bais;
        private DataInputStream dis;
        private int paramCnt;

        public ParamsFromCpp(byte[] params) throws IOException {
            paramCnt = getInt(params, params.length - 4);
            bais = new ByteArrayInputStream(params);
            dis = new DataInputStream(bais);
            dis.readByte(); // magic number 0xEF
            dis.readInt();  // data length
        }

        public boolean isEmptyPkg() {
            return paramCnt == 0;
        }

        public int getRemainSize() {
            return paramCnt;
        }

        public Object getNext() throws IOException {
            if (paramCnt > 0) {
                paramCnt--;
                byte type = dis.readByte();
                switch (type) {
                    case TYPE_BOOLEAN:
                        return dis.readByte() == '1';
                    case TYPE_BYTE:
                        return dis.readByte();
                    case TYPE_SHORT:
                        return dis.readShort();
                    case TYPE_INT:
                        return dis.readInt();
                    case TYPE_FLOAT:
                        return dis.readFloat();
                    case TYPE_LONG:
                        return dis.readLong();
                    case TYPE_STRING:
                        byte[] strBytes = new byte[dis.readInt()];
                        //noinspection ResultOfMethodCallIgnored
                        dis.read(strBytes, 0, strBytes.length);
                        return new String(strBytes);
                    default:
                        break;
                }
            }
            close();
            return null;
        }

        private void close() throws IOException {
            if (bais != null) {
                dis.close();
                bais.close();
                dis = null;
                bais = null;
            }
        }

        private int getInt(byte[] data, int offset) {
            return (data[offset] << 24) & 0xff000000 |
                    (data[offset + 1] << 16) & 0x00ff0000 |
                    (data[offset + 2] << 8) & 0x0000ff00 |
                    (data[offset + 3]) & 0x000000ff;
        }
    }

    // 传给jni层的参数
    public static class ParamsToCpp {
        private ByteArrayOutputStream baos;
        private DataOutputStream dos;
        private int paramCnt;

        public ParamsToCpp() throws IOException {
            paramCnt = 0;
            baos = new ByteArrayOutputStream();
            dos = new DataOutputStream(baos);
            dos.writeByte(PKG_MAGIC_NUMBER);
            dos.writeInt(0);    // 参数的字节长度
        }

        public ParamsToCpp addBool(boolean param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_BOOLEAN);
            dos.writeByte(param ? '1' : '0');
            return this;
        }

        public ParamsToCpp addByte(byte param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_BYTE);
            dos.writeByte(param);
            return this;
        }

        public ParamsToCpp addShort(short param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_SHORT);
            dos.writeShort(param);
            return this;
        }

        public ParamsToCpp addInt(int param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_INT);
            dos.writeInt(param);
            return this;
        }

        public ParamsToCpp addLong(long param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_LONG);
            dos.writeLong(param);
            return this;
        }

        public ParamsToCpp addFloat(float param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_FLOAT);
            dos.writeFloat(param);
            return this;
        }

        public ParamsToCpp addDouble(double param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_DOUBLE);
            dos.writeDouble(param);
            return this;
        }

        // 仅传ascii字符
        public ParamsToCpp addString(String param) throws IOException {
            paramCnt++;
            dos.writeByte(TYPE_STRING);
            byte[] bytes = param.getBytes("UTF-8");
            dos.writeInt(bytes.length);
            dos.write(bytes);
            return this;
        }

        public byte[] build() throws IOException {
            dos.writeInt(paramCnt);
            dos.flush();

            byte[] bytes = baos.toByteArray();
            setDataLength(bytes);
            dos.close();
            baos.close();
            return bytes;
        }

        private void setDataLength(byte[] bytes) {
            int dataLen = bytes.length - 4 - 1;
            bytes[1] = (byte) ((dataLen >> 24) & 0xFF);
            bytes[2] = (byte) ((dataLen >> 16) & 0xFF);
            bytes[3] = (byte) ((dataLen >> 8) & 0xFF);
            bytes[4] = (byte) ((dataLen) & 0xFF);
        }
    }

    private static byte[] VOID_PKG = {PKG_MAGIC_NUMBER, 0, 0, 0, 0, 0, 0, 0, 0};

    public static byte[] voidPkg() {
        return VOID_PKG;
    }
}
