package com.tencent.wecast.sender.cloud.activity

import android.Manifest
import android.app.Dialog
import android.content.Intent
import android.content.pm.PackageManager
import android.graphics.Color
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.support.v4.app.ActivityCompat
import android.support.v4.content.ContextCompat
import android.support.v7.app.AlertDialog
import android.text.TextUtils
import android.widget.Button
import android.widget.Toast
import com.tencent.wecast.WeCastConfigManager
import com.tencent.wecast.WeCastEnv
import com.tencent.wecast.WeCastUIConfigManager
import com.tencent.wecast.sender.cloud.R
import com.tencent.wecast.sender.cloud.bean.WeCastInfo
import com.tencent.wecast.sender.cloud.utils.CloudUtil
import com.tencent.wecast.sender.eb.SenderEvents
import com.tencent.wecast.utils.*
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode
import java.util.*

open class CloudSplashActivity : BaseActivity() {

    companion object {

        private val TAG = "CloudSplashActivity"

        private val PERMISSION_REQUEST_CODE = 10001

        /**
         * 闪屏最少显示时间
         */
        private val SPLASH_SHOW_MIN_TIME = 1000

        @JvmField
        val LAUNCHER_FLAG_KEY = "launcher_flag"

        @JvmField
        val LAUNCHER_FLAG_DEFAULT = 0
        @JvmField
        val LAUNCHER_FLAG_INTEGRATE = 1

        @JvmField
        val DEVICE_NAME_KEY = "device_name"
    }

    /**
     * 环境切换标志
     */
    private var mLastChangeEnvTapTimeMs: Long = 0
    private var mChangeEnvTapTimes: Int = 0

    private var mDelayFinishFlag: Boolean = false
    private var mLoginFinishFlag: Boolean = true

    private var mLoginResult: Boolean = false
    private var mLoginInfo: String? = null
    private var mReturnCode: Int = 0

    private var mFatalDialog: AlertDialog? = null

    private val mPermissionMap = HashMap<String, String>()
    private val mPermissionToRequest = ArrayList<String>()

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        if (intent.getBooleanExtra("isInitialized", false)) {
            jump()
        }

        val curActivityLabel = if (!WeCastInfo.instance.usingMirror
                || WeCastConfigManager.getInstance().isLanScheme) {
            WeCastUIConfigManager.ActivityLabel.SENDER_SPLASH_1
        } else {
            WeCastUIConfigManager.ActivityLabel.SENDER_SPLASH_2
        }
        val rootView = WeCastUIConfigManager.getInstance().getLayoutRootView(curActivityLabel, applicationContext)
        if (null == rootView) {
            Logger.t(TAG).e("rootView is null.")
            finish()
            return
        }
        WeCastUIConfigManager.getInstance().applyUIConfig(curActivityLabel, rootView)
        setContentView(rootView)

        mPermissionMap.put(Manifest.permission.WRITE_EXTERNAL_STORAGE, getString(R.string.wecast_rw_sdcard))
        /** 申请权限  */
        if (Build.VERSION.SDK_INT >= 23) {
            mPermissionToRequest.clear()

            var shouldShowDialog = false
            var permissionDialogStr = ""

            for ((key, value) in mPermissionMap) {
                if (ContextCompat.checkSelfPermission(this@CloudSplashActivity, key) != PackageManager.PERMISSION_GRANTED) {
                    mPermissionToRequest.add(key)
                    permissionDialogStr += value + "，"
                    if (ActivityCompat.shouldShowRequestPermissionRationale(this@CloudSplashActivity, key)) {
                        shouldShowDialog = true
                    }
                }
            }

            if (permissionDialogStr.isNotEmpty()) {
                permissionDialogStr = permissionDialogStr.substring(0, permissionDialogStr.length - 1)
            }

            if (!mPermissionToRequest.isEmpty()) {
                val names = mPermissionToRequest.toTypedArray()
                if (shouldShowDialog) {
                    showExplanation(getString(R.string.wecast_apply_permission),
                            String.format(getString(R.string.wecast_apply_relative_permission), permissionDialogStr),
                            names,
                            PERMISSION_REQUEST_CODE)
                } else {
                    ActivityCompat.requestPermissions(this@CloudSplashActivity, names, PERMISSION_REQUEST_CODE)
                }
            } else {
                login()
            }
        } else {
            login()
        }

        EventBusUtil.register(this)
    }


    /**
     * 自动登录，同时保证闪屏最少显示2s
     */
    private fun login() {
        if (mLoginFinishFlag) {
            mLoginFinishFlag = false
            var nickName = intent.getStringExtra(DEVICE_NAME_KEY)
            if (TextUtils.isEmpty(nickName)) {
                nickName = DeviceInfoUtil.getDeviceName()
            } else {
                DeviceInfoUtil.setDeviceName(nickName)
            }
            CloudUtil.JniMethod.login(nickName)
            mBaseHandler?.postDelayed({
                mDelayFinishFlag = true
                tryJump()
            }, SPLASH_SHOW_MIN_TIME.toLong())
        }
    }

    override fun onPause() {
        super.onPause()
        WeCastInfo.instance.isRunning = false
    }

    override fun onDestroy() {
        super.onDestroy()
        EventBusUtil.unregister(this)
        mFatalDialog?.dismiss()
        mFatalDialog = null
    }

    override fun onRequestPermissionsResult(requestCode: Int, permissions: Array<String>, grantResults: IntArray) {
        when (requestCode) {
            PERMISSION_REQUEST_CODE -> {
                var allPermissionGranted = true
                var permissionNotGrantedStr = ""
                if (permissions.isNotEmpty() && grantResults.isNotEmpty() && permissions.size == grantResults.size) {
                    for (i in grantResults.indices) {
                        if (grantResults[i] != PackageManager.PERMISSION_GRANTED) {
                            allPermissionGranted = false
                            permissionNotGrantedStr += mPermissionMap[permissions[i]] + "，"
                        }
                    }
                } else {
                    allPermissionGranted = false
                }

                if (permissionNotGrantedStr.isNotEmpty()) {
                    permissionNotGrantedStr = permissionNotGrantedStr.substring(0, permissionNotGrantedStr.length - 1)
                }

                if (allPermissionGranted) {
                    login()
                } else {
                    Toast.makeText(this,
                            String.format(getString(R.string.wecast_apply_permission_failed), permissionNotGrantedStr),
                            Toast.LENGTH_SHORT).show()
                    mBaseHandler?.postDelayed({ finish() }, 3000)
                }
            }
            else -> {
                super.onRequestPermissionsResult(requestCode, permissions, grantResults)
            }
        }
    }

    /**
     * 申请权限显示说明
     *
     * @param title
     * @param message
     * @param permissions
     * @param permissionRequestCode
     */
    private fun showExplanation(title: String,
                                message: String,
                                permissions: Array<String>,
                                permissionRequestCode: Int) {
        val builder = AlertDialog.Builder(this)
        builder.setTitle(title)
                .setMessage(message)
                .setPositiveButton(android.R.string.ok) { dialog, id ->
                    ActivityCompat.requestPermissions(this@CloudSplashActivity, permissions, permissionRequestCode)
                }
        val dialog = builder.create()
        dialog.show()
        dialog.getButton(Dialog.BUTTON_POSITIVE)?.setTextColor(
                ContextCompat.getColor(applicationContext,
                        R.color.wecast_common_font_color_1))
    }


    /**
     * 根据登录结果跳转
     */
    private fun tryJump() {
        if (mDelayFinishFlag && mLoginFinishFlag) {
            if (mLoginResult) {
                jump()
            } else {
                showFatalDialog(mLoginInfo)
            }
        }
    }

    private fun jump() {
        if (WeCastConfigManager.getInstance().isLanScheme) {
            startActivity(Intent(this@CloudSplashActivity, CloudMainActivity1::class.java))
        } else {
            startActivity(Intent(this@CloudSplashActivity, CloudMainActivity2::class.java))
        }
        finish()
    }

    /**
     * 严重错误对话框
     *
     * @param info
     */
    private fun showFatalDialog(info: String?) {
        if (mFatalDialog != null && mFatalDialog!!.isShowing) {
            return
        }
        Logger.t(TAG).e("showFatalDialog info = $info")

        mFatalDialog = AlertDialog.Builder(this@CloudSplashActivity)
                .setTitle(getString(R.string.wecast_failed_text))
                .setPositiveButton(getString(R.string.wecast_retry_text)
                ) { _, _ ->
                    mFatalDialog?.dismiss()
                    login()
                }
                .setNegativeButton(getString(R.string.wecast_exit_app)
                ) { _, _ ->
                    mFatalDialog?.dismiss()
                    CloudUtil.JniMethod.unInitSender()
                    finish()
                }.setNeutralButton("       ", null)
                .setCancelable(false)
                .create()
        mFatalDialog?.setMessage(getString(R.string.wecast_start_cast_failed))
        mFatalDialog?.show()
        mFatalDialog?.getButton(Dialog.BUTTON_POSITIVE)?.setTextColor(Color.BLUE)
        mFatalDialog?.getButton(Dialog.BUTTON_NEGATIVE)?.setTextColor(Color.BLUE)
        val neutralButton: Button? = mFatalDialog?.getButton(Dialog.BUTTON_NEUTRAL)
        neutralButton?.background = null
        neutralButton?.setOnClickListener {
            /**
             * 5秒10下切换环境
             */
            val currentTimeMs = System.currentTimeMillis()
            if (currentTimeMs - mLastChangeEnvTapTimeMs > 5 * 1000) {
                mLastChangeEnvTapTimeMs = currentTimeMs
                mChangeEnvTapTimes = 0
            } else {
                mChangeEnvTapTimes++
            }

            if (mChangeEnvTapTimes >= 9) {
                mLastChangeEnvTapTimeMs = 0
                mChangeEnvTapTimes = 0

                Thread(Runnable {
                    WeCastEnv.changeNextEnv()
                    val resultStr = WeCastEnv.getCurrentEnvDesc()
                    mBaseHandler?.post {
                        Toast.makeText(this, resultStr, Toast.LENGTH_LONG).show()
                    }
                }).start()
            }
        }
    }

    override fun onBackPressed() {
        Logger.t(TAG).d("onBackPressed forbidden.")
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onLoginSuccessEvent(event: SenderEvents.ELoginSuccess) {
        Logger.t(TAG).d("onLoginSuccessEvent")
        mLoginFinishFlag = true
        mLoginResult = true
        mReturnCode = 0
        mLoginInfo = "login success"
        tryJump()
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onLoginFailedEvent(event: SenderEvents.ELoginFailed) {
        Logger.t(TAG).d("onLoginFailedEvent errorCode:${event.errorCode}")
        if (event.errorCode == 15) { // 封版本
            return
        }
        mLoginFinishFlag = true
        mLoginResult = false
        mReturnCode = -1
        mLoginInfo = "login failed"
        tryJump()
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onAppHasForceUpdateEvent(event: SenderEvents.EAppUpdate) {
        Logger.t(TAG).d("onAppHasForceUpdateEvent:" +
                "forceUpdate:${event.forceUpdate}, " +
                "url:${event.url}, " +
                "forceUrl:${event.forceUrl}, " +
                "md5:${event.md5}, " +
                "msg:${event.msg}")
        if (event.forceUpdate) {
            try {
                if (mFatalDialog != null && mFatalDialog!!.isShowing) {
                    return
                }
                if (TextUtils.isEmpty(event.forceUrl)) {
                    mFatalDialog = AlertDialog.Builder(this@CloudSplashActivity)
                            .setPositiveButton(R.string.wecast_confirm_text
                            ) { _, _ ->
                                ReportUtils.AddEventReport(ReportUtils.EVENT_BTN_FORBIDDEN_VER_CONFIRM, 0, "")
                                mFatalDialog?.dismiss()
                                finish()
                            }
                            .setCancelable(false)
                            .create()
                } else {
                    mFatalDialog = AlertDialog.Builder(this@CloudSplashActivity)
                            .setPositiveButton(R.string.wecast_confirm_text
                            ) { _, _ ->
                                ReportUtils.AddEventReport(ReportUtils.EVENT_BTN_FORBIDDEN_VER_UPDATE_NOW, 0, "")
                                mFatalDialog?.dismiss()
                                try {
                                    val uri = Uri.parse(event.forceUrl)
                                    val intent = Intent(Intent.ACTION_VIEW, uri)
                                    startActivity(intent)
                                    finish()
                                } catch (e: Exception) {
                                    Toast.makeText(this@CloudSplashActivity,
                                            getString(R.string.wecast_illegal_url),
                                            Toast.LENGTH_LONG).show()
                                    finish()
                                }
                            }
                            .setCancelable(false)
                            .create()
                }
                mFatalDialog?.setTitle(R.string.wecast_note_title)
                mFatalDialog?.setMessage(event.msg)
                mFatalDialog?.show()
                mFatalDialog?.getButton(Dialog.BUTTON_POSITIVE)?.setTextColor(Color.BLUE)
            } catch (e: Exception) {
                Logger.t(TAG).e("onAppHasForceUpdateEvent e = $e")
            }
        }
        return
    }

    @Subscribe(sticky = true, threadMode = ThreadMode.MAIN)
    fun onConfigUpdateEvent(event: SenderEvents.EUIConfigUpdate) {
        Logger.t(TAG).d("onConfigUpdateEvent castScheme = ${event.castScheme} ui:${event.uiConfig}")
        // 发送端xcast的使用是接口传参决定
        WeCastConfigManager.getInstance().setSchemeConfig(event.castScheme, false, true)
        WeCastUIConfigManager.getInstance().updateConfig(event.uiConfig)
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onShowErrorEvent(event: SenderEvents.EShowError) {
        Logger.t(TAG).d("onShowErrorEvent errorCode = ${event.errorCode}")
        if (event.errorCode == 15) { // 封版本
            return
        }
        mLoginFinishFlag = true
        mLoginResult = false
        mReturnCode = -1
        mLoginInfo = "login failed"
        tryJump()
    }
}
