package com.tencent.wecast.sender.cloud

import android.app.Activity
import android.content.Context
import android.os.Build
import android.os.LocaleList
import android.support.annotation.AnimRes
import com.tencent.wecast.WeCastConfigManager
import com.tencent.wecast.WeCastConstant
import com.tencent.wecast.sender.cloud.bean.WeCastInfo
import com.tencent.wecast.sender.cloud.utils.CloudUtil
import com.tencent.wecast.sender.eb.SenderEvents
import com.tencent.wecast.utils.EventBusUtil
import com.tencent.wecast.utils.Logger
import org.greenrobot.eventbus.EventBus
import org.greenrobot.eventbus.Subscribe
import org.greenrobot.eventbus.ThreadMode
import java.util.*


object WeCast {
    @JvmField
    val LANGUAGE_ZH_CN = 0
    @JvmField
    val LANGUAGE_ZH_TW = 1
    @JvmField
    val LANGUAGE_EN = 2

    @JvmField
    val CAST_STATE_READY = 0
    @JvmField
    val CAST_STATE_CASTING = 1
    @JvmField
    val CAST_STATE_CLOSED = 2


    private const val TAG = "WeCast"

    private var mOnWeCastRunListener: OnWeCastRunListener? = null

    /**
     * AuthCode错误或者过期Listener
     */
    public interface OnWeCastRunListener {
        fun onAuthCodeError()
    }

    fun run(activity: Activity, corpId: String, authCode: String, userName: String) {
        run(activity, corpId, authCode, userName, null)
    }

    /**
     * 启动WeCast
     *
     * @param activity Activity
     * @param corpId 企业CorpId
     * @param authCode AuthCode
     * @param userName 投屏用户名
     * @param onWeCastRunListener AuthCode错误或者过期Listener
     * @param usingMirror 是否使用镜像
     */
    fun run(activity: Activity,
            corpId: String,
            authCode: String,
            userName: String,
            onWeCastRunListener: OnWeCastRunListener?,
            usingMirror: Boolean = true) {
        EventBusUtil.register(this)
        if (onWeCastRunListener != null) {
            this.mOnWeCastRunListener = onWeCastRunListener
        }
        WeCastInfo.instance.corpId = corpId
        WeCastInfo.instance.usingMirror = usingMirror
        if (!isRunning() && !isCasting()) {
            WeCastInfo.instance.authCode = authCode
            CloudUtil.jump2WeCastPinPage(activity, WeCastInfo.instance.corpId,
                    WeCastInfo.instance.authCode, userName, usingMirror)
        } else {
            CloudUtil.jump2WeCastPage(activity)
        }
        WeCastInfo.instance.isRunning = true
    }

    /**
     * 停止WeCast
     */
    fun stop() {
        // 退出Casting
        if (WeCastInfo.instance.isCasting) {
            CloudUtil.JniMethod.exitCast(0)
        }
        // 退出WeCast
        CloudUtil.JniMethod.unInitSender()
        EventBus.getDefault().post(SenderEvents.EStopCast())
        EventBusUtil.unregister(this)
        WeCastInfo.instance.isRunning = false
        WeCastInfo.instance.isCasting = false
        WeCastInfo.instance.mStateChangeListener?.onChanged(CAST_STATE_CLOSED)
    }

    /**
     * 是否已经启动WeCast
     */
    fun isRunning(): Boolean {
        return WeCastInfo.instance.isRunning
    }

    /**
     * 是否正在投屏
     */
    fun isCasting(): Boolean {
        return WeCastInfo.instance.isCasting
    }

    /**
     * 设置SSID限制
     */
    fun setLimitSSID(ssid: String) {
        CloudUtil.setLimitSSID(ssid)
    }

    /**
     * 设置投屏语言
     * @params: type - 语言类型
     *              0 - 中文简体
     *              1 - 中文繁体
     *              2 - 英文
     */
    fun setLanguage(context: Context, type: Int) {
        WeCastConfigManager.getInstance().languageType = type
        setApplicationLanguage(context)
    }

    /**
     * 设置投屏状态改变Listener
     */
    fun setCastStateChangeListener(listener: WeCastInfo.OnCastStateChangeListener) {
        WeCastInfo.instance.mStateChangeListener = listener
    }

    /**
     * 获取WeCast版本号
     */
    fun version(): String {
        return WeCastConstant.WECAST_VERSION
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    fun onAuthCodeError(event: SenderEvents.EAuthCodeError) {
        Logger.t(TAG).d("onAuthCodeError")
        mOnWeCastRunListener?.onAuthCodeError()
    }

    private fun setApplicationLanguage(context: Context) {
        val resources = context.applicationContext.resources
        val dm = resources.displayMetrics
        val config = resources.configuration
        val locale = WeCastConfigManager.getInstance().languageLocale
        config.locale = locale
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            val localeList = LocaleList(locale)
            LocaleList.setDefault(localeList)
            config.locales = localeList
            context.applicationContext.createConfigurationContext(config)
            Locale.setDefault(locale)
        }
        resources.updateConfiguration(config, dm)
    }

    /**
     * 设置投屏页面返回动画效果
     */
    fun setCastingBackAnim(@AnimRes animID: Int) {
        WeCastInfo.instance.castingBackAnim = animID
    }
}