package com.tencent.wecast.sender.cloud.widget

import android.content.Context
import android.graphics.Typeface
import android.text.Editable
import android.text.TextUtils
import android.text.TextWatcher
import android.util.AttributeSet
import android.util.TypedValue
import android.view.*
import com.tencent.wecast.sender.cloud.R


class PinCodeEditText @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null)
    : android.support.v7.widget.AppCompatEditText(context, attrs)
        , View.OnLongClickListener {

    companion object {

        private val TAG = "PinCodeEditText"

        private val SPACE_LETTER = "\u00A0"

        private val DEFAULT_STEP = 3
        private val DEFAULT_MAX_NUM = 6
        private val DEFAULT_HINT_TEXT_SIZE = 12f
        private val DEFAULT_HINT_PADDING_BOTTOM = 15f
    }

    interface InputListener {
        fun inputComplete(completeFlag: Boolean)
    }

    private var mInputListener: InputListener? = null

    fun setInputListener(inputListener: InputListener) {
        this.mInputListener = inputListener
    }

    private var mStep = DEFAULT_STEP
    private var mMaxNum = DEFAULT_MAX_NUM
    private var mHintTextSize = DEFAULT_HINT_TEXT_SIZE
    private var mHintPaddingBottom = DEFAULT_HINT_PADDING_BOTTOM

    private var mTextBold = false
    private var mTextTypefaceAttr: String? = null


    private var mShouldStopChange = false

    private var mHintStyleFlag = false

    private var mTextSize = 0f

    private var mLetterSpacing = 0f

    private var mPaddingBottom = 0

    private var mTypeface: Typeface? = null

    init {
        if (attrs != null) {
            val array = context.obtainStyledAttributes(attrs, R.styleable.PinCodeEditText)
            mStep = array.getInteger(R.styleable.PinCodeEditText_wecast_step, mStep)
            mMaxNum = array.getInteger(R.styleable.PinCodeEditText_wecast_maxNum, mMaxNum)
            mHintTextSize = array.getDimension(R.styleable.PinCodeEditText_wecast_hintTextSize, mHintTextSize)
            mHintPaddingBottom = array.getDimension(R.styleable.PinCodeEditText_wecast_hintPaddingBottom, mHintPaddingBottom)
            mTextBold = array.getBoolean(R.styleable.PinCodeEditText_wecast_textBold, mTextBold)
            mTextTypefaceAttr = array.getString(R.styleable.PinCodeEditText_wecast_textTypeface)

            array.recycle()
        }

        if (!TextUtils.isEmpty(mTextTypefaceAttr)) {
            try {
                mTypeface = Typeface.createFromAsset(context.assets, mTextTypefaceAttr)
            } catch (e: Exception) {

            }
        }

        /** 记录设置的值 **/
        mTextSize = textSize
        mLetterSpacing = letterSpacing
        mPaddingBottom = paddingBottom

        /** 初始化 **/
        setTextSize(TypedValue.COMPLEX_UNIT_PX, mHintTextSize)
        letterSpacing = 0f
        typeface = null
        setPadding(0, 0, 0, mHintPaddingBottom.toInt())
        mHintStyleFlag = true


        addTextChangedListener(object : TextWatcher {
            override fun afterTextChanged(editable: Editable?) {
                if (editable == null) {
                    return
                }
                format(editable)
            }

            override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
            }

            override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {
            }

        })

        /**
         * 禁止复制粘贴
         */
        blockContextMenu()


    }

    private fun format(editable: Editable) {
        if (mShouldStopChange) {
            mShouldStopChange = false
            return
        }
        mShouldStopChange = true

        var originalText = editable.toString().replace(SPACE_LETTER, "")
        var length = originalText.length

        if (length > mMaxNum) {
            originalText = originalText.substring(0, mMaxNum)
            length = mMaxNum
        }

        /** 处理 hint 和数字样式 **/
        if (length == 0 && !mHintStyleFlag) {
            setTextSize(TypedValue.COMPLEX_UNIT_PX, mHintTextSize)
            typeface = null
            letterSpacing = 0f
            setPadding(0, 0, 0, mHintPaddingBottom.toInt())

            mHintStyleFlag = true

        } else if (length > 0 && mHintStyleFlag) {
            setTextSize(TypedValue.COMPLEX_UNIT_PX, mTextSize)

            if (mTypeface != null) {
                typeface = mTypeface
            } else if (mTextBold) {
                typeface = Typeface.defaultFromStyle(Typeface.BOLD)
            }

            letterSpacing = mLetterSpacing
            setPadding(0, 0, 0, mPaddingBottom)

            mHintStyleFlag = false
        }


        val builder = StringBuilder()
        /** 每 step 个字符增加一个空格 **/
        for (i in 0 until length) {
            builder.append(originalText[i])
            if (i % mStep == mStep - 1) {
                if (i + 1 < length) {
                    builder.append(SPACE_LETTER)
                }
            }
        }

        val finalText = builder.toString()

        if (length == mMaxNum) {
            mInputListener?.inputComplete(true)
        } else {
            mInputListener?.inputComplete(false)
        }

        super.setText(finalText)
        setSelection(builder.length)
    }


    fun getPinCode(): CharSequence? {
        return text.toString().replace(SPACE_LETTER, "")
    }

    /**
     * 禁止复制粘贴
     */
    private fun blockContextMenu() {
        this.customSelectionActionModeCallback = BlockedActionModeCallback()
        this.isLongClickable = false
        this.setOnTouchListener { v, event ->
            this@PinCodeEditText.clearFocus()
            false
        }
    }

    override fun onLongClick(v: View?): Boolean { // 长按事件直接消费掉 避免长按弹出复制粘贴菜单框
        return true
    }

    override fun isSuggestionsEnabled(): Boolean {
        return false
    }

    private inner class BlockedActionModeCallback : ActionMode.Callback {

        override fun onCreateActionMode(mode: ActionMode, menu: Menu): Boolean {
            return false
        }

        override fun onPrepareActionMode(mode: ActionMode, menu: Menu): Boolean {
            return false
        }

        override fun onActionItemClicked(mode: ActionMode, item: MenuItem): Boolean {
            return false
        }

        override fun onDestroyActionMode(mode: ActionMode) {

        }
    }


}