package com.supcon.common.view.view.picker;

import android.app.Activity;
import android.support.annotation.FloatRange;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.view.Gravity;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.supcon.common.view.util.LogUtils;
import com.supcon.common.view.view.picker.entity.LinkageFirst;
import com.supcon.common.view.view.picker.entity.LinkageSecond;
import com.supcon.common.view.view.picker.entity.LinkageThird;
import com.supcon.common.view.view.picker.widget.WheelView;

import java.util.ArrayList;
import java.util.List;

/**
 * 两级、三级联动选择器。默认只初始化第一级数据，第二三级数据由联动获得。
 *
 * @param <Fst> the type parameter
 * @param <Snd> the type parameter
 * @param <Trd> the type parameter
 * @see Provider
 * @see DataProvider
 */
public class LinkagePicker<Fst extends LinkageFirst<Snd>, Snd extends LinkageSecond<Trd>, Trd> extends WheelPicker {
    /**
     * The Selected first item.
     */
    protected Fst selectedFirstItem;
    /**
     * The Selected second item.
     */
    protected Snd selectedSecondItem;
    /**
     * The Selected third item.
     */
    protected Trd selectedThirdItem;
    /**
     * The First label.
     */
    protected String firstLabel = "", /**
     * The Second label.
     */
    secondLabel = "", /**
     * The Third label.
     */
    thirdLabel = "";
    /**
     * The Selected first index.
     */
    protected int selectedFirstIndex = 0, /**
     * The Selected second index.
     */
    selectedSecondIndex = 0, /**
     * The Selected third index.
     */
    selectedThirdIndex = 0;
    /**
     * The Provider.
     */
    protected Provider provider;
    /**
     * The First column weight.
     */
    protected float firstColumnWeight = 1.0f;//第一级显示的宽度比重
    /**
     * The Second column weight.
     */
    protected float secondColumnWeight = 1.0f;//第二级显示的宽度比重
    /**
     * The Third column weight.
     */
    protected float thirdColumnWeight = 1.0f;//第三级显示的宽度比重
    private OnPickListener onPickListener;
    private OnLinkageListener onLinkageListener;
    private OnWheelListener onWheelListener;
    private OnWheelLinkageListener onWheelLinkageListener;

    /**
     * Instantiates a new Linkage picker.
     *
     * @param activity the activity
     */
    public LinkagePicker(Activity activity) {
        super(activity);
    }

    /**
     * Instantiates a new Linkage picker.
     *
     * @param activity the activity
     * @param provider the provider
     */
    public LinkagePicker(Activity activity, DataProvider provider) {
        super(activity);
        this.provider = provider;
    }

    /**
     * Instantiates a new Linkage picker.
     *
     * @param activity the activity
     * @param provider the provider
     */
    public LinkagePicker(Activity activity, Provider<Fst, Snd, Trd> provider) {
        super(activity);
        this.provider = provider;
    }

    /**
     * 二级联动选择器构造函数
     *
     * @param activity   the activity
     * @param firstList  the first list
     * @param secondList the second list
     * @deprecated use {@link #LinkagePicker(Activity, Provider)} instead
     */
    @Deprecated
    public LinkagePicker(Activity activity, List<Fst> firstList, List<List<Snd>> secondList) {
        this(activity, firstList, secondList, null);
    }

    /**
     * 三级联动选择器构造函数
     *
     * @param activity the activity
     * @param f        the f
     * @param s        the s
     * @param t        the t
     * @deprecated use {@link #LinkagePicker(Activity, Provider)} instead
     */
    @Deprecated
    public LinkagePicker(Activity activity, List<Fst> f, List<List<Snd>> s, List<List<List<Trd>>> t) {
        super(activity);
        this.provider = new DefaultDataProvider<>(f, s, t);
    }

    /**
     * Sets provider.
     *
     * @param provider the provider
     */
    protected void setProvider(DataProvider provider) {
        this.provider = provider;
    }

    /**
     * Sets provider.
     *
     * @param provider the provider
     */
    protected void setProvider(Provider<Fst, Snd, Trd> provider) {
        this.provider = provider;
    }

    /**
     * Sets selected index.
     *
     * @param firstIndex  the first index
     * @param secondIndex the second index
     */
    public void setSelectedIndex(int firstIndex, int secondIndex) {
        setSelectedIndex(firstIndex, secondIndex, 0);
    }

    /**
     * Sets selected index.
     *
     * @param firstIndex  the first index
     * @param secondIndex the second index
     * @param thirdIndex  the third index
     */
    public void setSelectedIndex(int firstIndex, int secondIndex, int thirdIndex) {
        selectedFirstIndex = firstIndex;
        selectedSecondIndex = secondIndex;
        selectedThirdIndex = thirdIndex;
    }

    /**
     * Sets selected item.
     *
     * @param fst the fst
     * @param snd the snd
     */
    public void setSelectedItem(Fst fst, Snd snd) {
        setSelectedItem(fst, snd, null);
    }

    /**
     * Sets selected item.
     *
     * @param fst the fst
     * @param snd the snd
     * @param trd the trd
     */
    public void setSelectedItem(Fst fst, Snd snd, Trd trd) {
        if (null == provider) {
            throw new IllegalArgumentException("please set data provider at first");
        }
        //noinspection unchecked
        List<Fst> fsts = provider.initFirstData();
        int i = 0;
        for (Fst f : fsts) {
            if (f.equals(fst)) {
                selectedFirstIndex = i;
                break;
            } else if (f.getId().equals(fst.getId()) || f.getName().contains(fst.getName())) {
                selectedFirstIndex = i;
                break;
            }
            i++;
        }
        LogUtils.verbose("init select first: " + fst.getName() + ", index:" + selectedFirstIndex);
        //noinspection unchecked
        List<Snd> snds = provider.linkageSecondData(selectedFirstIndex);
        int j = 0;
        for (Snd s : snds) {
            if (s.equals(snd)) {
                selectedFirstIndex = i;
                break;
            } else if (s.getId().equals(snd.getId()) || s.getName().contains(snd.getName())) {
                selectedSecondIndex = j;
                break;
            }
            j++;
        }
        LogUtils.verbose("init select second: " + snd.getName() + ", index:" + selectedSecondIndex);
        if (provider.isOnlyTwo()) {
            return;//仅仅二级联动
        }
        //noinspection unchecked
        List<Trd> trds = provider.linkageThirdData(selectedFirstIndex, selectedSecondIndex);
        int k = 0;
        for (Trd t : trds) {
            if (t.equals(trd)) {
                selectedThirdIndex = k;
                break;
            } else if (t instanceof LinkageThird) {
                LinkageThird ltrd = (LinkageThird) trd;
                LinkageThird lt = (LinkageThird) t;
                if (lt.getId().equals(ltrd.getId()) || lt.getName().contains(ltrd.getName())) {
                    selectedThirdIndex = k;
                    break;
                }
            }
            k++;
        }
        LogUtils.verbose("init select third: " + trd + ", index:" + selectedThirdIndex);
    }

    /**
     * Sets label.
     *
     * @param firstLabel  the first label
     * @param secondLabel the second label
     */
    public void setLabel(String firstLabel, String secondLabel) {
        setLabel(firstLabel, secondLabel, "");
    }

    /**
     * Sets label.
     *
     * @param firstLabel  the first label
     * @param secondLabel the second label
     * @param thirdLabel  the third label
     */
    public void setLabel(String firstLabel, String secondLabel, String thirdLabel) {
        this.firstLabel = firstLabel;
        this.secondLabel = secondLabel;
        this.thirdLabel = thirdLabel;
    }

    /**
     * Gets selected first item.
     *
     * @return the selected first item
     */
    public Fst getSelectedFirstItem() {
        if (selectedFirstItem == null) {
            //noinspection unchecked
            selectedFirstItem = (Fst) provider.initFirstData().get(selectedFirstIndex);
        }
        return selectedFirstItem;
    }

    /**
     * Gets selected second item.
     *
     * @return the selected second item
     */
    public Snd getSelectedSecondItem() {
        if (selectedSecondItem == null) {
            //noinspection unchecked
            selectedSecondItem = (Snd) provider.linkageSecondData(selectedFirstIndex).get(selectedSecondIndex);
        }
        return selectedSecondItem;
    }

    /**
     * Gets selected third item.
     *
     * @return the selected third item
     */
    public Trd getSelectedThirdItem() {
        if (selectedThirdItem == null) {
            List<Trd> thirdData = provider.linkageThirdData(selectedFirstIndex, selectedSecondIndex);
            if (thirdData.size() > 0) {
                selectedThirdItem = thirdData.get(selectedThirdIndex);
            }
        }
        return selectedThirdItem;
    }

    /**
     * Gets selected first index.
     *
     * @return the selected first index
     */
    public int getSelectedFirstIndex() {
        return selectedFirstIndex;
    }

    /**
     * Gets selected second index.
     *
     * @return the selected second index
     */
    public int getSelectedSecondIndex() {
        return selectedSecondIndex;
    }

    /**
     * Gets selected third index.
     *
     * @return the selected third index
     */
    public int getSelectedThirdIndex() {
        return selectedThirdIndex;
    }

    /**
     * 设置每列的宽度比例，将屏幕分为三列，每列范围为0.0～1.0，如0.3333表示约占宽度的三分之一。
     *
     * @param firstColumnWeight  the first column weight
     * @param secondColumnWeight the second column weight
     * @param thirdColumnWeight  the third column weight
     */
    public void setColumnWeight(@FloatRange(from = 0, to = 1) float firstColumnWeight,
                                @FloatRange(from = 0, to = 1) float secondColumnWeight,
                                @FloatRange(from = 0, to = 1) float thirdColumnWeight) {
        this.firstColumnWeight = firstColumnWeight;
        this.secondColumnWeight = secondColumnWeight;
        this.thirdColumnWeight = thirdColumnWeight;
    }

    /**
     * 设置每列的宽度比例，将屏幕分为两列，每列范围为0.0～1.0，如0.5表示占宽度的一半。
     *
     * @param firstColumnWeight  the first column weight
     * @param secondColumnWeight the second column weight
     */
    public void setColumnWeight(@FloatRange(from = 0, to = 1) float firstColumnWeight,
                                @FloatRange(from = 0, to = 1) float secondColumnWeight) {
        this.firstColumnWeight = firstColumnWeight;
        this.secondColumnWeight = secondColumnWeight;
        this.thirdColumnWeight = 0;
    }

    /**
     * 设置滑动过程数据联动监听器
     *
     * @param onWheelLinkageListener the on wheel linkage listener
     */
    public void setOnWheelLinkageListener(OnWheelLinkageListener onWheelLinkageListener) {
        this.onWheelLinkageListener = onWheelLinkageListener;
    }

    /**
     * Sets on wheel listener.
     *
     * @param onWheelListener the on wheel listener
     * @deprecated use {@link #setOnWheelLinkageListener} instead
     */
    @Deprecated
    public void setOnWheelListener(OnWheelListener onWheelListener) {
        this.onWheelListener = onWheelListener;
    }

    /**
     * 设置完成选泽监听器
     *
     * @param onPickListener the on pick listener
     */
    public void setOnPickListener(OnPickListener<Fst, Snd, Trd> onPickListener) {
        this.onPickListener = onPickListener;
    }

    /**
     * 设置完成选泽监听器
     *
     * @param onStringPickListener the on string pick listener
     */
    public void setOnStringPickListener(OnStringPickListener onStringPickListener) {
        this.onPickListener = onStringPickListener;
    }

    /**
     * Sets on linkage listener.
     *
     * @param onLinkageListener the on linkage listener
     * @deprecated use {@link #setOnStringPickListener} instead
     */
    @Deprecated
    public void setOnLinkageListener(OnLinkageListener onLinkageListener) {
        this.onLinkageListener = onLinkageListener;
    }

    @NonNull
    @Override
    protected View makeCenterView() {
        if (null == provider) {
            throw new IllegalArgumentException("please set data provider before make view");
        }
        LinearLayout layout = new LinearLayout(activity);
        layout.setOrientation(LinearLayout.HORIZONTAL);
        layout.setGravity(Gravity.CENTER);

        final WheelView firstView = createWheelView();
        firstView.setLayoutParams(new LinearLayout.LayoutParams(0, WRAP_CONTENT, firstColumnWeight));
        layout.addView(firstView);
        if (!TextUtils.isEmpty(firstLabel)) {
            TextView labelView = createLabelView();
            labelView.setText(firstLabel);
            layout.addView(labelView);
        }

        final WheelView secondView = createWheelView();
        secondView.setLayoutParams(new LinearLayout.LayoutParams(0, WRAP_CONTENT, secondColumnWeight));
        layout.addView(secondView);
        if (!TextUtils.isEmpty(secondLabel)) {
            TextView labelView = createLabelView();
            labelView.setText(secondLabel);
            layout.addView(labelView);
        }

        final WheelView thirdView = createWheelView();
        if (!provider.isOnlyTwo()) {
            thirdView.setLayoutParams(new LinearLayout.LayoutParams(0, WRAP_CONTENT, thirdColumnWeight));
            layout.addView(thirdView);
            if (!TextUtils.isEmpty(thirdLabel)) {
                TextView labelView = createLabelView();
                labelView.setText(thirdLabel);
                layout.addView(labelView);
            }
        }

        firstView.setItems(provider.initFirstData(), selectedFirstIndex);
        firstView.setOnItemSelectListener(new WheelView.OnItemSelectListener() {
            @Override
            public void onSelected(int index) {
                //noinspection unchecked
                selectedFirstItem = (Fst) provider.initFirstData().get(index);
                selectedFirstIndex = index;
                LogUtils.verbose(this, "change second data after first wheeled");
                selectedSecondIndex = 0;//重置第二级索引
                selectedThirdIndex = 0;//重置第三级索引
                //根据第一级数据获取第二级数据
                //noinspection unchecked
                List<Snd> snds = provider.linkageSecondData(selectedFirstIndex);
                selectedSecondItem = snds.get(selectedSecondIndex);
                secondView.setItems(snds, selectedSecondIndex);
                if (!provider.isOnlyTwo()) {
                    //根据第二级数据获取第三级数据
                    //noinspection unchecked
                    List<Trd> trds = provider.linkageThirdData(selectedFirstIndex, selectedSecondIndex);
                    selectedThirdItem = trds.get(selectedThirdIndex);
                    thirdView.setItems(trds, selectedThirdIndex);
                }
                if (onWheelLinkageListener != null) {
                    onWheelLinkageListener.onLinkage(selectedFirstIndex, 0, 0);
                }
                if (onWheelListener != null) {
                    onWheelListener.onFirstWheeled(selectedFirstIndex, selectedFirstItem.getName());
                }
            }
        });

        secondView.setItems(provider.linkageSecondData(selectedFirstIndex), selectedSecondIndex);
        secondView.setOnItemSelectListener(new WheelView.OnItemSelectListener() {
            @Override
            public void onSelected(int index) {
                //noinspection unchecked
                selectedSecondItem = (Snd) provider.linkageSecondData(selectedFirstIndex).get(index);
                selectedSecondIndex = index;
                if (!provider.isOnlyTwo()) {
                    LogUtils.verbose(this, "change third data after second wheeled");
                    selectedThirdIndex = 0;//重置第三级索引
                    //noinspection unchecked
                    List<Trd> trds = provider.linkageThirdData(selectedFirstIndex, selectedSecondIndex);
                    selectedThirdItem = trds.get(selectedThirdIndex);
                    //根据第二级数据获取第三级数据
                    thirdView.setItems(trds, selectedThirdIndex);
                }
                if (onWheelLinkageListener != null) {
                    onWheelLinkageListener.onLinkage(selectedFirstIndex, selectedSecondIndex, 0);
                }
                if (onWheelListener != null) {
                    onWheelListener.onSecondWheeled(selectedSecondIndex, selectedSecondItem.getName());
                }
            }
        });
        if (provider.isOnlyTwo()) {
            return layout;//仅仅二级联动
        }

        thirdView.setItems(provider.linkageThirdData(selectedFirstIndex, selectedSecondIndex), selectedThirdIndex);
        thirdView.setOnItemSelectListener(new WheelView.OnItemSelectListener() {
            @Override
            public void onSelected(int index) {
                //noinspection unchecked
                selectedThirdItem = (Trd) provider.linkageThirdData(selectedFirstIndex, selectedSecondIndex).get(index);
                selectedThirdIndex = index;
                if (onWheelLinkageListener != null) {
                    onWheelLinkageListener.onLinkage(selectedFirstIndex, selectedSecondIndex, selectedThirdIndex);
                }
                if (onWheelListener != null) {
                    String thirdName;
                    if (selectedThirdItem instanceof LinkageThird) {
                        thirdName = ((LinkageThird) selectedThirdItem).getName();
                    } else {
                        thirdName = selectedThirdItem.toString();
                    }
                    onWheelListener.onThirdWheeled(selectedThirdIndex, thirdName);
                }
            }
        });
        return layout;
    }

    @Override
    public void onSubmit() {
        Fst fst = getSelectedFirstItem();
        Snd snd = getSelectedSecondItem();
        Trd trd = getSelectedThirdItem();
        if (provider.isOnlyTwo()) {
            if (onPickListener != null) {
                //noinspection unchecked
                onPickListener.onPicked(fst, snd, null);
            }
            if (onLinkageListener != null) {
                onLinkageListener.onPicked(fst.getName(), snd.getName(), null);
            }
        } else {
            if (onPickListener != null) {
                //noinspection unchecked
                onPickListener.onPicked(fst, snd, trd);
            }
            if (onLinkageListener != null) {
                String thirdName;
                if (trd instanceof LinkageThird) {
                    thirdName = ((LinkageThird) trd).getName();
                } else {
                    thirdName = trd.toString();
                }
                onLinkageListener.onPicked(fst.getName(), snd.getName(), thirdName);
            }
        }
    }

    /**
     * 数据选择完成监听器
     *
     * @param <Fst> the type parameter
     * @param <Snd> the type parameter
     * @param <Trd> the type parameter
     */
    public interface OnPickListener<Fst, Snd, Trd> {

        /**
         * On picked.
         *
         * @param first  the first
         * @param second the second
         * @param third  the third
         */
        void onPicked(Fst first, Snd second, Trd third);

    }

    /**
     * 数据选择完成监听器
     */
    public static abstract class OnStringPickListener implements OnPickListener<StringLinkageFirst, StringLinkageSecond, String> {

        /**
         * On picked.
         *
         * @param first  the first
         * @param second the second
         * @param third  the third
         */
        public abstract void onPicked(String first, String second, String third);

        @Override
        public void onPicked(StringLinkageFirst first, StringLinkageSecond second, String third) {
            onPicked(first.getName(), second.getName(), third);
        }

    }

    /**
     * 兼容旧版API
     *
     * @deprecated use {@link OnStringPickListener} instead
     */
    @Deprecated
    public static abstract class OnLinkageListener extends OnStringPickListener {

    }

    /**
     * 滑动过程数据联动监听器
     */
    public interface OnWheelLinkageListener {

        /**
         * On linkage.
         *
         * @param firstIndex  the first index
         * @param secondIndex the second index
         * @param thirdIndex  the third index
         */
        void onLinkage(int firstIndex, int secondIndex, int thirdIndex);

    }

    /**
     * 兼容旧版API
     *
     * @deprecated use {@link OnWheelLinkageListener} instead
     */
    @Deprecated
    public static abstract class OnWheelListener {

        /**
         * On first wheeled.
         *
         * @param index the index
         * @param item  the item
         */
        public abstract void onFirstWheeled(int index, String item);

        /**
         * On second wheeled.
         *
         * @param index the index
         * @param item  the item
         */
        public abstract void onSecondWheeled(int index, String item);

        /**
         * On third wheeled.
         *
         * @param index the index
         * @param item  the item
         */
        public void onThirdWheeled(int index, String item) {

        }

    }

    /**
     * 数据提供接口
     *
     * @param <Fst> the type parameter
     * @param <Snd> the type parameter
     * @param <Trd> the type parameter
     */
    public interface Provider<Fst extends LinkageFirst<Snd>, Snd extends LinkageSecond<Trd>, Trd> {

        /**
         * 是否只是二级联动
         *
         * @return the boolean
         */
        boolean isOnlyTwo();

        /**
         * 初始化第一级数据
         *
         * @return the list
         */
        @NonNull
        List<Fst> initFirstData();

        /**
         * 根据第一级数据联动第二级数据
         *
         * @param firstIndex the first index
         * @return the list
         */
        @NonNull
        List<Snd> linkageSecondData(int firstIndex);

        /**
         * 根据第一二级数据联动第三级数据
         *
         * @param firstIndex  the first index
         * @param secondIndex the second index
         * @return the list
         */
        @NonNull
        List<Trd> linkageThirdData(int firstIndex, int secondIndex);

    }

    private static class StringLinkageFirst implements LinkageFirst<StringLinkageSecond> {
        private String name;
        private List<StringLinkageSecond> seconds = new ArrayList<>();

        private StringLinkageFirst(String name, List<StringLinkageSecond> seconds) {
            this.name = name;
            this.seconds = seconds;
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public Object getId() {
            return name;
        }

        @Override
        public List<StringLinkageSecond> getSeconds() {
            return seconds;
        }
    }

    private static class StringLinkageSecond implements LinkageSecond<String> {
        private String name;
        private List<String> thirds = new ArrayList<>();

        private StringLinkageSecond(String name, List<String> thirds) {
            this.name = name;
            this.thirds = thirds;
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public Object getId() {
            return name;
        }

        @Override
        public List<String> getThirds() {
            return thirds;
        }

    }

    /**
     * 兼容旧版API
     */
    public static abstract class DataProvider implements Provider<StringLinkageFirst, StringLinkageSecond, String> {

        /**
         * Provide first data list.
         *
         * @return the list
         */
        @NonNull
        public abstract List<String> provideFirstData();

        /**
         * Provide second data list.
         *
         * @param firstIndex the first index
         * @return the list
         */
        @NonNull
        public abstract List<String> provideSecondData(int firstIndex);

        /**
         * Provide third data list.
         *
         * @param firstIndex  the first index
         * @param secondIndex the second index
         * @return the list
         */
        @Nullable
        public abstract List<String> provideThirdData(int firstIndex, int secondIndex);

        @NonNull
        @Override
        public List<StringLinkageFirst> initFirstData() {
            List<StringLinkageFirst> firsts = new ArrayList<>();
            List<String> data = provideFirstData();
            int i = 0;
            for (String str : data) {
                firsts.add(new StringLinkageFirst(str, linkageSecondData(i)));
                i++;
            }
            return firsts;
        }

        @NonNull
        @Override
        public List<StringLinkageSecond> linkageSecondData(int firstIndex) {
            List<StringLinkageSecond> seconds = new ArrayList<>();
            List<String> data = provideSecondData(firstIndex);
            int i = 0;
            for (String str : data) {
                seconds.add(new StringLinkageSecond(str, linkageThirdData(firstIndex, i)));
                i++;
            }
            return seconds;
        }

        @NonNull
        @Override
        public List<String> linkageThirdData(int firstIndex, int secondIndex) {
            List<String> thirdData = provideThirdData(firstIndex, secondIndex);
            if (thirdData == null) {
                thirdData = new ArrayList<>();
            }
            return thirdData;
        }

    }

    /**
     * 默认的数据提供者
     */
    private static class DefaultDataProvider<Fst extends LinkageFirst<Snd>, Snd extends LinkageSecond<Trd>, Trd> implements Provider<Fst, Snd, Trd> {
        private List<Fst> firstList = new ArrayList<>();
        private List<List<Snd>> secondList = new ArrayList<>();
        private List<List<List<Trd>>> thirdList = new ArrayList<>();
        private boolean onlyTwo = false;

        /**
         * Instantiates a new Default data provider.
         *
         * @param f the f
         * @param s the s
         * @param t the t
         */
        public DefaultDataProvider(List<Fst> f, List<List<Snd>> s, List<List<List<Trd>>> t) {
            this.firstList = f;
            this.secondList = s;
            if (t == null || t.size() == 0) {
                this.onlyTwo = true;
            } else {
                this.thirdList = t;
            }
        }

        public boolean isOnlyTwo() {
            return onlyTwo;
        }

        @Override
        @NonNull
        public List<Fst> initFirstData() {
            return firstList;
        }

        @Override
        @NonNull
        public List<Snd> linkageSecondData(int firstIndex) {
            return secondList.get(firstIndex);
        }

        @Override
        @NonNull
        public List<Trd> linkageThirdData(int firstIndex, int secondIndex) {
            if (onlyTwo) {
                return new ArrayList<>();
            } else {
                return thirdList.get(firstIndex).get(secondIndex);
            }
        }

    }

}
