package com.stripe.android.stripe3ds2.views

import android.app.Dialog
import android.os.Bundle
import android.view.View
import android.view.Window
import androidx.activity.OnBackPressedCallback
import androidx.annotation.ColorInt
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.commit
import com.stripe.android.stripe3ds2.R
import com.stripe.android.stripe3ds2.databinding.StripeChallengeProgressDialogFragmentBinding
import com.stripe.android.stripe3ds2.observability.DefaultErrorReporter
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.observability.Stripe3ds2ErrorReporterConfig
import com.stripe.android.stripe3ds2.transaction.SdkTransactionId

class ChallengeProgressDialogFragment :
    DialogFragment(R.layout.stripe_challenge_progress_dialog_fragment) {

    private val errorReporter: ErrorReporter by lazy {
        val config = nullableArgs?.sdkTransactionId?.let {
            Stripe3ds2ErrorReporterConfig(it)
        } ?: DefaultErrorReporter.EmptyConfig

        DefaultErrorReporter(
            requireContext(),
            config = config
        )
    }

    private val nullableArgs: ChallengeProgressArgs? by lazy {
        ChallengeProgressArgs.fromBundle(arguments)
    }

    override fun onCreateDialog(savedInstanceState: Bundle?): Dialog {
        return super.onCreateDialog(savedInstanceState).also { dialog ->
            dialog.requestWindowFeature(Window.FEATURE_NO_TITLE)
        }
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setStyle(STYLE_NORMAL, R.style.Stripe3ds2ProgressTheme)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val viewBinding = StripeChallengeProgressDialogFragmentBinding.bind(view)

        val args = this.nullableArgs
        if (args == null) {
            errorReporter.reportError(
                IllegalArgumentException("ChallengeProgressDialogFragment's Args was null.")
            )
            return
        }

        activity?.let {
            it.onBackPressedDispatcher.addCallback(
                object : OnBackPressedCallback(true) {
                    override fun handleOnBackPressed() {
                        // ignore default behavior
                    }
                }
            )
        }

        val brand = Brand.lookup(
            args.directoryServerName,
            errorReporter
        )

        childFragmentManager.commit {
            replace(
                viewBinding.fragmentContainer.id,
                ChallengeProgressFragment::class.java,
                ChallengeProgressFragment.Args(
                    brand = brand,
                    accentColor = args.accentColor
                ).toBundle()
            )
        }
    }

    companion object {
        fun show(
            fragmentManager: FragmentManager,
            directoryServerName: String,
            @ColorInt accentColor: Int?,
            sdkTransactionId: SdkTransactionId
        ): DialogFragment {
            val fragment = ChallengeProgressDialogFragment().also {
                it.arguments = ChallengeProgressArgs(
                    directoryServerName,
                    accentColor,
                    sdkTransactionId
                ).toBundle()
            }

            fragmentManager.commit {
                add(android.R.id.content, fragment)
                addToBackStack(null)
            }

            return fragment
        }
    }
}
