package com.stripe.android.stripe3ds2.views

import android.app.Activity
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Bundle
import android.os.Parcelable
import androidx.activity.viewModels
import androidx.annotation.Keep
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import androidx.core.view.isVisible
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.stripe.android.stripe3ds2.databinding.StripeProgressViewLayoutBinding
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.observability.DefaultErrorReporter
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.observability.Stripe3ds2ErrorReporterConfig
import com.stripe.android.stripe3ds2.transaction.SdkTransactionId
import com.stripe.android.stripe3ds2.utils.CustomizeUtils
import com.ults.listeners.SdkChallengeInterface.UL_HANDLE_CHALLENGE_ACTION
import kotlinx.parcelize.Parcelize

class ChallengeProgressActivity : AppCompatActivity() {

    private val localBroadcastManager: LocalBroadcastManager by lazy {
        LocalBroadcastManager.getInstance(this)
    }
    private var dialogBroadcastReceiver: DialogBroadcastReceiver? = null

    internal val viewBinding: StripeProgressViewLayoutBinding by lazy {
        StripeProgressViewLayoutBinding.inflate(layoutInflater)
    }
    private val viewModel: ChallengeProgressViewModel by viewModels()

    private val errorReporter: ErrorReporter by lazy {
        val config = nullableArgs?.sdkTransactionId?.let {
            Stripe3ds2ErrorReporterConfig(it)
        } ?: DefaultErrorReporter.EmptyConfig

        DefaultErrorReporter(
            applicationContext,
            config = config
        )
    }

    private val nullableArgs: Args? by lazy {
        intent.getParcelableExtra(EXTRA_ARGS)
    }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        val args = this.nullableArgs
        if (args == null) {
            errorReporter.reportError(
                IllegalArgumentException("ChallengeProgressActivity's Args was null.")
            )
            finish()
            return
        }

        setContentView(viewBinding.root)

        viewModel.finishLiveData.observe(this) { shouldFinish ->
            if (shouldFinish) {
                finish()
            }
        }

        supportActionBar?.hide()

        val uiCustomization = args.uiCustomization
        uiCustomization.toolbarCustomization?.let { toolbarCustomization ->
            HeaderZoneCustomizer.customizeStatusBar(this, toolbarCustomization)
        }

        val brand = Brand.lookup(
            args.directoryServerName,
            errorReporter
        )

        viewBinding.brandLogo.let { brandLogo ->
            brandLogo.setImageDrawable(ContextCompat.getDrawable(this, brand.drawableResId))
            brandLogo.contentDescription = brand.nameResId?.let { getString(it) }
            brandLogo.isVisible = true
        }

        CustomizeUtils.applyProgressBarColor(viewBinding.progressBar, uiCustomization)

        val dialogBroadcastReceiver = DialogBroadcastReceiver(viewModel.finishLiveData)
        this.dialogBroadcastReceiver = dialogBroadcastReceiver
        localBroadcastManager.registerReceiver(
            dialogBroadcastReceiver,
            IntentFilter(UL_HANDLE_CHALLENGE_ACTION)
        )
    }

    override fun onStop() {
        dialogBroadcastReceiver?.let {
            localBroadcastManager.unregisterReceiver(it)
        }
        dialogBroadcastReceiver = null
        super.onStop()
    }

    private class DialogBroadcastReceiver(
        private val finishLiveData: MutableLiveData<Boolean>
    ) : BroadcastReceiver() {

        override fun onReceive(context: Context, intent: Intent) {
            finishLiveData.value = true
        }
    }

    override fun onBackPressed() {
        if (nullableArgs?.cancelable == true) {
            super.onBackPressed()
        }
    }

    class ChallengeProgressViewModel : ViewModel() {
        val finishLiveData = MutableLiveData<Boolean>()
    }

    @Parcelize
    internal data class Args(
        val directoryServerName: String,
        val cancelable: Boolean,
        val uiCustomization: StripeUiCustomization,
        val sdkTransactionId: SdkTransactionId
    ) : Parcelable

    @Keep
    companion object {
        private const val EXTRA_ARGS = "com.stripe.android.stripe3ds2.views.ChallengeProgressActivity.ARGS"

        @JvmStatic
        fun show(
            activity: Activity,
            directoryServerName: String,
            sdkTransactionId: SdkTransactionId
        ) = show(
            activity,
            directoryServerName,
            false,
            StripeUiCustomization.createWithAppTheme(activity),
            sdkTransactionId
        )

        @JvmStatic
        fun show(
            context: Context,
            directoryServerName: String,
            cancelable: Boolean,
            uiCustomization: StripeUiCustomization,
            sdkTransactionId: SdkTransactionId
        ) {
            context.startActivity(
                createIntent(
                    context,
                    Args(
                        directoryServerName,
                        cancelable,
                        uiCustomization,
                        sdkTransactionId
                    )
                )
            )
        }

        internal fun createIntent(
            context: Context,
            args: Args
        ): Intent {
            return Intent(context, ChallengeProgressActivity::class.java)
                .putExtra(EXTRA_ARGS, args)
        }
    }
}
