package com.stripe.android.stripe3ds2.transaction

import androidx.annotation.VisibleForTesting
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import com.stripe.android.stripe3ds2.transactions.ErrorData
import com.stripe.android.stripe3ds2.transactions.ProtocolError
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.concurrent.TimeUnit
import kotlin.coroutines.CoroutineContext

interface TransactionTimer {
    fun start()
    fun stop()
    val timeout: Flow<Boolean>

    class Default(
        private val challengeStatusReceiver: ChallengeStatusReceiver,
        timeoutMinutes: Int,
        private val errorRequestExecutor: ErrorRequestExecutor,
        private val creqData: ChallengeRequestData,
        private val transactionTimerProvider: TransactionTimerProvider,
        private val workContext: CoroutineContext = Dispatchers.IO
    ) : TransactionTimer {
        private val timeoutMillis = TimeUnit.MINUTES.toMillis(timeoutMinutes.toLong())

        private val mutableTimeoutFlow = MutableStateFlow(false)
        override val timeout: StateFlow<Boolean> = mutableTimeoutFlow
        private var uiTypeCode: String? = null
        private var activeJob: Job? = null

        /**
         * Called when challenge flow has been initiated.
         *
         * Should start the timer and associate the [TransactionTimer] with the given
         * SDK transaction id.
         */
        override fun start() {
            transactionTimerProvider.put(creqData.sdkTransId, this)
            activeJob = CoroutineScope(workContext).launch {
                delay(timeoutMillis)

                withContext(Dispatchers.Main) {
                    onTimeout()
                }
            }
        }

        /**
         * Should stop the timer and disassociate the [TransactionTimer] from the current
         * SDK transaction id.
         */
        override fun stop() {
            activeJob?.cancel()
            activeJob = null
            transactionTimerProvider.remove(creqData.sdkTransId)
        }

        fun setUiTypeCode(uiTypeCode: String?) {
            this.uiTypeCode = uiTypeCode
        }

        @VisibleForTesting
        internal fun onTimeout() {
            activeJob = null
            transactionTimerProvider.remove(creqData.sdkTransId)
            errorRequestExecutor.executeAsync(createTimeoutErrorData())
            challengeStatusReceiver.timedout(uiTypeCode.orEmpty())
            mutableTimeoutFlow.value = true
        }

        private fun createTimeoutErrorData() = ErrorData(
            serverTransId = creqData.threeDsServerTransId,
            acsTransId = creqData.acsTransId,
            errorCode = ProtocolError.TransactionTimedout.code.toString(),
            errorComponent = ErrorData.ErrorComponent.ThreeDsSdk,
            errorDescription = ProtocolError.TransactionTimedout.description,
            errorDetail = "Timeout expiry reached for the transaction",
            messageVersion = creqData.messageVersion,
            sdkTransId = creqData.sdkTransId
        )
    }
}
