package com.stripe.android.stripe3ds2.views

import android.app.Dialog
import android.content.Context
import android.view.View
import android.view.WindowManager
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import androidx.core.content.ContextCompat
import com.stripe.android.stripe3ds2.R
import com.stripe.android.stripe3ds2.databinding.StripeProgressViewLayoutBinding
import com.stripe.android.stripe3ds2.exceptions.SDKRuntimeException
import com.stripe.android.stripe3ds2.init.ui.UiCustomization
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.utils.CustomizeUtils

/**
 * Creates the processing screen that includes
 * - Directory Server logo
 * - graphical element to indicate that an activity is being processed.
 */
interface ProgressViewFactory {
    fun create(context: Context, brand: Brand, uiCustomization: UiCustomization): Dialog

    class Default : ProgressViewFactory {

        /**
         * @param brand the DS brand whose logo should be shown
         * @return a Dialog as defined in "4.2.1 - Processing Screen Requirements"
         */
        override fun create(
            context: Context,
            brand: Brand,
            uiCustomization: UiCustomization
        ): Dialog {
            return ProgressViewDialog(context, brand, uiCustomization)
        }

        private class ProgressViewDialog(
            context: Context,
            private val brand: Brand,
            private val uiCustomization: UiCustomization
        ) : Dialog(context, R.style.Stripe3DS2FullScreenDialog) {

            private val viewBinding: StripeProgressViewLayoutBinding by lazy {
                StripeProgressViewLayoutBinding.inflate(layoutInflater)
            }

            init {
                setCancelable(false)
            }

            override fun onStart() {
                super.onStart()

                window?.setLayout(
                    WindowManager.LayoutParams.MATCH_PARENT,
                    WindowManager.LayoutParams.MATCH_PARENT
                )

                setContentView(viewBinding.root)

                viewBinding.brandLogo.also {
                    it.setImageDrawable(ContextCompat.getDrawable(context, brand.drawableResId))
                    it.contentDescription = context.getString(brand.nameResId)
                    it.visibility = View.VISIBLE
                }

                CustomizeUtils.applyProgressBarColor(viewBinding.progressBar, uiCustomization)
            }
        }
    }

    enum class Brand(
        internal val directoryServerName: String,
        @DrawableRes internal val drawableResId: Int,
        @StringRes internal val nameResId: Int
    ) {
        VISA(
            "visa",
            R.drawable.stripe_3ds2_ic_visa,
            R.string.stripe_3ds2_brand_visa
        ),
        MASTERCARD(
            "mastercard",
            R.drawable.stripe_3ds2_ic_mastercard,
            R.string.stripe_3ds2_brand_mastercard
        ),
        AMEX(
            "american_express",
            R.drawable.stripe_3ds2_ic_amex,
            R.string.stripe_3ds2_brand_amex
        ),
        DISCOVER(
            "discover",
            R.drawable.stripe_3ds2_ic_discover,
            R.string.stripe_3ds2_brand_discover
        );

        internal companion object {
            @Throws(SDKRuntimeException::class)
            internal fun lookup(
                directoryServerName: String,
                errorReporter: ErrorReporter
            ): Brand {
                val brand = values().firstOrNull {
                    it.directoryServerName.equals(directoryServerName.trim(), ignoreCase = true)
                }

                return when {
                    brand != null -> Result.success(brand)
                    else -> {
                        val supportedNames = values().map { it.directoryServerName }
                        Result.failure(
                            SDKRuntimeException(
                                "Directory server name '$directoryServerName' is not supported. Must be one of $supportedNames."
                            )
                        )
                    }
                }.onFailure {
                    errorReporter.reportError(it)
                }.getOrThrow()
            }
        }
    }
}
