package com.stripe.android.stripe3ds2.views

import android.content.Intent
import android.os.Bundle
import androidx.core.os.bundleOf
import com.stripe.android.stripe3ds2.exceptions.InvalidInputException
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.transaction.ChallengeRequestExecutor
import com.stripe.android.stripe3ds2.transaction.ErrorRequestExecutor
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData

internal class ChallengeViewArgs(
    val cresData: ChallengeResponseData,
    val creqData: ChallengeRequestData,
    val uiCustomization: StripeUiCustomization,
    val creqExecutorConfig: ChallengeRequestExecutor.Config,
    val creqExecutorFactory: ChallengeRequestExecutor.Factory,
    val errorExecutorFactory: ErrorRequestExecutor.Factory,
    internal val challengeCompletionIntent: Intent? = null,
    internal val challengeCompletionRequestCode: Int = 0
) {
    internal val sdkTransactionId = creqData.sdkTransId

    /**
     * Convert the fields to a [Bundle]
     */
    fun toBundle() = bundleOf(
        EXTRA_CREQ_DATA to creqData,
        EXTRA_CRES_DATA to cresData,
        EXTRA_UI_CUSTOMIZATION to uiCustomization,
        EXTRA_CREQ_EXECUTOR_CONFIG to creqExecutorConfig,
        EXTRA_CREQ_EXECUTOR_FACTORY to creqExecutorFactory,
        EXTRA_ERROR_EXECUTOR_FACTORY to errorExecutorFactory,
        EXTRA_CHALLENGE_COMPLETION_INTENT to challengeCompletionIntent,
        EXTRA_CHALLENGE_COMPLETION_REQUEST_CODE to challengeCompletionRequestCode
    )

    companion object {
        internal const val EXTRA_CREQ_DATA = "extra_creq_data"
        internal const val EXTRA_CRES_DATA = "extra_cres_data"
        internal const val EXTRA_UI_CUSTOMIZATION = "extra_ui_customization"
        internal const val EXTRA_CREQ_EXECUTOR_CONFIG = "extra_creq_executor_config"
        internal const val EXTRA_CREQ_EXECUTOR_FACTORY = "extra_creq_executor_factory"
        internal const val EXTRA_ERROR_EXECUTOR_FACTORY = "extra_error_executor_factory"
        internal const val EXTRA_CHALLENGE_COMPLETION_INTENT = "extra_challenge_completion_intent"
        internal const val EXTRA_CHALLENGE_COMPLETION_REQUEST_CODE =
            "extra_challenge_completion_request_code"

        /**
         * Create a [ChallengeViewArgs] from Intent extras
         */
        fun create(extras: Bundle?): ChallengeViewArgs {
            if (extras == null) {
                throw InvalidInputException("Intent extras required")
            }

            val cresData =
                extras.getParcelable<ChallengeResponseData>(EXTRA_CRES_DATA)
                    ?: throw InvalidInputException("ChallengeResponseData is required")

            val creqData =
                extras.getSerializable(EXTRA_CREQ_DATA) as ChallengeRequestData

            val uiCustomization =
                extras.getParcelable<StripeUiCustomization>(EXTRA_UI_CUSTOMIZATION)
                    ?: throw InvalidInputException("UiCustomization is required")

            val creqExecutorConfig =
                extras.getSerializable(EXTRA_CREQ_EXECUTOR_CONFIG)
                    as ChallengeRequestExecutor.Config

            val creqExecutorFactory =
                extras.getSerializable(EXTRA_CREQ_EXECUTOR_FACTORY)
                    as ChallengeRequestExecutor.Factory

            val errorExecutorFactory =
                extras.getSerializable(EXTRA_ERROR_EXECUTOR_FACTORY) as ErrorRequestExecutor.Factory

            val challengeCompletionIntent =
                extras.getParcelable(EXTRA_CHALLENGE_COMPLETION_INTENT) as? Intent

            val challengeCompletionRequestCode =
                extras.getInt(EXTRA_CHALLENGE_COMPLETION_REQUEST_CODE, 0)

            return ChallengeViewArgs(
                cresData,
                creqData,
                uiCustomization,
                creqExecutorConfig,
                creqExecutorFactory,
                errorExecutorFactory,
                challengeCompletionIntent = challengeCompletionIntent,
                challengeCompletionRequestCode = challengeCompletionRequestCode
            )
        }
    }
}
