package com.stripe.android.stripe3ds2.views

import android.app.Dialog
import android.content.Intent
import android.graphics.Color
import android.view.View
import android.webkit.WebView
import android.widget.Button
import android.widget.CheckBox
import androidx.annotation.VisibleForTesting
import com.stripe.android.stripe3ds2.R
import com.stripe.android.stripe3ds2.init.ui.StripeUiCustomization
import com.stripe.android.stripe3ds2.init.ui.UiCustomization
import com.stripe.android.stripe3ds2.observability.ErrorReporter
import com.stripe.android.stripe3ds2.transaction.ChallengeAction
import com.stripe.android.stripe3ds2.transaction.ChallengeCompletionIntentStarter
import com.stripe.android.stripe3ds2.transaction.ChallengeFlowOutcome
import com.stripe.android.stripe3ds2.transaction.ChallengeRequestExecutor
import com.stripe.android.stripe3ds2.transaction.ChallengeRequestResult
import com.stripe.android.stripe3ds2.transaction.ChallengeStarter
import com.stripe.android.stripe3ds2.transaction.ChallengeStatusReceiver
import com.stripe.android.stripe3ds2.transaction.CompletionEvent
import com.stripe.android.stripe3ds2.transaction.ErrorRequestExecutor
import com.stripe.android.stripe3ds2.transaction.ProtocolErrorEventFactory
import com.stripe.android.stripe3ds2.transaction.RuntimeErrorEvent
import com.stripe.android.stripe3ds2.transaction.Stripe3ds2ActivityStarterHost
import com.stripe.android.stripe3ds2.transactions.ChallengeRequestData
import com.stripe.android.stripe3ds2.transactions.ChallengeResponseData
import com.stripe.android.stripe3ds2.transactions.ErrorData
import com.stripe.android.stripe3ds2.utils.Factory0
import com.stripe.android.stripe3ds2.utils.ImageCache
import com.ults.listeners.ChallengeType

/**
 * The presenter for [ChallengeActivity].
 */
internal class ChallengePresenter(
    private val activity: ChallengeActivity,
    private val viewModel: ChallengeActivityViewModel,
    private val creqData: ChallengeRequestData,
    private val cresData: ChallengeResponseData,
    private val challengeStatusReceiver: ChallengeStatusReceiver,
    private val requestExecutorConfig: ChallengeRequestExecutor.Config,
    private val uiCustomization: StripeUiCustomization,
    private val progressDialogFactory: Factory0<Dialog>,
    private val errorRequestExecutor: ErrorRequestExecutor,
    private val challengeCompletionIntent: Intent? = null,
    private val headerZoneCustomizer: HeaderZoneCustomizer = HeaderZoneCustomizer(activity),
    challengeEntryViewFactory: ChallengeEntryViewFactory = ChallengeEntryViewFactory(activity),
    private val imageCache: ImageCache = ImageCache.Default,
    private val challengeCompletionIntentStarter: ChallengeCompletionIntentStarter =
        ChallengeCompletionIntentStarter.Default(
            Stripe3ds2ActivityStarterHost(activity)
        )
) {
    private val informationZoneView: InformationZoneView = activity.viewBinding.caInformationZone
    private val challengeZoneView: ChallengeZoneView = activity.viewBinding.caChallengeZone
    private val brandZoneView = activity.viewBinding.caBrandZone

    val challengeZoneTextView =
        if (cresData.uiType == ChallengeResponseData.UiType.TEXT) {
            challengeEntryViewFactory.createChallengeEntryTextView(cresData, uiCustomization)
        } else {
            null
        }
    val challengeZoneSelectView =
        if (
            cresData.uiType == ChallengeResponseData.UiType.SINGLE_SELECT ||
            cresData.uiType == ChallengeResponseData.UiType.MULTI_SELECT
        ) {
            challengeEntryViewFactory
                .createChallengeEntrySelectView(cresData, uiCustomization)
        } else {
            null
        }
    val challengeZoneWebView =
        if (cresData.uiType == ChallengeResponseData.UiType.HTML) {
            challengeEntryViewFactory.createChallengeEntryWebView(cresData)
        } else {
            null
        }

    private var progressDialog: Dialog? = null

    internal val userEntry: String
        @VisibleForTesting
        get() {
            return challengeZoneTextView?.textEntry ?: when {
                challengeZoneSelectView != null -> {
                    challengeZoneSelectView.selectedOptions
                        .joinToString(separator = ",") { it.name }
                }
                challengeZoneWebView != null -> {
                    challengeZoneWebView.userEntry.orEmpty()
                }
                else -> ""
            }
        }

    val challengeType: ChallengeType?
        get() {
            val uiType = cresData.uiType
            return uiType?.challengeType
        }

    val challengeEntryCheckBoxes: List<CheckBox?>?
        get() = challengeZoneSelectView?.checkBoxes

    val challengeWebView: WebView?
        get() = challengeZoneWebView?.webView

    private val uiTypeCode = cresData.uiType?.code.orEmpty()

    constructor(
        activity: ChallengeActivity,
        args: ChallengeViewArgs,
        challengeStatusReceiver: ChallengeStatusReceiver,
        errorReporter: ErrorReporter,
        errorExecutorFactory: ErrorRequestExecutor.Factory,
        viewModel: ChallengeActivityViewModel
    ) : this(
        activity,
        viewModel,
        args.creqData,
        args.cresData,
        challengeStatusReceiver,
        args.creqExecutorConfig,
        args.uiCustomization,
        ChallengeSubmitDialogFactory(activity, args.uiCustomization),
        errorExecutorFactory.create(args.creqExecutorConfig.acsUrl, errorReporter),
        challengeCompletionIntent = args.challengeCompletionIntent
    )

    fun start() {
        configureHeaderZone(headerZoneCustomizer)

        updateBrandZoneImages()

        when {
            challengeZoneTextView != null -> {
                challengeZoneView.setChallengeEntryView(challengeZoneTextView)
                challengeZoneView.setSubmitButton(
                    cresData.submitAuthenticationLabel,
                    uiCustomization.getButtonCustomization(UiCustomization.ButtonType.SUBMIT)
                )
                challengeZoneView.setResendButtonLabel(
                    cresData.resendInformationLabel,
                    uiCustomization.getButtonCustomization(UiCustomization.ButtonType.RESEND)
                )
            }
            challengeZoneSelectView != null -> {
                challengeZoneView.setChallengeEntryView(challengeZoneSelectView)
                challengeZoneView.setSubmitButton(
                    cresData.submitAuthenticationLabel,
                    uiCustomization.getButtonCustomization(UiCustomization.ButtonType.NEXT)
                )
                challengeZoneView.setResendButtonLabel(
                    cresData.resendInformationLabel,
                    uiCustomization.getButtonCustomization(UiCustomization.ButtonType.RESEND)
                )
            }
            challengeZoneWebView != null -> {
                challengeZoneView.setChallengeEntryView(challengeZoneWebView)
                challengeZoneView.setInfoHeaderText(null, null)
                challengeZoneView.setInfoText(null, null)
                challengeZoneView.setSubmitButton(null, null)
                challengeZoneWebView.setOnClickListener { onSubmitClicked() }
                brandZoneView.visibility = View.GONE
            }
            cresData.uiType == ChallengeResponseData.UiType.OOB -> {
                challengeZoneView.setSubmitButton(
                    cresData.oobContinueLabel,
                    uiCustomization.getButtonCustomization(UiCustomization.ButtonType.CONTINUE)
                )
            }
        }

        configureChallengeZoneView()
        configureInformationZoneView()
    }

    fun onResume() {
        viewModel.getTimeout().observe(
            activity,
            { isTimeout ->
                if (isTimeout == true) {
                    onTimeout()
                }
            }
        )
    }

    private fun updateBrandZoneImages() {
        mapOf(
            brandZoneView.issuerImageView to cresData.issuerImage,
            brandZoneView.paymentSystemImageView to cresData.paymentSystemImage
        ).forEach { (imageView, imageData) ->
            viewModel.getImage(imageData)
                .observe(
                    activity,
                    { bitmap ->
                        when {
                            bitmap != null -> {
                                imageView.visibility = View.VISIBLE
                                imageView.setImageBitmap(bitmap)
                            }
                            else -> imageView.visibility = View.GONE
                        }
                    }
                )
        }
    }

    fun onSubmitClicked() {
        if (!activity.isFinishing) {
            activity.dismissKeyboard()
            activity.runOnUiThread {
                progressDialog = progressDialogFactory.create().also {
                    it.show()
                }

                when (cresData.uiType) {
                    ChallengeResponseData.UiType.OOB -> ChallengeAction.Oob
                    ChallengeResponseData.UiType.HTML -> ChallengeAction.HtmlForm(userEntry)
                    else -> ChallengeAction.NativeForm(userEntry)
                }.let(::submit)
            }
        }
    }

    private fun submit(action: ChallengeAction) {
        viewModel.submit(action).observe(
            activity,
            { result ->
                when (result) {
                    is ChallengeRequestResult.Success -> onSuccess(
                        result.creqData,
                        result.cresData
                    )
                    is ChallengeRequestResult.ProtocolError -> onError(result.data)
                    is ChallengeRequestResult.RuntimeError -> onError(result.throwable)
                    is ChallengeRequestResult.Timeout -> onTimeout(result.data)
                }
            }
        )
    }

    private fun onSuccess(
        creqData: ChallengeRequestData,
        cresData: ChallengeResponseData
    ) {
        if (cresData.isChallengeCompleted) {
            viewModel.stopTimer()
            if (creqData.cancelReason != null) {
                challengeStatusReceiver.cancelled(uiTypeCode) {
                    startChallengeCompletionIntent(ChallengeFlowOutcome.Cancel)
                    activity.finish()
                }
            } else {
                val transStatus = cresData.transStatus.orEmpty()

                challengeStatusReceiver.completed(
                    CompletionEvent(
                        sdkTransactionId = cresData.sdkTransId,
                        transactionStatus = transStatus
                    ),
                    uiTypeCode
                ) {
                    startChallengeCompletionIntent(
                        if ("Y" == transStatus) {
                            ChallengeFlowOutcome.CompleteSuccessful
                        } else {
                            ChallengeFlowOutcome.CompleteUnsuccessful
                        }
                    )
                    activity.finish()
                }
            }
        } else {
            ChallengeStarter
                .create(
                    Stripe3ds2ActivityStarterHost(activity),
                    this.creqData,
                    cresData,
                    uiCustomization,
                    requestExecutorConfig,
                    challengeCompletionIntent = challengeCompletionIntent
                )
                .start()
            activity.finish()
        }
    }

    private fun onError(data: ErrorData) {
        challengeStatusReceiver.protocolError(
            ProtocolErrorEventFactory().create(data)
        ) {
            startChallengeCompletionIntent(ChallengeFlowOutcome.ProtocolError)
            activity.finish()
        }
        viewModel.stopTimer()
        errorRequestExecutor.executeAsync(data)
    }

    private fun onTimeout(data: ErrorData) {
        viewModel.stopTimer()
        errorRequestExecutor.executeAsync(data)

        challengeStatusReceiver.runtimeError(RuntimeErrorEvent(data)) {
            startChallengeCompletionIntent(ChallengeFlowOutcome.RuntimeError)
        }

        activity.finish()
    }

    private fun onError(throwable: Throwable) {
        challengeStatusReceiver.runtimeError(RuntimeErrorEvent(throwable)) {
            startChallengeCompletionIntent(ChallengeFlowOutcome.RuntimeError)
        }
    }

    private fun startChallengeCompletionIntent(outcome: ChallengeFlowOutcome) {
        if (challengeCompletionIntent != null) {
            ChallengeCompletionIntentStarter.Default(
                Stripe3ds2ActivityStarterHost(activity)
            ).start(
                intent = challengeCompletionIntent,
                outcome = outcome
            )
        }
    }

    fun onCancelClicked(cancelButton: Button? = null) {
        cancelButton?.run {
            isClickable = false
        }
        cancelChallengeRequest()
    }

    fun cancelChallengeRequest() {
        submit(ChallengeAction.Cancel)
    }

    fun clearImageCache() {
        imageCache.clear()
    }

    fun onDestroy() {
        progressDialog?.let {
            if (it.isShowing) {
                it.dismiss()
            }
        }
        progressDialog = null
    }

    fun updateChallengeText(text: String) {
        challengeZoneTextView?.textEntry = text
    }

    private fun configureChallengeZoneView() {
        challengeZoneView.setInfoHeaderText(
            cresData.challengeInfoHeader,
            uiCustomization.labelCustomization
        )
        challengeZoneView.setInfoText(
            cresData.challengeInfoText,
            uiCustomization.labelCustomization
        )
        challengeZoneView.setInfoTextIndicator(
            if (cresData.shouldShowChallengeInfoTextIndicator)
                R.drawable.ic_indicator
            else
                0
        )

        challengeZoneView.setWhitelistingLabel(
            cresData.whitelistingInfoText,
            uiCustomization.labelCustomization,
            uiCustomization.getButtonCustomization(UiCustomization.ButtonType.SELECT)
        )

        challengeZoneView.setSubmitButtonClickListener {
            onSubmitClicked()
        }
        challengeZoneView.setResendButtonClickListener {
            submit(ChallengeAction.Resend)
        }
    }

    private fun configureInformationZoneView() {
        informationZoneView.setWhyInfo(
            cresData.whyInfoLabel, cresData.whyInfoText,
            uiCustomization.labelCustomization
        )
        informationZoneView.setExpandInfo(
            cresData.expandInfoLabel, cresData.expandInfoText,
            uiCustomization.labelCustomization
        )
        uiCustomization.accentColor?.let { accentColor ->
            informationZoneView.toggleColor = Color.parseColor(accentColor)
        }
    }

    private fun configureHeaderZone(headerZoneCustomizer: HeaderZoneCustomizer) {
        val cancelButton = headerZoneCustomizer.customize(
            uiCustomization.toolbarCustomization,
            uiCustomization.getButtonCustomization(UiCustomization.ButtonType.CANCEL)
        )
        cancelButton?.setOnClickListener { onCancelClicked(cancelButton) }
    }

    fun expandInformationZoneViews() {
        informationZoneView.expandViews()
    }

    fun selectChallengeOption(index: Int) {
        challengeZoneSelectView?.selectOption(index)
    }

    fun refreshUi() {
        if (cresData.uiType == ChallengeResponseData.UiType.HTML &&
            !cresData.acsHtmlRefresh.isNullOrBlank()
        ) {
            challengeZoneWebView?.loadHtml(cresData.acsHtmlRefresh)
        } else if (cresData.uiType == ChallengeResponseData.UiType.OOB &&
            !cresData.challengeAdditionalInfoText.isNullOrBlank()
        ) {
            challengeZoneView.setInfoText(
                cresData.challengeAdditionalInfoText,
                uiCustomization.labelCustomization
            )
            challengeZoneView.setInfoTextIndicator(0)
        }
    }

    private fun onTimeout() {
        challengeCompletionIntent?.let { intent ->
            challengeCompletionIntentStarter.start(
                intent = intent,
                outcome = ChallengeFlowOutcome.Timeout
            )
        }
        if (!activity.isFinishing) {
            activity.finish()
        }
    }
}
