package com.stripe.android.stripe3ds2.transaction

/**
 * A callback object that implements the ChallengeStatusReceiver interface shall receive challenge
 * status notification from the 3DS SDK at the end of the challenge process. This receiver object
 * may be notified by calling various methods. Depending on the result of the challenge process, the
 * app may display a message or redirect the Cardholder to a screen in the app.
 */
interface ChallengeStatusReceiver {

    /**
     * The completed method shall be called when the challenge process is completed. When a
     * transaction is completed, the transaction status shall be available.
     *
     * @param completionEvent Information about completion of the challenge process.
     * @param uiTypeCode the UI type presented to the user when they canceled
     */
    fun completed(
        completionEvent: CompletionEvent,
        uiTypeCode: String,
        onReceiverCompleted: () -> Unit
    )

    /**
     * The cancelled method shall be called when the Cardholder selects the option to cancel the
     * transaction on the challenge screen. Before sending notification about the cancelled event
     * to the app, the 3DS SDK shall end the challenge flow. The app displays subsequent screens
     * after it receives notification about this event.
     *
     * @param uiTypeCode the UI type presented to the user when they canceled
     */
    fun cancelled(
        uiTypeCode: String,
        onReceiverCompleted: () -> Unit
    )

    /**
     * The timedout method shall be called when the challenge process reaches or exceeds the timeout
     * specified during the doChallenge call on the 3DS SDK. On timeout, the SDK shall make a best
     * effort to stop the challenge flow as soon as possible. Before sending notification about the
     * timed out event to the app, the 3DS SDK shall end the challenge flow. The app displays
     * subsequent screens after it receives notification about this event.
     *
     * @param uiTypeCode the UI type presented to the user when they canceled
     */
    fun timedout(
        uiTypeCode: String,
        onReceiverCompleted: () -> Unit
    )

    /**
     * In the 3DS SDK context, a protocol error is any error message that is returned by the ACS.
     * The protocolError method shall be called when the 3DS SDK receives such an error message.
     * The 3DS SDK sends the error code and details from this error message as part of the
     * notification to the app. Note: A protocol error is not covered by the runtimeError method.
     * For information about errors covered by the runtimeError method, refer to
     * [RuntimeErrorEvent]. Before sending notification about the Protocol Error event to the
     * app, the 3DS SDK shall end the challenge flow. The app displays subsequent screens after it
     * receives notification about this event.
     *
     * @param protocolErrorEvent Error code and details.
     */
    fun protocolError(
        protocolErrorEvent: ProtocolErrorEvent,
        onReceiverCompleted: () -> Unit
    )

    /**
     * The runtimeError method shall be called when the 3DS SDK encounters errors during the
     * challenge process. Note: A run-time error is not covered by the protocolError method. For
     * information about errors covered by the protocolError method, refer to
     * [ProtocolErrorEvent]. Before sending notification about the run-time error event to
     * the app, the 3DS SDK shall end the challenge flow. The app displays subsequent screens after
     * it receives notification about this event.
     */
    fun runtimeError(
        runtimeErrorEvent: RuntimeErrorEvent,
        onReceiverCompleted: () -> Unit
    )
}
